package be.recipe.api.prescription;

import be.recipe.api.Prescription;
import be.recipe.api.PrescriptionContent;
import be.recipe.api.crypto.AESCipher;
import be.recipe.api.crypto.InmemKeyDB;
import be.recipe.api.crypto.Message;
import be.recipe.api.executor.Executor;
import be.recipe.api.patient.Patient;
import be.recipe.api.prescriber.Prescriber;
import be.recipe.api.timestamping.SimpleTimestampingContext;
import java8.util.function.Supplier;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;

import static be.recipe.api.Prescription.Type.prescriptionType;
import static be.recipe.api.executor.Executor.ID.executorId;
import static be.recipe.api.patient.Patient.ID.patientID;
import static be.recipe.api.prescriber.Prescriber.ID.prescriberID;
import static be.recipe.api.prescriber.Prescriber.Simple.prescriber;
import static be.recipe.api.prescriber.PrescriberType.DOCTOR;
import static java.nio.charset.StandardCharsets.UTF_8;
import static org.junit.jupiter.api.Assertions.assertEquals;

public class PrescriptionContentTimestampingTest {
  private final AESCipher cipher = new AESCipher();
  private final InmemKeyDB db = new InmemKeyDB(cipher.toSpec());
  private final SimpleTimestampingContext ctx = new SimpleTimestampingContext();
  private PrescriptionContent.Factory factory;

  @BeforeEach
  public void setup() {
    factory = new PrescriptionContent.Factory(new Message.Factory(cipher, ctx), db);
  }

  @Test
  public void decipherTimestampedPrescription() {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    Prescriber.Simple prescriber = prescriber(DOCTOR, prescriberID("-"));
    PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {

              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    final PrescriptionContent.Encrypted.Timestamped timestamped =
        factory.timestamped(ctx.compact(encrypted.bytes()), encrypted.key());

    as(
        executorId("-"),
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                PrescriptionContentTimestampingTest.this.string(timestamped.decrypt().bytes()));
          }
        });
  }

  @Test
  public void decipherTimestampedPrescription_legacyBytes() {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    Prescriber.Simple prescriber = prescriber(DOCTOR, prescriberID("-"));
    PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                 return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    final PrescriptionContent.Encrypted.Timestamped timestamped =
        factory.timestamped(encrypted.bytes(), encrypted.key());

    as(
        executorId("-"),
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                PrescriptionContentTimestampingTest.this.string(timestamped.decrypt().bytes()));
          }
        });
  }

  private <T> T as(Message.Cipher.Key.Owner<Prescription.OnContent> owner, Supplier<T> task) {
    db.owner = owner;
    return task.get();
  }

  private void as(Executor.ID executor, final Runnable task) {
    as(
        executor,
        new Supplier<Object>() {
          @Override
          public Object get() {
            task.run();
            return null;
          }
        });
  }

  private String string(byte[] enciphered) {
    return new String(enciphered, UTF_8);
  }
}
