package be.recipe.api.prescription;

import be.recipe.api.PrefetchingPrescriptionService;
import be.recipe.api.PrescriptionService;
import be.recipe.api.crypto.Cipher;
import be.recipe.api.executor.ListPrescriptions;
import be.recipe.api.series.PartialResult;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.function.Executable;
import org.mockito.invocation.InvocationOnMock;
import org.mockito.stubbing.Answer;

import java.util.concurrent.CountDownLatch;

import static be.recipe.api.Prescription.RID.prescriptionID;
import static be.recipe.api.executor.ListPrescriptions.Response.Encrypted.Simple.response;
import static be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.Simple.toPrescriptionContent;
import static java.util.Arrays.asList;
import static java8.util.stream.Collectors.toList;
import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.*;

@SuppressWarnings("ALL")
public class PrefetchingPrescriptionServiceTest {
  private PrescriptionService.Ciphering.Simplified target =
      mock(PrescriptionService.Ciphering.Simplified.class);
  private PrefetchingPrescriptionService service = new PrefetchingPrescriptionService(target);

  @Test
  public void listAsPrescriper_returnsResponseFromTarget() {
    when(target.list(any(be.recipe.api.prescriber.ListPrescriptions.class)))
        .thenReturn(
            (be.recipe.api.prescriber.ListPrescriptions.PartialResult)
                be.recipe.api.prescriber.ListPrescriptions.PartialResult.Simple.wrap(
                        PartialResult.Simple.of(
                                be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.Simple
                                    .response(prescriptionID("-"))
                                    .content("t"))
                            .hasMore(true))
                    .hasHidden(true));

    be.recipe.api.prescriber.ListPrescriptions.PartialResult<
            be.recipe.api.prescriber.ListPrescriptions.Response.PlainText>
        response = service.list(new be.recipe.api.prescriber.ListPrescriptions());

    assertEquals(asList("t"), response.map(toPrescriptionContent()).stream().collect(toList()));
    assertTrue(response.hasMore());
    assertTrue(response.hasHidden());
  }

  @Test
  public void listAsPrescriber_prefetchesTreatment() {
    be.recipe.api.prescriber.ListPrescriptions.Response.PlainText item =
        mock(be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.class);
    when(target.list(any(be.recipe.api.prescriber.ListPrescriptions.class)))
        .thenReturn(
            be.recipe.api.prescriber.ListPrescriptions.PartialResult.Simple.wrap(
                PartialResult.Simple.of(item)));

    service.list(new be.recipe.api.prescriber.ListPrescriptions()).stream().collect(toList());

    verify(item).prescriptionContent();
  }

  @Test
  public void listAsPrescriber_prefetchesTreatment_throwsUndecipherableExceptions() {
    be.recipe.api.prescriber.ListPrescriptions.Response.PlainText x =
        mock(be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.class);
    be.recipe.api.prescriber.ListPrescriptions.Response.PlainText y =
        mock(be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.class);
    be.recipe.api.prescriber.ListPrescriptions.Response.PlainText z =
        mock(be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.class);

    when(target.list(any(be.recipe.api.prescriber.ListPrescriptions.class)))
        .thenReturn(
            be.recipe.api.prescriber.ListPrescriptions.PartialResult.Simple.wrap(
                PartialResult.Simple.of(x, y, z)));
    when(x.prescriptionContent()).thenReturn("a");
    when(y.prescriptionContent()).thenThrow(Cipher.Undecipherable.class);
    when(z.prescriptionContent()).thenReturn("c");

    final be.recipe.api.prescriber.ListPrescriptions.PartialResult<
            be.recipe.api.prescriber.ListPrescriptions.Response.PlainText>
        response = service.list(new be.recipe.api.prescriber.ListPrescriptions());

    assertEquals("a", response.stream().findFirst().map(toPrescriptionContent()).get());
    assertEquals("c", response.stream().skip(2).findFirst().map(toPrescriptionContent()).get());
    assertThrows(
        Cipher.Undecipherable.class,
        new Executable() {
          @Override
          public void execute() throws Throwable {
            response.stream().skip(1).findFirst().map(toPrescriptionContent()).get();
          }
        });
  }

  @Test
  public void listAsPrescriber_cachesTreatment() {
    be.recipe.api.prescriber.ListPrescriptions.Response.PlainText item =
        mock(be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.class);
    when(target.list(any(be.recipe.api.prescriber.ListPrescriptions.class)))
        .thenReturn(
            be.recipe.api.prescriber.ListPrescriptions.PartialResult.Simple.wrap(
                PartialResult.Simple.of(item)));
    when(item.prescriptionContent()).thenReturn("x", "-");

    assertEquals(
        asList("x"),
        service.list(new be.recipe.api.prescriber.ListPrescriptions()).map(toPrescriptionContent()).stream()
            .collect(toList()));
  }

  @Test
  public void listAsPrescriber_prefetchesTreatmentInParallel() {
    be.recipe.api.prescriber.ListPrescriptions.Response.PlainText slow =
        mock(be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.class);
    be.recipe.api.prescriber.ListPrescriptions.Response.PlainText fast =
        mock(be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.class);
    when(target.list(any(be.recipe.api.prescriber.ListPrescriptions.class)))
        .thenReturn(
            be.recipe.api.prescriber.ListPrescriptions.PartialResult.Simple.wrap(
                PartialResult.Simple.of(slow, fast)));
    when(slow.prescriptionContent())
        .thenAnswer(
            new Answer<Object>() {
              @Override
              public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                latch.await();
                return "-";
              }
            });
    when(fast.prescriptionContent())
        .thenAnswer(
            new Answer<Object>() {
              @Override
              public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                try {
                  return "-";
                } finally {
                  latch.countDown();
                }
              }
            });

    service.list(new be.recipe.api.prescriber.ListPrescriptions()).stream().collect(toList());
  }

  @Test
  public void listAsPrescriber_streamMultipleTimes() {
    when(target.list(any(be.recipe.api.prescriber.ListPrescriptions.class)))
        .thenReturn(
            be.recipe.api.prescriber.ListPrescriptions.PartialResult.Simple.wrap(
                PartialResult.Simple
                    .<be.recipe.api.prescriber.ListPrescriptions.Response.PlainText>empty()));

    PartialResult<be.recipe.api.prescriber.ListPrescriptions.Response.PlainText> result =
        service.list(new be.recipe.api.prescriber.ListPrescriptions());

    result.stream().collect(toList());
    result.stream().collect(toList());
  }

  @Test
  public void listAsExecutor_returnsResponseFromTarget() {
    when(target.list(any(ListPrescriptions.class)))
        .thenReturn(
            PartialResult.Simple.of(
                (ListPrescriptions.Response.PlainText) response(prescriptionID("-")).prescriptionContent("t")));
    assertEquals(
        asList("t"),
        service
            .list(new ListPrescriptions())
            .map(ListPrescriptions.Response.PlainText.Simple.toContent())
            .stream()
            .collect(toList()));
  }

  @Test
  public void listAsExecutor_prefetchesTreatment() {
    ListPrescriptions.Response.PlainText item = mock(ListPrescriptions.Response.PlainText.class);
    when(target.list(any(ListPrescriptions.class))).thenReturn(PartialResult.Simple.of(item));

    service.list(new ListPrescriptions()).stream().collect(toList());

    verify(item).prescriptionContent();
  }

  @Test
  public void listAsExecutor_cachesTreatment() {
    ListPrescriptions.Response.PlainText item = mock(ListPrescriptions.Response.PlainText.class);
    when(target.list(any(ListPrescriptions.class))).thenReturn(PartialResult.Simple.of(item));
    when(item.prescriptionContent()).thenReturn("x", "-");

    assertEquals(
        asList("x"),
        service
            .list(new ListPrescriptions())
            .map(ListPrescriptions.Response.PlainText.Simple.toContent())
            .stream()
            .collect(toList()));
  }

  private CountDownLatch latch = new CountDownLatch(1);

  @Test
  public void listAsExecutor_prefetchesTreatmentInParallel() {
    ListPrescriptions.Response.PlainText slow = mock(ListPrescriptions.Response.PlainText.class);
    ListPrescriptions.Response.PlainText fast = mock(ListPrescriptions.Response.PlainText.class);
    when(target.list(any(ListPrescriptions.class))).thenReturn(PartialResult.Simple.of(slow, fast));
    when(slow.prescriptionContent())
        .thenAnswer(
            new Answer<Object>() {
              @Override
              public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                latch.await();
                return "-";
              }
            });
    when(fast.prescriptionContent())
        .thenAnswer(
            new Answer<Object>() {
              @Override
              public Object answer(InvocationOnMock invocationOnMock) throws Throwable {
                try {
                  return "-";
                } finally {
                  latch.countDown();
                }
              }
            });

    service.list(new ListPrescriptions()).stream().collect(toList());
  }

  @Test
  public void listAsExecutor_streamMultipleTimes() {
    when(target.list(any(ListPrescriptions.class)))
        .thenReturn(PartialResult.Simple.<ListPrescriptions.Response.PlainText>empty());

    PartialResult<ListPrescriptions.Response.PlainText> result =
        service.list(new ListPrescriptions());

    result.stream().collect(toList());
    result.stream().collect(toList());
  }
}
