package be.recipe.api.prescription;

import be.recipe.api.CipheringPrescriptionService;
import be.recipe.api.Prescription;
import be.recipe.api.PrescriptionService;
import be.recipe.api.PrescriptionContent;
import be.recipe.api.crypto.AESCipher;
import be.recipe.api.crypto.InmemKeyDB;
import be.recipe.api.crypto.Message;
import be.recipe.api.patient.Patient;
import be.recipe.api.prescriber.CreatePrescription;
import be.recipe.api.prescriber.GetPrescription;
import be.recipe.api.prescriber.ListPrescriptions;
import be.recipe.api.series.PartialResult;
import java8.util.function.Supplier;
import java8.util.stream.Stream;
import java8.util.stream.StreamSupport;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mockito.ArgumentCaptor;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Collections;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import static be.recipe.api.Prescription.RID.prescriptionID;
import static be.recipe.api.Prescription.Type.prescriptionType;
import static be.recipe.api.Registered.Simple.toRID;
import static be.recipe.api.crypto.Message.Cipher.Key.ID.keyID;
import static be.recipe.api.executor.ListPrescriptions.Response.Encrypted.Simple.response;
import static be.recipe.api.patient.Patient.ID.patientID;
import static be.recipe.api.prescriber.ListPrescriptions.Response.PlainText.Simple.toPrescriptionContent;
import static be.recipe.api.prescriber.Prescriber.ID.prescriberID;
import static be.recipe.api.prescriber.Prescriber.Simple.prescriber;
import static be.recipe.api.prescriber.PrescriberType.DOCTOR;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Arrays.asList;
import static java8.util.stream.Collectors.toList;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertNull;
import static org.mockito.Mockito.*;

@SuppressWarnings("ALL")
public class CipheringPrescriptionServiceTest {
  private AESCipher cipher = new AESCipher();
  private InmemKeyDB keyRegistry = new InmemKeyDB(cipher.toSpec());
  private PrescriptionContent.Factory factory =
      new PrescriptionContent.Factory(new Message.Factory(cipher, null), keyRegistry);
  private PrescriptionService.Simplified target = mock(PrescriptionService.Simplified.class);
  private CipheringPrescriptionService service = new CipheringPrescriptionService(target, factory);

  @BeforeEach
  public void setup() {
    keyRegistry.owner = prescriber(DOCTOR, prescriberID("-"));
    mockDefaultBehaviors();
  }

  private void mockDefaultBehaviors() {
    when(target.list(any(be.recipe.api.executor.ListPrescriptions.class)))
        .thenReturn(
            PartialResult.Simple.of(
                new Supplier<
                    java8.util.stream.Stream<
                        be.recipe.api.executor.ListPrescriptions.Response.Encrypted>>() {
                  @Override
                  public java8.util.stream.Stream<
                          be.recipe.api.executor.ListPrescriptions.Response.Encrypted>
                      get() {
                    return StreamSupport.stream(
                        Collections
                            .<be.recipe.api.executor.ListPrescriptions.Response.Encrypted>
                                emptyList());
                  }
                }));
    when(target.list(any(ListPrescriptions.class)))
        .thenReturn(
            ListPrescriptions.PartialResult.Simple.wrap(
                PartialResult.Simple.of(
                    new Supplier<Stream<ListPrescriptions.Response.Encrypted>>() {
                      @Override
                      public Stream<ListPrescriptions.Response.Encrypted> get() {
                        return StreamSupport.stream(
                            Collections.<ListPrescriptions.Response.Encrypted>emptyList());
                      }
                    })));
  }

  @Test
  public void add_acceptsNullTreatment() {
    service.add(addRequest(null));

    ArgumentCaptor<CreatePrescription<Prescription.Encrypted.Specification>> args =
        ArgumentCaptor.forClass(CreatePrescription.class);
    verify(target).add(args.capture());

    assertNull(args.getValue().customizations.encryptedContent);
    assertNull(args.getValue().customizations.encryptionKeyId);
  }

  @Test
  public void add_enciphersTreatment() {
    service.add(addRequest("t"));

    ArgumentCaptor<CreatePrescription<Prescription.Encrypted.Specification>> args =
        ArgumentCaptor.forClass(CreatePrescription.class);
    verify(target).add(args.capture());

    assertEquals("t", treatment(args.getValue().customizations));
  }

  @Test
  public void add_deciphersResponseFromTarget() {
    PrescriptionContent.Encrypted encrypted = treatment("t");

    Prescription.Encrypted.Simple prescription = new Prescription.Encrypted.Simple();
    prescription.setEncryptedContent(encrypted.bytes());
    prescription.setEncryptionKey(encrypted.key().toString());

    when(target.add(any(CreatePrescription.class))).thenReturn(prescription);

    assertEquals("t", service.add(addRequest()).content());
  }

  @Test
  public void listAsPrescriber_delegatesToTarget() {
    ListPrescriptions request = mock(ListPrescriptions.class);
    service.list(request);
    verify(target).list(request);
  }

  @Test
  public void listAsPrescriber_returnsResponseFromTarget() {
    ListPrescriptions.PartialResult.Simple<ListPrescriptions.Response.Encrypted> expected =
        ListPrescriptions.PartialResult.Simple.wrap(
            PartialResult.Simple.of(
                (ListPrescriptions.Response.Encrypted)
                    ListPrescriptions.Response.Encrypted.Simple.response(prescriptionID("p"))
                        .prescriptionContent(treatment("t"))));
    when(target.list(any(be.recipe.api.prescriber.ListPrescriptions.class))).thenReturn(expected);

    PartialResult<be.recipe.api.prescriber.ListPrescriptions.Response.PlainText> actual =
        service.list(mock(be.recipe.api.prescriber.ListPrescriptions.class));

    assertEquals(
        expected.stream().map(toRID()).collect(toList()),
        actual.stream().map(toRID()).collect(toList()));
    assertEquals(asList("t"), actual.stream().map(toPrescriptionContent()).collect(toList()));
  }

  @Test
  public void listAsPrescriber_returnsResponseFromTarget_withoutContent() {
    ListPrescriptions.PartialResult.Simple<ListPrescriptions.Response.Encrypted> expected =
        ListPrescriptions.PartialResult.Simple.wrap(
            PartialResult.Simple.of(
                (ListPrescriptions.Response.Encrypted)
                    ListPrescriptions.Response.Encrypted.Simple.response(prescriptionID("p"))
                        .prescriptionContent(emptyTreatment())));
    when(target.list(any(be.recipe.api.prescriber.ListPrescriptions.class))).thenReturn(expected);

    PartialResult<be.recipe.api.prescriber.ListPrescriptions.Response.PlainText> actual =
        service.list(mock(be.recipe.api.prescriber.ListPrescriptions.class));

    assertEquals(
        expected.stream().map(toRID()).collect(toList()),
        actual.stream().map(toRID()).collect(toList()));
    assertEquals(asList(""), actual.stream().map(toPrescriptionContent()).collect(toList()));
  }

  @Test
  public void listAsExecutor_delegatesToTarget() {
    be.recipe.api.executor.ListPrescriptions request =
        mock(be.recipe.api.executor.ListPrescriptions.class);
    service.list(request);
    verify(target).list(request);
  }

  @Test
  public void listAsExecutor_returnsResponseFromTarget() {
    PartialResult<be.recipe.api.executor.ListPrescriptions.Response.Encrypted> expected =
        PartialResult.Simple.of(
            (be.recipe.api.executor.ListPrescriptions.Response.Encrypted)
                response(prescriptionID("p")).prescriptionContent(treatment("t")));
    when(target.list(any(be.recipe.api.executor.ListPrescriptions.class))).thenReturn(expected);

    PartialResult<be.recipe.api.executor.ListPrescriptions.Response.PlainText> actual =
        service.list(mock(be.recipe.api.executor.ListPrescriptions.class));

    assertEquals(
        expected.stream().map(toRID()).collect(toList()),
        actual.stream().map(toRID()).collect(toList()));
    assertEquals(
        asList("t"),
        actual.stream()
            .map(be.recipe.api.executor.ListPrescriptions.Response.PlainText.Simple.toContent())
            .collect(toList()));
  }

  @Test
  public void get_delegatesToTarget() {
    GetPrescription request = mock(GetPrescription.class);
    service.get(request);
    verify(target).get(request);
  }

  @Test
  public void getAsPrescriber_returnsPlainPrescription() {
    PrescriptionContent.Encrypted encrypted = treatment("t");

    Prescription.Encrypted.Simple source = new Prescription.Encrypted.Simple();
    source.setEncryptedContent(encrypted.bytes());
    source.setEncryptionKey(encrypted.key().toString());
    when(target.get(any(GetPrescription.class))).thenReturn(source);

    Prescription.PlainText prescription = service.get(mock(GetPrescription.class));
    assertEquals("t", prescription.content());
  }

  @Test
  public void getAsPrescriber_returnsPrescription_withoutEncryptionKey() {
    Prescription.Encrypted.Simple source = new Prescription.Encrypted.Simple();
    source.setEncryptedContent(treatment("t").bytes());
    when(target.get(any(GetPrescription.class))).thenReturn(source);

    Prescription.PlainText prescription = service.get(mock(GetPrescription.class));

    assertNull(prescription.content());
  }

  @Test
  public void getAsPrescriber_returnsPrescription_withoutEncryptedContent() {
    Prescription.Encrypted.Simple source = new Prescription.Encrypted.Simple();
    source.setEncryptionKey("-");
    when(target.get(any(GetPrescription.class))).thenReturn(source);

    Prescription.PlainText prescription = service.get(mock(GetPrescription.class));

    assertNull(prescription.content());
  }

  @Test
  public void getAsExecutor_returnsPlainPrescription() {
    PrescriptionContent.Encrypted encrypted = treatment("t");

    Prescription.Encrypted.Simple source = new Prescription.Encrypted.Simple();
    source.setEncryptedContent(encrypted.bytes());
    source.setEncryptionKey(encrypted.key().toString());
    when(target.get(any(be.recipe.api.executor.GetPrescription.class))).thenReturn(source);

    Prescription.PlainText prescription =
        service.get(mock(be.recipe.api.executor.GetPrescription.class));
    assertEquals("t", prescription.content());
  }

  private PrescriptionContent.Encrypted treatment(String it) {
    return factory
        .create(compress(it.getBytes(UTF_8)))
        .encrypt(patientID("-"), prescriptionType("-"));
  }

  private PrescriptionContent.Encrypted emptyTreatment() {
    PrescriptionContent.Encrypted treatment = treatment("-");
    return factory.create(null, treatment.key());
  }

  private CreatePrescription addRequest(String treatment) {
    Prescription.PlainText.Specification spec = new Prescription.PlainText.Specification();
    spec.content = treatment;
    spec.patient = new Patient.Simple(patientID("-"));
    return new CreatePrescription(spec);
  }

  private CreatePrescription addRequest() {
    return addRequest("-");
  }

  private String treatment(Prescription.Encrypted.Specification spec) {
    return new String(
        decompress(
            factory
                .create(spec.encryptedContent, keyID(spec.encryptionKeyId))
                .decrypt()
                .bytes()),
        UTF_8);
  }

  private byte[] compress(byte[] bytes) {
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
      try (GZIPOutputStream out = new GZIPOutputStream(baos)) {
        out.write(bytes);
      }
      return baos.toByteArray();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private byte[] decompress(byte[] bytes) {
    try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
      try (GZIPInputStream in = new GZIPInputStream(new ByteArrayInputStream(bytes))) {
        int i;
        byte[] buffer = new byte[1024];
        while ((i = in.read(buffer)) > 0) {
          out.write(buffer, 0, i);
        }
      }
      return out.toByteArray();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }
}
