package be.recipe.api.constraints;

import java8.util.function.Supplier;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.function.Executable;

import static be.recipe.api.constraints.Violation.Simple.violation;
import static java.util.Arrays.asList;
import static java8.util.stream.Collectors.toList;
import static org.junit.jupiter.api.Assertions.*;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;

public class ValidationReportTest {
  private final ValidationReport.Simple report = new ValidationReport.Simple();

  @Test
  public void stream() {
    report.violations(violation("x"), violation("y"), violation("z"));
    assertEquals(
        asList("x", "y", "z"),
        report.stream().map(Violation.Simple.toViolation()).collect(toList()));
  }

  @Test
  public void ifRejectedThrow_returnsReport_ifNotRejected() {
    assertEquals(report, report.ifRejectedThrow());
  }

  @Test
  public void ifRejectedThrow() {
    report.violations(violation("-"));
    assertThrows(
        Rejected.class,
        new Executable() {
          @Override
          public void execute() throws Throwable {
            report.ifRejectedThrow();
          }
        });
  }

  @Test
  public void ifRejectedThrow_exposesReport() {
    try {
      report.violations(violation("-"));
      report.ifRejectedThrow();
      fail();
    } catch (Rejected e) {
      assertEquals(report, e.report());
    }
  }

  @Test
  public void executeTask_ifNotRejected() {
    Runnable task = mock(Runnable.class);
    report.ifRejectedThrow().orElse(task);
    verify(task).run();
  }

  @Test
  @SuppressWarnings("unchecked")
  public void executeSupplier_ifNotRejected() {
    Supplier<String> task = mock(Supplier.class);
    report.ifRejectedThrow().orElse(task);
    verify(task).get();
  }

  @Test
  public void returnValueFromSupplier_ifNotRejected() {
    assertEquals(
        "x",
        report
            .ifRejectedThrow()
            .orElse(
                new Supplier<String>() {
                  @Override
                  public String get() {
                    return "x";
                  }
                }));
  }
}
