package be.recipe.api;

import be.recipe.api.Prescription.OnContent;
import be.recipe.api.crypto.Cipher;
import be.recipe.api.crypto.Message;
import be.recipe.api.executor.Executor;
import be.recipe.api.patient.Patient;
import be.recipe.api.prescriber.Prescriber;
import be.recipe.api.prescriber.PrescriberType;
import java8.util.function.Supplier;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.function.Executable;

import java.io.ByteArrayOutputStream;
import java.io.IOException;

import static be.recipe.api.Prescription.Type.prescriptionType;
import static be.recipe.api.executor.Executor.ID.executorId;
import static be.recipe.api.patient.Patient.ID.patientID;
import static be.recipe.api.prescriber.Prescriber.ID.prescriberID;
import static be.recipe.api.prescriber.Prescriber.Simple.prescriber;
import static be.recipe.api.prescriber.PrescriberType.*;
import static java.nio.charset.StandardCharsets.UTF_8;
import static org.junit.jupiter.api.Assertions.*;

public abstract class AbstractPrescriptionContentCipheringTest {
  private PrescriptionContent.Factory factory;

  @BeforeEach
  public void setup() {
    Message.Cipher.Key.DB<OnContent> keyRegistry = keyRegistry();
    factory =
        new PrescriptionContent.Factory(
            new Message.Factory(
                cipher(),
                new Message.Encrypted.Timestamped.Extractor() {
                  @Override
                  public byte[] encrypted(byte[] it) {
                    return unmarshal(it);
                  }
                }),
            keyRegistry);
  }

  public abstract Message.Cipher.Key.DB<OnContent> keyRegistry();

  public abstract Message.Cipher cipher();

  // tag::documented[]
  @Test
  public void createPlainTreatment() {
    PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    assertEquals("Hello World!", string(plainText.bytes()));
  }

  @Test
  public void encipherTreatment() {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    Prescriber.Simple prescriber = prescriber(DOCTOR, prescriberID("-"));
    PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });
    assertNotEquals("Hello World!", string(encrypted.bytes()));
  }

  @Test
  public void decipherAsOriginalPhysician() {
    decipherAsOriginalPrescriber(DOCTOR);
  }

  private void decipherAsOriginalPrescriber(PrescriberType type) {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));

    Prescriber.Simple prescriber = prescriber(type, prescriberID("prescriber"));
    final PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    as(
        prescriber,
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                AbstractPrescriptionContentCipheringTest.this.string(
                    factory.create(encrypted.bytes(), encrypted.key()).decrypt().bytes()));
          }
        });
  }

  @Test
  public void decipherAsOriginalDentist() {
    decipherAsOriginalPrescriber(DENTIST);
  }

  @Test
  public void decipherAsOriginalMidwife() {
    decipherAsOriginalPrescriber(MIDWIFE);
  }

  @Test
  public void decipherAsOriginalNurse() {
    decipherAsOriginalPrescriber(NURSE);
  }

  @Test
  public void decipherAsOriginalPhysioTherapist() {
    decipherAsOriginalPrescriber(PHYSIOTHERAPIST);
  }

  @Test
  public void decipherAsOriginalHospital() {
    decipherAsOriginalPrescriber(HOSPITAL);
  }

  @Test
  public void decipherAsAnyPhysician() {
    decipherAsAnyPrescriberOfType(DOCTOR);
  }

  private void decipherAsAnyPrescriberOfType(PrescriberType type) {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    Prescriber.Simple prescriber = prescriber(type, prescriberID("-"));
    final PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    as(
        prescriber(type, prescriberID("x")),
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                AbstractPrescriptionContentCipheringTest.this.string(encrypted.decrypt().bytes()));
          }
        });
    as(
        prescriber(type, prescriberID("y")),
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                AbstractPrescriptionContentCipheringTest.this.string(encrypted.decrypt().bytes()));
          }
        });
  }

  @Test
  public void decipherAsAnotherDentist() {
      decipherAsAnyPrescriberOfType(DENTIST);
  }

  private void decipherAsAnotherPrescriber_rejected(PrescriberType type) {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    Prescriber.Simple prescriber = prescriber(type, prescriberID("-"));
    final PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    as(
        prescriber(type, prescriberID("?")),
        new Runnable() {
          @Override
          public void run() {
            assertThrows(
                Cipher.Undecipherable.class,
                new Executable() {
                  @Override
                  public void execute() throws Throwable {
                    encrypted.decrypt();
                  }
                });
          }
        });
  }

  @Test
  public void decipherAsAnotherMidwife() {
      decipherAsAnyPrescriberOfType(MIDWIFE);
  }

  @Test
  public void decipherAsAnotherNurse_rejected() {
    decipherAsAnotherPrescriber_rejected(NURSE);
  }

  @Test
  public void decipherAsAnotherPhysioTherapist_rejected() {
    decipherAsAnotherPrescriber_rejected(PHYSIOTHERAPIST);
  }

  @Test
  public void decipherAsAnyHospital() {
    decipherAsAnyPrescriberOfType(HOSPITAL);
  }

  @Test
  public void decipherHospitalPrescriptionAsAnyAmbulant() {
    decipherPrescriptionBetweenAmbulantAndHospital(HOSPITAL, DOCTOR);
  }

  private void decipherPrescriptionBetweenAmbulantAndHospital(
      PrescriberType from, PrescriberType to) {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    Prescriber.Simple prescriber = prescriber(from, prescriberID("-"));
    final PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    as(
        prescriber(to, prescriberID("-")),
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                AbstractPrescriptionContentCipheringTest.this.string(encrypted.decrypt().bytes()));
          }
        });
  }

  @Test
  public void decipherAmbulantPrescriptionAsAnyHospital() {
    decipherPrescriptionBetweenAmbulantAndHospital(DOCTOR, HOSPITAL);
  }

  @Test
  public void decipherAsPhysician_whenPrescribedByDentist() {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    Prescriber.Simple prescriber = prescriber(DENTIST, prescriberID("-"));
    final PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                  return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    as(
        prescriber(DOCTOR, prescriberID("x")),
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                AbstractPrescriptionContentCipheringTest.this.string(encrypted.decrypt().bytes()));
          }
        });
    as(
        prescriber(DOCTOR, prescriberID("y")),
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                AbstractPrescriptionContentCipheringTest.this.string(encrypted.decrypt().bytes()));
          }
        });
  }

  @Test
  public void decipherAsAnyExecutor() {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));

    Prescriber.Simple prescriber = prescriber(DOCTOR, prescriberID("-"));
    final PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    as(
        executorId("x"),
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                AbstractPrescriptionContentCipheringTest.this.string(encrypted.decrypt().bytes()));
          }
        });
    as(
        executorId("y"),
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                AbstractPrescriptionContentCipheringTest.this.string(encrypted.decrypt().bytes()));
          }
        });
  }

  @Test
  public void decipherAsUnknownPatient_rejected() {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    Prescriber.Simple prescriber = prescriber(DOCTOR, prescriberID("-"));
    final PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    as(
        patientID("?"),
        new Runnable() {
          @Override
          public void run() {
            assertThrows(
                Message.Cipher.Undecipherable.class,
                new Executable() {
                  @Override
                  public void execute() throws Throwable {
                    encrypted.decrypt();
                  }
                });
          }
        });
  }

  @Test
  public void decipherAsOtherPatient_rejected() {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));
    Prescriber.Simple prescriber = prescriber(DOCTOR, prescriberID("-"));
    final PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patientID("-"), prescriptionType("P0"));
              }
            });

    as(
        patientID("o"),
        new Runnable() {
          @Override
          public void run() {
            assertThrows(
                Message.Cipher.Undecipherable.class,
                new Executable() {
                  @Override
                  public void execute() throws Throwable {
                    encrypted.decrypt();
                  }
                });
          }
        });
  }

  @Test
  public void decipherAsOriginalPatient() {
    final PrescriptionContent.PlainText plainText = factory.create("Hello World!".getBytes(UTF_8));

    final Patient.ID patient = patient("patient");
    Prescriber.Simple prescriber = prescriber(DOCTOR, prescriberID("-"));
    final PrescriptionContent.Encrypted encrypted =
        as(
            prescriber,
            new Supplier<PrescriptionContent.Encrypted>() {
              @Override
              public PrescriptionContent.Encrypted get() {
                return plainText.encrypt(patient, prescriptionType("P0"));
              }
            });

    as(
        patient,
        new Runnable() {
          @Override
          public void run() {
            assertEquals(
                "Hello World!",
                AbstractPrescriptionContentCipheringTest.this.string(encrypted.decrypt().bytes()));
          }
        });
  }
  // end::documented[]

  protected Patient.ID patient(String patient) {
    return patientID(patient);
  }

  private void as(Prescriber.Simple prescriber, final Runnable task) {
    as(
        prescriber,
        new Supplier<Object>() {
          @Override
          public Object get() {
            task.run();
            return null;
          }
        });
  }

  protected abstract <T> T as(Prescriber.Simple prescriber, Supplier<T> task);

  private void as(Executor.ID executor, final Runnable task) {
    as(
        executor,
        new Supplier<Object>() {
          @Override
          public Object get() {
            task.run();
            return null;
          }
        });
  }

  protected abstract <T> T as(Executor.ID executor, Supplier<T> task);

  private void as(Patient.ID patient, final Runnable task) {
    as(
        patient,
        new Supplier<Object>() {
          @Override
          public Object get() {
            task.run();
            return null;
          }
        });
  }

  protected abstract <T> T as(Patient.ID patient, Supplier<T> task);

  private String string(byte[] enciphered) {
    return new String(enciphered, UTF_8);
  }

  private byte[] marshal(byte[] bytes) {
    try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
      out.write('M');
      out.write(bytes);
      return out.toByteArray();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  private byte[] unmarshal(byte[] bytes) {
    try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
      out.write(bytes, 1, bytes.length - 1);
      return out.toByteArray();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }
}
