package be.recipe.api.prescriber;

import be.recipe.api.Prescription;
import be.recipe.api.Registered;
import be.recipe.api.PrescriptionContent;
import be.recipe.api.constraints.SSIN;
import be.recipe.api.patient.Patient;
import be.recipe.api.projections.SearchResult;
import be.recipe.api.series.SortedBy;
import be.recipe.api.series.Window;
import be.recipe.api.time.LocalDateRange;
import java8.util.function.Function;
import java8.util.function.Functions;
import java8.util.function.Supplier;
import java8.util.stream.Collectors;
import java8.util.stream.RefStreams;
import java8.util.stream.Stream;
import org.threeten.bp.LocalDate;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;

import static be.recipe.api.constraints.Constraint.required;
import static be.recipe.api.series.Window.offset;

// tag::documented[]
public class ListPrescriptions {
  @SSIN
  @Valid
  @NotNull(message = required)
  public Patient.ID patient;

  public Prescription.Status[] status;

  public LocalDateRange between;
  public LocalDateRange expiringBetween;
  public Prescriber.ID prescriberId;
  @Valid public Window window = offset(0);
  public SortedBy<Prescription.Attribute>[] sortedBy;

  @SuppressWarnings({"unchecked", "SuspiciousToArrayCall", "UnusedReturnValue"})
  public ListPrescriptions sortedBy(SortedBy<Prescription.Attribute> sortedBy) {
    this.sortedBy =
        RefStreams.of(
                RefStreams.of(sortedBy),
                this.sortedBy == null
                    ? RefStreams.<SortedBy<Prescription.Attribute>>empty()
                    : RefStreams.of(this.sortedBy))
            .flatMap(Functions.<Stream<SortedBy<Prescription.Attribute>>>identity())
            .collect(Collectors.toList())
            .toArray(new SortedBy[0]);
    return this;
  }

  public interface Command<
      Request extends ListPrescriptions, Response extends ListPrescriptions.Response> {
    ListPrescriptions.PartialResult<Response> list(Request request);
  }

  public interface Response extends Registered {
    Prescription.Status status();

    SearchResult.Prescriber prescriber();

    LocalDate creationDate();

    LocalDate expirationDate();

    VisionOtherPrescribers visionOtherPrescribers();

    interface PlainText extends Response {
      String prescriptionContent();

      // end::documented[]

      class Simple extends AbstractSimpleResponse implements PlainText {
        public String prescriptionContent;

        public static PlainText.Simple response(Prescription.RID rid) {
          PlainText.Simple it = new PlainText.Simple();
          it.rid = rid;
          return it;
        }

        private Simple() {}

        public Simple(Encrypted from) {
          super(from);
        }

        @Override
        public String prescriptionContent() {
          return prescriptionContent;
        }

        public PlainText.Simple prescriptionContent(String prescriptionContent) {
          this.prescriptionContent = prescriptionContent;
          return this;
        }

        public PlainText.Simple content(String prescriptionContent) {
          this.prescriptionContent = prescriptionContent;
          return this;
        }

        public static Function<PlainText, String> toPrescriptionContent() {
          return new Function<PlainText, String>() {
            @Override
            public String apply(ListPrescriptions.Response.PlainText it) {
              return it.prescriptionContent();
            }
          };
        }
      }

      class Wrapper extends AbstractWrapper<PlainText> implements PlainText {
        public Wrapper(PlainText target) {
          super(target);
        }

        @Override
        public String prescriptionContent() {
          return target.prescriptionContent();
        }
      }
      // tag::documented[]
    }

    interface Encrypted extends Response {
      byte[] encryptedPrescriptionContent();

      String encryptionKey();

      // end::documented[]

      class Simple extends AbstractSimpleResponse implements Encrypted {
        public static Encrypted.Simple response(Prescription.RID rid) {
          Encrypted.Simple it = new Encrypted.Simple();
          it.rid = rid;
          return it;
        }

        public static Response.Encrypted wrapper(Prescription.Encrypted prescription) {
          return new PrescriptionWrapper.Encrypted(prescription);
        }

        public byte[] encryptedPrescriptionContent;
        public String encryptionKey;

        @Override
        public byte[] encryptedPrescriptionContent() {
          return encryptedPrescriptionContent;
        }

        @Override
        public String encryptionKey() {
          return encryptionKey;
        }

        public Encrypted.Simple prescriptionContent(PrescriptionContent.Encrypted encrypted) {
          encryptionKey = encrypted.key().toString();
          encryptedPrescriptionContent = encrypted.bytes();
          return this;
        }

        public PlainText.Simple prescriptionContent(String prescriptionContent) {
          return new PlainText.Simple(this).prescriptionContent(prescriptionContent);
        }
      }

      class Wrapper extends AbstractWrapper<Encrypted> implements Encrypted {
        public Wrapper(Encrypted target) {
          super(target);
        }

        @Override
        public byte[] encryptedPrescriptionContent() {
          return target.encryptedPrescriptionContent();
        }

        @Override
        public String encryptionKey() {
          return target.encryptionKey();
        }
      }
      // tag::documented[]
    }

    // end::documented[]

    abstract class AbstractWrapper<T extends Response> implements Response {
      final T target;

      public AbstractWrapper(T target) {
        this.target = target;
      }

      @Override
      public Prescription.RID rid() {
        return target.rid();
      }

      @Override
      public LocalDate creationDate() {
        return target.creationDate();
      }

      @Override
      public LocalDate expirationDate() {
        return target.expirationDate();
      }

      @Override
      public Prescription.Status status() {
        return target.status();
      }

      @Override
      public SearchResult.Prescriber prescriber() {
        return target.prescriber();
      }

      @Override
      public VisionOtherPrescribers visionOtherPrescribers() {
        return target.visionOtherPrescribers();
      }
    }

    abstract class AbstractSimpleResponse implements Response {
      public Prescription.RID rid;
      public SearchResult.Prescriber prescriber;
      public Prescription.Status status;
      public LocalDate creationDate, expirationDate;
      public VisionOtherPrescribers visionOtherPrescribers;

      public AbstractSimpleResponse() {}

      public AbstractSimpleResponse(Response from) {
        visionOtherPrescribers = from.visionOtherPrescribers();
        expirationDate = from.expirationDate();
        creationDate = from.creationDate();
        prescriber = from.prescriber();
        status = from.status();
        rid = from.rid();
      }

      @Override
      public Prescription.RID rid() {
        return rid;
      }

      @Override
      public Prescription.Status status() {
        return status;
      }

      @Override
      public SearchResult.Prescriber prescriber() {
        return prescriber;
      }

      @Override
      public LocalDate creationDate() {
        return creationDate;
      }

      @Override
      public LocalDate expirationDate() {
        return expirationDate;
      }

      @Override
      public VisionOtherPrescribers visionOtherPrescribers() {
        return visionOtherPrescribers;
      }
    }

    class Wrapper implements Response {
      protected final Response target;

      public Wrapper(Response target) {
        this.target = target;
      }

      @Override
      public Prescription.RID rid() {
        return target.rid();
      }

      @Override
      public Prescription.Status status() {
        return target.status();
      }

      @Override
      public SearchResult.Prescriber prescriber() {
        return target.prescriber();
      }

      @Override
      public LocalDate creationDate() {
        return target.creationDate();
      }

      @Override
      public LocalDate expirationDate() {
        return target.expirationDate();
      }

      @Override
      public VisionOtherPrescribers visionOtherPrescribers() {
        return target.visionOtherPrescribers();
      }
    }
    // tag::documented[]
  }

  public interface PartialResult<T extends Response> extends be.recipe.api.series.PartialResult<T> {
    boolean hasHidden();

    // end::documented[]

    class Simple<T extends Response> extends Wrapper<T> implements PartialResult<T> {
      private boolean hidden;

      public static <T extends ListPrescriptions.Response> PartialResult.Simple<T> wrap(
          be.recipe.api.series.PartialResult<T> target) {
        return new PartialResult.Simple<>(target);
      }

      public static <T extends ListPrescriptions.Response> PartialResult.Simple<T> wrap(
          PartialResult<T> target, Supplier<be.recipe.api.series.PartialResult<T>> factory) {
        return new PartialResult.Simple<T>(target, factory);
      }

      public Simple(be.recipe.api.series.PartialResult<T> target) {
        super(target);
      }

      public Simple(
          PartialResult<T> target, Supplier<be.recipe.api.series.PartialResult<T>> factory) {
        super(factory.get());
        hasHidden(target.hasHidden());
      }

      @Override
      public boolean hasHidden() {
        return hidden;
      }

      public PartialResult.Simple<T> hasHidden(boolean it) {
        this.hidden = it;
        return this;
      }
    }
    // tag::documented[]
  }
}

// end::documented[]

abstract class PrescriptionWrapper<T extends Prescription> implements ListPrescriptions.Response {
  protected final T target;

  PrescriptionWrapper(T target) {
    this.target = target;
  }

  @Override
  public Prescription.RID rid() {
    return target.rid();
  }

  @Override
  public Prescription.Status status() {
    return target.status();
  }

  @Override
  public SearchResult.Prescriber prescriber() {
    return target.prescriber();
  }

  @Override
  public LocalDate creationDate() {
    return target.creationDate().toLocalDate();
  }

  @Override
  public LocalDate expirationDate() {
    return target.expirationDate();
  }

  @Override
  public VisionOtherPrescribers visionOtherPrescribers() {
    return target.visionOtherPrescribers();
  }

  public static class Encrypted extends PrescriptionWrapper<Prescription.Encrypted>
      implements ListPrescriptions.Response.Encrypted {
    public Encrypted(Prescription.Encrypted prescription) {
      super(prescription);
    }

    @Override
    public byte[] encryptedPrescriptionContent() {
      return target.encryptedContent();
    }

    @Override
    public String encryptionKey() {
      return target.encryptionKey();
    }
  }
}
