package be.recipe.api.executor;

import static be.recipe.api.constraints.Constraint.required;
import static be.recipe.api.executor.ListPrescriptions.Response.Encrypted.Simple.response;
import static java.util.Arrays.asList;

import be.recipe.api.Prescription;
import be.recipe.api.Registered;
import be.recipe.api.reservation.ContactPreference;
import be.recipe.api.reservation.Reservation;
import be.recipe.api.reservation.Reservation.Attribute;
import be.recipe.api.series.Page;
import be.recipe.api.series.PartialResult;
import be.recipe.api.series.SortedBy;
import java.util.Collections;
import java8.util.function.Function;
import java8.util.stream.Collectors;
import java8.util.stream.Stream;
import java8.util.stream.StreamSupport;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import org.threeten.bp.LocalDateTime;

// tag::class[]
public interface ListReservations<R extends ListReservations.Request> {
  // end::class[]
  // tag::methods[]
  PartialResult<Response> list(R request);

  // end::methods[]

  // tag::class[]
  class Request {
    // end::class[]
    // tag::attributes[]
    @Valid
    @NotNull(message = required)
    public Page page = Page.first();

    public SortedBy<Attribute>[] sortedBy;
    // end::attributes[]

    public static ListReservations.Request request() {
      return new Request();
    }

    @SuppressWarnings({"unchecked", "UnusedReturnValue"})
    public Request sortedBy(SortedBy<Reservation.Attribute> sortedBy) {
      this.sortedBy =
          StreamSupport.stream(
                  asList(
                      StreamSupport.stream(Collections.singleton(sortedBy)),
                      this.sortedBy == null
                          ? StreamSupport.stream(Collections.<SortedBy<Attribute>>emptyList())
                          : StreamSupport.stream(asList(this.sortedBy))))
              .flatMap(
                  new Function<Stream<SortedBy<Attribute>>, Stream<SortedBy<Attribute>>>() {
                    @Override
                    public Stream<SortedBy<Attribute>> apply(
                        Stream<SortedBy<Attribute>> sortedByStream) {
                      return sortedByStream;
                    }
                  })
              .collect(Collectors.<SortedBy<Attribute>>toList())
              .toArray(new SortedBy[0]);
      return this;
    }
    // tag::class[]
  }

  interface Response extends Registered {
    // end::class[]
    // tag::methods[]
    ExecutorViewer visibleTo();

    Reservation.Status status();

    LocalDateTime createdOn();

    LocalDateTime updatedOn();

    String email();

    String phoneNumber();

    ContactPreference contactPreference();

    ListPrescriptions.Response.Encrypted prescription();
    // end::methods[]

    class Simple implements Response {
      public ExecutorViewer visibleTo;
      public Prescription.RID rid;
      public ListPrescriptions.Response.Encrypted prescription;
      public Reservation.Status reservationStatus;
      public ContactPreference contactPreference;
      public LocalDateTime createdOn, updatedOn;
      public String email, phoneNumber;

      public static Response response(Reservation target) {
        return new ReservationWrapper(target);
      }

      @Override
      public Prescription.RID rid() {
        return rid;
      }

      @Override
      public ExecutorViewer visibleTo() {
        return visibleTo;
      }

      @Override
      public Reservation.Status status() {
        return reservationStatus;
      }

      @Override
      public LocalDateTime createdOn() {
        return createdOn;
      }

      @Override
      public LocalDateTime updatedOn() {
        return updatedOn;
      }

      @Override
      public String email() {
        return email;
      }

      @Override
      public String phoneNumber() {
        return phoneNumber;
      }

      @Override
      public ContactPreference contactPreference() {
        return contactPreference;
      }

      @Override
      public ListPrescriptions.Response.Encrypted prescription() {
        return prescription;
      }
    }

    class Wrapper implements ListReservations.Response {
      private final ListReservations.Response target;

      public static Response response(ListReservations.Response target) {
        return new Wrapper(target);
      }

      protected Wrapper(Response target) {
        this.target = target;
      }

      @Override
      public Prescription.RID rid() {
        return target.rid();
      }

      @Override
      public ExecutorViewer visibleTo() {
        return target.visibleTo();
      }

      @Override
      public Reservation.Status status() {
        return target.status();
      }

      @Override
      public LocalDateTime createdOn() {
        return target.createdOn();
      }

      @Override
      public LocalDateTime updatedOn() {
        return target.updatedOn();
      }

      @Override
      public String email() {
        return target.email();
      }

      @Override
      public String phoneNumber() {
        return target.phoneNumber();
      }

      @Override
      public ContactPreference contactPreference() {
        return target.contactPreference();
      }

      @Override
      public ListPrescriptions.Response.Encrypted prescription() {
        return target.prescription();
      }
    }
    // tag::class[]
  }
}
// end::class[]

class ReservationWrapper implements ListReservations.Response {
  private final Reservation target;

  public ReservationWrapper(Reservation target) {
    this.target = target;
  }

  @Override
  public Prescription.RID rid() {
    return target.prescription().rid();
  }

  @Override
  public ExecutorViewer visibleTo() {
    return target.prescription().visibleToExecutors();
  }

  @Override
  public Reservation.Status status() {
    return target.status();
  }

  @Override
  public LocalDateTime createdOn() {
    return target.creationDateTime();
  }

  @Override
  public LocalDateTime updatedOn() {
    return target.updateDateTime();
  }

  @Override
  public String email() {
    return target.email();
  }

  @Override
  public String phoneNumber() {
    return target.phoneNumber();
  }

  @Override
  public ContactPreference contactPreference() {
    return target.contactPreference();
  }

  @Override
  public ListPrescriptions.Response.Encrypted prescription() {
    return response(target.prescription());
  }
}
