package be.recipe.api.executor;

import static be.recipe.api.constraints.Constraint.required;
import static be.recipe.api.viewer.AllViewer.all;

import be.recipe.api.Prescription;
import be.recipe.api.PrescriptionContent;
import be.recipe.api.Registered;
import be.recipe.api.constraints.SSIN;
import be.recipe.api.patient.Patient;
import be.recipe.api.prescriber.Prescriber;
import be.recipe.api.series.Page;
import be.recipe.api.series.PartialResult;
import java8.util.function.Function;
import javax.validation.Valid;
import javax.validation.constraints.NotNull;
import org.threeten.bp.LocalDate;
import org.threeten.bp.LocalDateTime;

// tag::class[]
// tag::class-response[]
public class ListPrescriptions {
  // end::class[]
  // end::class-response[]
  // tag::attributes[]
  @SSIN
  @NotNull(message = required)
  public Patient.ID patient;

  @SSIN public Patient.ID mandateHolder;

  @Valid
  @NotNull(message = required)
  public Page page = Page.first();

  // end::attributes[]

  public static ListPrescriptions list(Patient.ID patient) {
    return new ListPrescriptions(patient);
  }

  public ListPrescriptions() {}

  @Deprecated
  public ListPrescriptions(Patient.ID patient) {
    this.patient = patient;
  }

  public ListPrescriptions(Patient.ID patient, Page page) {
    this(patient, null, page);
  }

  public ListPrescriptions(Patient.ID patient, Patient.ID mandateHolder, Page page) {
    this.patient = patient;
    this.mandateHolder = mandateHolder;
    this.page = page;
  }

  public ListPrescriptions mandateHolder(Patient.ID mandateHolder) {
    this.mandateHolder = mandateHolder;
    return this;
  }

  // tag::class[]

  public interface Command<
      Request extends ListPrescriptions, Response extends ListPrescriptions.Response> {
    // end::class[]
    // tag::methods[]
    PartialResult<Response> list(Request request);
    // end::methods[]

    // tag::class[]
  }

  // tag::class-response[]
  public interface Response extends Registered {
    // end::class-response[]
    // end::class[]
    // tag::methods[]
    // tag::response-methods[]
    Prescription.Type type();

    Patient.ID patient();

    Prescriber.ID prescriber();

    LocalDateTime creationDate();

    LocalDate expirationDate();

    ExecutorViewer visibleTo();

    Prescription.Status status();

    boolean feedbackAllowed();

    // end::methods[]
    // end::response-methods[]
    // tag::class-response[]

    interface PlainText extends Response {
      // end::class-response[]
      // tag::response-methods[]
      String prescriptionContent();

      // end::response-methods[]

      class Simple extends AbstractSimpleResponse implements PlainText {
        public String prescriptionContent;

        public Simple(Encrypted from) {
          super(from);
        }

        @Override
        public String prescriptionContent() {
          return prescriptionContent;
        }

        public PlainText.Simple prescriptionContent(String prescriptionContent) {
          this.prescriptionContent = prescriptionContent;
          return this;
        }

        public static Function<PlainText, String> toContent() {
          return new Function<PlainText, String>() {
            @Override
            public String apply(PlainText it) {
              return it.prescriptionContent();
            }
          };
        }
      }

      class Wrapper extends AbstractWrapper<PlainText> implements PlainText {
        public Wrapper(PlainText target) {
          super(target);
        }

        @Override
        public String prescriptionContent() {
          return target.prescriptionContent();
        }
      }
      // tag::class-response[]
    }

    // end::class-response[]

    interface Encrypted extends Response {
      byte[] encryptedPrescriptionContent();

      String encryptionKey();

      class Simple extends AbstractSimpleResponse implements Encrypted {
        public byte[] encryptedPrescriptionContent;
        public String encryptionKey;

        public static Encrypted.Simple response(Prescription.RID rid) {
          Encrypted.Simple it = new Encrypted.Simple();
          it.rid = rid;
          return it;
        }

        public static Encrypted response(Prescription.Encrypted target) {
          return new PrescriptionWrapper(target);
        }

        @Override
        public byte[] encryptedPrescriptionContent() {
          return encryptedPrescriptionContent;
        }

        @Override
        public String encryptionKey() {
          return encryptionKey;
        }

        public Encrypted.Simple prescriptionContent(PrescriptionContent.Encrypted encrypted) {
          encryptionKey = encrypted.key().toString();
          encryptedPrescriptionContent = encrypted.bytes();
          return this;
        }

        public PlainText.Simple prescriptionContent(String prescriptionContent) {
          return new PlainText.Simple(this).prescriptionContent(prescriptionContent);
        }
      }

      class Wrapper extends AbstractWrapper<Encrypted> implements Encrypted {
        public Wrapper(Encrypted target) {
          super(target);
        }

        @Override
        public byte[] encryptedPrescriptionContent() {
          return target.encryptedPrescriptionContent();
        }

        @Override
        public String encryptionKey() {
          return target.encryptionKey();
        }
      }
    }

    abstract class AbstractWrapper<T extends Response> implements Response {
      final T target;

      public AbstractWrapper(T target) {
        this.target = target;
      }

      @Override
      public Prescription.RID rid() {
        return target.rid();
      }

      @Override
      public Prescription.Type type() {
        return target.type();
      }

      @Override
      public Patient.ID patient() {
        return target.patient();
      }

      @Override
      public Prescriber.ID prescriber() {
        return target.prescriber();
      }

      @Override
      public LocalDateTime creationDate() {
        return target.creationDate();
      }

      @Override
      public LocalDate expirationDate() {
        return target.expirationDate();
      }

      @Override
      public ExecutorViewer visibleTo() {
        return target.visibleTo();
      }

      @Override
      public Prescription.Status status() {
        return target.status();
      }

      @Override
      public boolean feedbackAllowed() {
        return target.feedbackAllowed();
      }
    }
    // tag::class[]
    // tag::class-response[]
  }
}

// end::class[]
// end::class-response[]

class AbstractSimpleResponse implements ListPrescriptions.Response {
  public LocalDateTime creationDate;
  public LocalDate expirationDate;
  public Prescriber.ID prescriber;
  public ExecutorViewer visibleTo = all();
  public Patient.ID patient;
  public Prescription.Status status;
  public boolean feedbackAllowed;
  public Prescription.Type type;
  public Prescription.RID rid;

  public AbstractSimpleResponse() {}

  public AbstractSimpleResponse(ListPrescriptions.Response from) {
    feedbackAllowed = from.feedbackAllowed();
    creationDate = from.creationDate();
    expirationDate = from.expirationDate();
    prescriber = from.prescriber();
    visibleTo = from.visibleTo();
    patient = from.patient();
    status = from.status();
    type = from.type();
    rid = from.rid();
  }

  @Override
  public Prescription.RID rid() {
    return rid;
  }

  @Override
  public Prescription.Type type() {
    return type;
  }

  @Override
  public Patient.ID patient() {
    return patient;
  }

  @Override
  public Prescriber.ID prescriber() {
    return prescriber;
  }

  @Override
  public LocalDateTime creationDate() {
    return creationDate;
  }

  @Override
  public LocalDate expirationDate() {
    return expirationDate;
  }

  @Override
  public ExecutorViewer visibleTo() {
    return visibleTo;
  }

  @Override
  public Prescription.Status status() {
    return status;
  }

  @Override
  public boolean feedbackAllowed() {
    return feedbackAllowed;
  }
}

class PrescriptionWrapper implements ListPrescriptions.Response.Encrypted {
  private final Prescription.Encrypted target;

  public PrescriptionWrapper(Prescription.Encrypted target) {
    this.target = target;
  }

  @Override
  public Prescription.RID rid() {
    return target.rid();
  }

  @Override
  public Prescription.Type type() {
    return target.type();
  }

  @Override
  public Patient.ID patient() {
    return target.patient().id();
  }

  @Override
  public Prescriber.ID prescriber() {
    return target.prescriber().id();
  }

  @Override
  public LocalDateTime creationDate() {
    return target.creationDate();
  }

  @Override
  public LocalDate expirationDate() {
    return target.expirationDate();
  }

  @Override
  public byte[] encryptedPrescriptionContent() {
    return target.encryptedContent();
  }

  @Override
  public String encryptionKey() {
    return target.encryptionKey();
  }

  @Override
  public ExecutorViewer visibleTo() {
    return target.visibleToExecutors();
  }

  @Override
  public Prescription.Status status() {
    return target.status();
  }

  @Override
  public boolean feedbackAllowed() {
    return target.feedbackAllowed();
  }
}
