package be.recipe.api.crypto;

import be.recipe.api.Prescription;
import be.recipe.api.crypto.Message.Cipher.Key;
import be.recipe.api.executor.Executor;
import be.recipe.api.patient.Patient;
import be.recipe.api.prescriber.Prescriber;
import be.recipe.api.prescriber.PrescriberType;
import java8.util.function.Function;
import java8.util.function.Predicate;
import java8.util.function.Predicates;
import java8.util.function.Supplier;
import java8.util.stream.Collectors;
import java8.util.stream.Stream;
import java8.util.stream.StreamSupport;

import javax.crypto.SecretKey;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;

import static be.recipe.api.crypto.InmemKeyDB.Key.byOwner;
import static be.recipe.api.crypto.Message.Cipher.Key.ByID.byID;
import static be.recipe.api.crypto.Message.Cipher.Key.ID.keyID;
import static be.recipe.api.prescriber.PrescriberType.*;
import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Arrays.asList;

public class InmemKeyDB implements Key.DB<Prescription.OnContent> {
  private final AtomicInteger sequence = new AtomicInteger();
  private final List<Key> keys = new ArrayList<>();
  private final Function<byte[], SecretKey> secret;
  public Key.Owner<Prescription.OnContent> owner;

  public InmemKeyDB(Function<byte[], SecretKey> secret) {
    this.secret = secret;
  }

  @Override
  @SuppressWarnings("unchecked")
  public final Key create(String patientId) {
    Stream<Message.Cipher.Key.Owner<Prescription.OnContent>> stream =
        StreamSupport.stream(Collections.singleton(owner));
    List<Message.Cipher.Key.Owner<Prescription.OnContent>> owners = new ArrayList<>();
    owners.add(Executor.All.all());
    owners.add(Patient.ID.patientID(patientId));
    owners.add(DOCTOR);
    owners.add(HOSPITAL);
    owners.add(DENTIST);
    owners.add(MIDWIFE);
    Stream<Message.Cipher.Key.Owner<Prescription.OnContent>> stream1 =
        StreamSupport.stream(owners);
    List<Message.Cipher.Key.Owner<Prescription.OnContent>> totalOwners =
        StreamSupport.stream(asList(stream, stream1))
            .flatMap(
                new Function<
                    Stream<Message.Cipher.Key.Owner<Prescription.OnContent>>,
                    Stream<Message.Cipher.Key.Owner<Prescription.OnContent>>>() {
                  @Override
                  public Stream<Message.Cipher.Key.Owner<Prescription.OnContent>> apply(
                      Stream<Message.Cipher.Key.Owner<Prescription.OnContent>> ownerStream) {
                    return ownerStream;
                  }
                })
            .collect(Collectors.<Message.Cipher.Key.Owner<Prescription.OnContent>>toList());
    Key key =
        Key.key(
            keyID("key-" + sequence.incrementAndGet()),
            secret.apply(
                StreamSupport.stream(totalOwners)
                    .flatMap(InmemKeyDB.normalize())
                    .collect(Collectors.joining())
                    .getBytes(UTF_8)),
            totalOwners.toArray(new Key.Owner[0]));
    keys.add(key);
    return key;
  }

  private static Stream<String> normalize(Key.Owner<Prescription.OnContent> it) {
    return normalize(it, new DefaultKeyProcessor());
  }

  private static Function<Message.Cipher.Key.Owner<Prescription.OnContent>, Stream<String>>
      normalize() {
    return new Function<Message.Cipher.Key.Owner<Prescription.OnContent>, Stream<String>>() {
      @Override
      public Stream<String> apply(Message.Cipher.Key.Owner<Prescription.OnContent> owner) {
        return normalize(owner);
      }
    };
  }

  private static Stream<String> normalize(
          Key.Owner<Prescription.OnContent> it, DefaultKeyProcessor processor) {
    it.process(processor);
    return processor.owners();
  }

  @Override
  public Key get(Key.ID id) {
    return get(id, owner);
  }

  public Key get(Key.ID id, Key.Owner<Prescription.OnContent> owner) {
    return StreamSupport.stream(keys)
        .filter(Predicates.and(byOwner(owner), byID(id)))
        .findFirst()
        .orElseThrow(
            new Supplier<Key.NotFound>() {
              @Override
              public Key.NotFound get() {
                return new Key.NotFound();
              }
            });
  }

  static class Key extends Message.Cipher.Key.Simple {
    private final List<Owner<Prescription.OnContent>> owners;

    @SafeVarargs
    public Key(ID id, SecretKey spec, Owner<Prescription.OnContent>... owners) {
      super(id, spec);
      this.owners = asList(owners);
    }

    @SafeVarargs
    static Key key(ID id, SecretKey spec, Owner<Prescription.OnContent>... owners) {
      return new Key(id, spec, owners);
    }

    static ByOwner byOwner(Owner<Prescription.OnContent> owner) {
      return new ByOwner(owner);
    }

    private static class ByOwner implements Predicate<Key> {
      private final Owner<Prescription.OnContent> owner;

      public ByOwner(Owner<Prescription.OnContent> owner) {
        this.owner = owner;
      }

      @Override
      public boolean test(Key key) {
        return key.contains(owner);
      }
    }

    private boolean contains(Owner<Prescription.OnContent> owner) {
      final List<String> actuals =
          StreamSupport.stream(owners)
              .flatMap(InmemKeyDB.normalize())
              .collect(Collectors.<String>toList());
      List<String> expected =
          normalize(owner, new ProvidedOwnerProcessor()).collect(Collectors.<String>toList());
      return StreamSupport.stream(expected)
          .anyMatch(
              new Predicate<String>() {
                @Override
                public boolean test(String it) {
                  return actuals.contains(it);
                }
              });
    }
  }

  private static class DefaultKeyProcessor implements Prescription.OnContent {
    private final List<String> owners = new ArrayList<>();

    @Override
    public void process(Patient.ID patient) {
      owners.add("patient:" + patient.toString());
    }

    @Override
    public void process(Executor.ID executor) {
      owners.add("executor:" + executor.toString());
    }

    @Override
    public void process(Executor.All all) {
      owners.add("executor:all");
    }

    @Override
    public void process(PrescriberType type) {
      owners.add(type.name());
    }

    @Override
    public void process(Prescriber prescriber) {
      owners.add("prescriber:" + prescriber.id().toString());
    }

    public Stream<String> owners() {
      return StreamSupport.stream(owners);
    }
  }

  private static class ProvidedOwnerProcessor extends DefaultKeyProcessor {
    @Override
    public void process(Executor.ID executor) {
      process(Executor.All.all());
    }

    @Override
    public void process(Prescriber prescriber) {
      super.process(prescriber);
      process(prescriber.type());
    }
  }
}
