package be.recipe.api.crypto;

import javax.crypto.BadPaddingException;
import javax.crypto.IllegalBlockSizeException;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.SecretKeySpec;
import java.security.InvalidKeyException;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.Arrays;

import static javax.crypto.Cipher.DECRYPT_MODE;
import static javax.crypto.Cipher.ENCRYPT_MODE;

public interface Cipher<Key> {
  byte[] encrypt(byte[] decryptedMessage, Key key);

  byte[] decrypt(byte[] encryptedMessage, Key key);

  class Undecipherable extends IllegalArgumentException {
    public Undecipherable(Throwable cause) {
      super(cause);
    }
  }

  class NotCipherable extends IllegalArgumentException {
    public NotCipherable(Throwable cause) {
      super(cause);
    }
  }

  class AES implements Cipher<SecretKey> {
    private final MessageDigest hasher;

    public AES() {
      try {
        hasher = MessageDigest.getInstance("SHA-1");
      } catch (NoSuchAlgorithmException e) {
        throw new RuntimeException(e);
      }
    }

    @Override
    public byte[] encrypt(byte[] decryptedMessage, SecretKey key) {
      try {
        return process(ENCRYPT_MODE, decryptedMessage, key);
      } catch (InvalidKeyException | IllegalBlockSizeException | BadPaddingException e) {
        throw new NotCipherable(e);
      }
    }

    @Override
    public byte[] decrypt(byte[] encryptedMessage, SecretKey key) {
      try {
        return process(DECRYPT_MODE, encryptedMessage, key);
      } catch (InvalidKeyException | IllegalBlockSizeException | BadPaddingException e) {
        throw new Undecipherable(e);
      }
    }

    public byte[] process(int mode, byte[] msg, SecretKey key)
        throws InvalidKeyException, IllegalBlockSizeException, BadPaddingException {
      javax.crypto.Cipher cipher = cipher();
      cipher.init(mode, key);
      return cipher.doFinal(msg);
    }

    private static javax.crypto.Cipher cipher() {
      try {
        return javax.crypto.Cipher.getInstance("AES/ECB/PKCS5Padding");
      } catch (NoSuchAlgorithmException | NoSuchPaddingException e) {
        throw new RuntimeException(e);
      }
    }

    public SecretKeySpec spec(byte[] it) {
      return new SecretKeySpec(Arrays.copyOf(hasher.digest(it), 32), "AES");
    }
  }
}
