package be.recipe.api.constraints;

import java8.util.function.Supplier;
import java8.util.stream.Collectors;
import java8.util.stream.Stream;
import java8.util.stream.StreamSupport;

import java.util.List;

import static be.recipe.api.constraints.Violation.Simple.Builder.toViolation;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;

// tag::class[]
public interface ValidationReport {
  Stream<Violation> stream();
  // end::class[]

  ValidationReport ifRejectedThrow();

  void orElse(Runnable task);

  <T> T orElse(Supplier<T> task);

  class Simple implements ValidationReport {
    private List<Violation> violations = emptyList();

    public static ValidationReport empty() {
      return new Simple();
    }

    public static ValidationReport from(List<Violation> violations) {
      return new Simple(violations);
    }

    public static ValidationReport from(Violation.Simple.Builder... violations) {
      return from(
          StreamSupport.stream(asList(violations))
              .map(toViolation())
              .collect(Collectors.<Violation>toList()));
    }

    public Simple() {}

    public Simple(List<Violation> violations) {
      this.violations = violations;
    }

    @Override
    public void orElse(Runnable task) {
      task.run();
    }

    @Override
    public <T> T orElse(Supplier<T> task) {
      return task.get();
    }

    @Override
    public ValidationReport ifRejectedThrow() {
      if (!violations.isEmpty()) throw new Rejected(this);
      return this;
    }

    @Override
    public Stream<Violation> stream() {
      return StreamSupport.stream(violations);
    }

    public void violations(Violation.Simple.Builder... violations) {
      this.violations =
          StreamSupport.stream(asList(violations))
              .map(Violation.Simple.Builder.toViolation())
              .collect(Collectors.<Violation>toList());
    }

    @Override
    public String toString() {
      return "ValidationReport{" + "violations=" + violations + '}';
    }
  }
  // tag::class[]
}
// end::class[]
