package be.recipe.api;

import be.recipe.api.Prescription.OnContent;
import be.recipe.api.crypto.Cipher;
import be.recipe.api.crypto.Message;
import be.recipe.api.crypto.Message.Cipher.Key;
import be.recipe.api.executor.Executor;
import be.recipe.api.patient.Patient;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import static be.recipe.api.prescriber.PrescriberType.*;

public class PrescriptionContent<M extends Message> {
  protected final Key.DB<OnContent> keyRegistry;
  protected final M target;

  public PrescriptionContent(M target, Key.DB<OnContent> keyRegistry) {
    this.keyRegistry = keyRegistry;
    this.target = target;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof PrescriptionContent)) return false;
    PrescriptionContent<?> prescriptionContent = (PrescriptionContent<?>) o;
    return target.equals(prescriptionContent.target);
  }

  @Override
  public int hashCode() {
    return Objects.hash(target);
  }

  public byte[] bytes() {
    return target.bytes();
  }

  public static class PlainText extends PrescriptionContent<Message.PlainText> {
    public PlainText(Message.PlainText target, Key.DB<OnContent> keyRegistry) {
      super(target, keyRegistry);
    }

    public Encrypted encrypt(Patient.ID patient, Prescription.Type prescription) {
      Key key = keyRegistry.create(patient.toString());
      return new Encrypted(target.encipher(key), keyRegistry, key.id());
    }
  }

  public static class Encrypted extends PrescriptionContent<Message.Encrypted> {
    private final Key.ID key;

    public Encrypted(Message.Encrypted target, Key.DB<OnContent> keyRegistry, Key.ID key) {
      super(target, keyRegistry);
      this.key = key;
    }

    public PlainText decrypt() {
      try {
        return new PlainText(target.decrypt(keyRegistry.get(key)), keyRegistry);
      } catch (Key.NotFound e) {
        throw new Cipher.Undecipherable(e);
      }
    }

    public Key.ID key() {
      return key;
    }

    public static class Timestamped extends Encrypted {
      public Timestamped(
              Message.Encrypted.Timestamped target, Key.DB<OnContent> keyRegistry, Key.ID key) {
        super(target, keyRegistry, key);
      }
    }
  }

  public static class Factory {
    private final Message.Factory messageFactory;
    private final Key.DB<OnContent> keyRegistry;

    public Factory(Message.Factory messageFactory, Key.DB<OnContent> keyRegistry) {
      this.messageFactory = messageFactory;
      this.keyRegistry = keyRegistry;
    }

    public PlainText create(byte[] bytes) {
      return new PlainText(messageFactory.plain(bytes), keyRegistry);
    }

    public Encrypted create(byte[] bytes, Key.ID key) {
      return new Encrypted(messageFactory.encrypted(bytes), keyRegistry, key);
    }

    public Encrypted.Timestamped timestamped(byte[] bytes, Key.ID key) {
      return new Encrypted.Timestamped(messageFactory.timestamped(bytes), keyRegistry, key);
    }
  }
}
