package be.recipe.api;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;
import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;

import be.recipe.api.executor.Executor;
import be.recipe.api.executor.ExecutorViewer;
import be.recipe.api.patient.Patient;
import be.recipe.api.prescriber.Prescriber;
import be.recipe.api.prescriber.PrescriberType;
import be.recipe.api.prescriber.VisionOtherPrescribers;
import be.recipe.api.reservation.Reservation;
import be.recipe.api.text.ExtendableString;
import java.lang.annotation.Documented;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java8.util.Optional;
import java8.util.function.Function;
import javax.validation.Constraint;
import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import javax.validation.Payload;
import javax.validation.constraints.NotNull;
import org.threeten.bp.LocalDate;
import org.threeten.bp.LocalDateTime;

@SuppressWarnings("unused")
// tag::class[]
public interface Prescription
    extends Registered
        // end::class[]
        ,
        GetPrescriptionStatusResponse
// tag::class[]
{
  // end::class[]
  // tag::violations[]
  String not_found = "error.prescription.rid.unknown";
  String status_should_be_x = "status.should.be.%s";
  String status_should_not_be_x = "status.should.not.be.%s";
  String status_should_not_be_x_legacy = "error.request.prescription.status";
  String not_status_with_updater = "error.request.prescription.status2";
  String illegal_access = "error.getPrescription.illegalaccess";
  String delivery_in_progress = "prescription.not.available.for.delivery";
  String vision_incompatible_with_reservation = "error.vision.reservation.notcompatible1";
  String reserved = "reserved";
  String expired = "error.prescription.expired";
  String in_process = "error.updatePrescriptionStatus.locked";
  String expirationdate_required = "error.validation.expirationdate1";
  String expirationdate_too_far_in_future = "error.validation.expirationdate2";
  String expirationdate_already_passed = "error.validation.expirationdate3";
  String expirationdate_invalid_format = "error.validation.expirationdate4";
  String mandate_required = "error.missing.patient.relation";
  String permission_required_to_prescribe = "error.permission.required.to.prescribe.medication";
  String prescription_not_available = "error.prescription.not.available";
  // end::violations[]

  // tag::structure[]
  Prescriber prescriber();

  Patient patient();

  Prescription.Type type();

  LocalDateTime creationDate();

  LocalDate expirationDate();

  boolean feedbackAllowed();

  ExecutorViewer visibleToExecutors();

  // tag::vision-other-prescribers[]
  VisionOtherPrescribers visionOtherPrescribers();
  // end::vision-other-prescribers[]

  Status status();

  Actor.ID statusUpdater();

  Executor executor();

  Optional<Reservation> reservation();

  class RID extends ExtendableString {
    // end::structure[]
    private RID(String value) {
      super(value);
    }

    public static RID prescriptionID(String id) {
      return construct(
          id,
          new Function<String, RID>() {
            @Override
            public RID apply(String it) {
              return new RID(it);
            }
          });
    }
    // tag::structure[]
  }

  class Type extends ExtendableString {
    // end::structure[]
    private Type(String value) {
      super(value);
    }

    public static Type prescriptionType(String it) {
      return construct(
          it,
          new Function<String, Type>() {
            @Override
            public Type apply(String it) {
              return new Type(it);
            }
          });
    }
    // tag::structure[]
  }

  interface Product {
    class Code extends ExtendableString {
      // end::structure[]
      private Code(String value) {
        super(value);
      }

      public static Code productCode(String value) {
        return construct(
            value,
            new Function<String, Code>() {
              @Override
              public Code apply(String it) {
                return new Code(it);
              }
            });
      }
      // tag::structure[]
    }
  }
  // end::structure[]
  // tag::deciphered[]

  // tag::specification-deciphered[]
  interface PlainText extends Prescription {
    // end::specification-deciphered[]
    String content();
    // end::deciphered[]

    // tag::specification-deciphered[]
    class Specification extends Prescription.Specification {
      public String content;
    }
    // end::specification-deciphered[]
    // tag::deciphered[]

    class Simple extends Abstract implements PlainText {
      private String content;
      // end::deciphered[]

      @Override
      public String content() {
        return content;
      }

      public void setContent(String content) {
        this.content = content;
      }
    }

    class Wrapper extends Prescription.Wrapper<PlainText> implements PlainText {
      public Wrapper(PlainText target) {
        super(target);
      }

      @Override
      public String content() {
        return target.content();
      }

      // tag::deciphered[]
    }
    // tag::specification-deciphered[]
  }
  // end::specification-deciphered[]
  // end::deciphered[]
  // tag::enciphered[]

  // tag::specification-enciphered[]
  interface Encrypted extends Prescription {
    // end::specification-enciphered[]
    // end::enciphered[]
    // tag::treatment[]
    byte[] encryptedContent();

    String encryptionKey();
    // end::treatment[]

    // tag::specification-enciphered[]
    class Specification extends Prescription.Specification {
      @NotNull public byte[] encryptedContent;

      @NotNull public String encryptionKeyId;
      // end::specification-enciphered[]

      public Specification() {}

      public Specification(Prescription.Specification spec) {
        super(spec);
      }

      public Specification(Specification spec) {
        super(spec);
        encryptedContent = spec.encryptedContent;
        encryptionKeyId = spec.encryptionKeyId;
      }
      // tag::specification-enciphered[]
    }
    // end::specification-enciphered[]
    // tag::enciphered[]

    class Simple extends Abstract implements Encrypted {
      private byte[] encryptedContent;
      private String encryptionKey;
      // end::enciphered[]

      public static Encrypted.Simple encrypted() {
        return new Encrypted.Simple();
      }

      @Override
      public Encrypted.Simple rid(RID rid) {
        return (Encrypted.Simple) super.rid(rid);
      }

      @Override
      public byte[] encryptedContent() {
        return encryptedContent;
      }

      @Override
      public String encryptionKey() {
        return encryptionKey;
      }

      public void setEncryptedContent(byte[] encryptedContent) {
        this.encryptedContent = encryptedContent;
      }

      public void setEncryptionKey(String encryptionKey) {
        this.encryptionKey = encryptionKey;
      }
    }

    class Wrapper extends Prescription.Wrapper<Encrypted> implements Encrypted {
      public Wrapper(Encrypted target) {
        super(target);
      }

      @Override
      public byte[] encryptedContent() {
        return target.encryptedContent();
      }

      @Override
      public String encryptionKey() {
        return target.encryptionKey();
      }

      // tag::enciphered[]
    }

    // tag::specification-enciphered[]
  }
  // end::specification-enciphered[]
  // end::enciphered[]

  // tag::structure[]

  // end::structure[]

  interface OnContent {
    void process(Patient.ID patient);

    void process(Executor.ID executor);

    void process(Executor.All executor);

    void process(PrescriberType type);

    void process(Prescriber prescriber);
  }

  // tag::specification-structure[]
  abstract class Specification {
    @NotNull public Patient patient;

    @NotNull(message = expirationdate_required)
    public LocalDate expirationDate;

    public boolean feedbackAllowed;

    @NotNull public Type type;
    public Product.Code productCode;

    public ExecutorViewer visibleToExecutor;
    public VisionOtherPrescribers visionOtherPrescribers;
    // end::specification-structure[]

    public Specification() {}

    public Specification(Specification spec) {
      this.type = spec.type;
      this.patient = spec.patient;
      this.productCode = spec.productCode;
      this.expirationDate = spec.expirationDate;
      this.feedbackAllowed = spec.feedbackAllowed;
      this.visibleToExecutor = spec.visibleToExecutor;
      this.visionOtherPrescribers = spec.visionOtherPrescribers;
    }
    // tag::specification-structure[]
  }
  // end::specification-structure[]

  // tag::status[]
  enum Status {
    NotDelivered,
    InProcess,
    Delivered,
    Revoked,
    Archived,
    Expired;
    // end::status[]

    private static final Map<Status, List<Status>> transitionMap =
        new HashMap<Status, List<Status>>() {
          {
            put(NotDelivered, singletonList(InProcess));
            put(InProcess, asList(NotDelivered, Delivered));
          }
        };

    public boolean transitionsTo(Status status) {
      return this == status
          || Optional.ofNullable(transitionMap.get(this))
              .orElse(Collections.<Status>emptyList())
              .contains(status);
    }

    public boolean notFinal() {
      return transitionMap.containsKey(this);
    }

    @Inherited
    @Documented
    @Retention(RUNTIME)
    @Target(FIELD)
    @Constraint(validatedBy = ShouldBe.Validator.class)
    public @interface ShouldBe {
      String message() default status_should_be_x;

      Status status();

      Class<?>[] groups() default {};

      Class<? extends Payload>[] payload() default {};

      class Validator implements ConstraintValidator<ShouldBe, Status> {
        private Status expected;

        @Override
        public void initialize(ShouldBe constraintAnnotation) {
          expected = constraintAnnotation.status();
        }

        @Override
        public boolean isValid(Status status, ConstraintValidatorContext context) {
          return status == expected;
        }
      }
    }

    @Inherited
    @Documented
    @Retention(RUNTIME)
    @Target(FIELD)
    @Constraint(validatedBy = ShouldNotBe.Validator.class)
    public @interface ShouldNotBe {
      String message() default status_should_not_be_x;

      Status[] status();

      Class<?>[] groups() default {};

      Class<? extends Payload>[] payload() default {};

      class Validator implements ConstraintValidator<ShouldNotBe, Status> {
        private List<Status> exclusions;

        @Override
        public void initialize(ShouldNotBe constraintAnnotation) {
          exclusions = asList(constraintAnnotation.status());
        }

        @Override
        public boolean isValid(Status status, ConstraintValidatorContext context) {
          return !exclusions.contains(status);
        }
      }
    }
    // tag::status[]
  }
  // end::status[]

  // tag::attributes[]
  enum Attribute {
    prescriptionDate,
    reservationModificationDate
  }
  // end::attributes[]

  interface Predicate {}

  class Abstract implements Prescription {
    private LocalDateTime creationDate;
    private LocalDate expirationDate;
    private VisionOtherPrescribers visionOtherPrescribers;
    private ExecutorViewer visibleToExecutors;
    private boolean feedbackAllowed;
    private Reservation reservation;
    private Actor.ID statusUpdater;
    private Prescriber prescriber;
    private Executor executor;
    private Patient patient;
    private Status status;
    private Type type;
    private RID rid;

    public void setRid(RID rid) {
      this.rid = rid;
    }

    public Abstract rid(RID rid) {
      setRid(rid);
      return this;
    }

    public void setPrescriber(Prescriber prescriber) {
      this.prescriber = prescriber;
    }

    public void setPatient(Patient patient) {
      this.patient = patient;
    }

    public void setCreationDate(LocalDateTime creationDate) {
      this.creationDate = creationDate;
    }

    public void setExpirationDate(LocalDate expirationDate) {
      this.expirationDate = expirationDate;
    }

    public void setFeedbackAllowed(boolean feedbackAllowed) {
      this.feedbackAllowed = feedbackAllowed;
    }

    public void setVisibleToExecutors(ExecutorViewer visibleToExecutors) {
      this.visibleToExecutors = visibleToExecutors;
    }

    public void setVisionOtherPrescribers(VisionOtherPrescribers visionOtherPrescribers) {
      this.visionOtherPrescribers = visionOtherPrescribers;
    }

    public void setReservation(Reservation reservation) {
      this.reservation = reservation;
    }

    public void setStatusUpdater(Actor.ID statusUpdater) {
      this.statusUpdater = statusUpdater;
    }

    public void setType(Type type) {
      this.type = type;
    }

    @Override
    public RID rid() {
      return rid;
    }

    @Override
    public Prescriber prescriber() {
      return prescriber;
    }

    @Override
    public Patient patient() {
      return patient;
    }

    @Override
    public Type type() {
      return type;
    }

    @Override
    public LocalDateTime creationDate() {
      return creationDate;
    }

    @Override
    public LocalDate expirationDate() {
      return expirationDate;
    }

    @Override
    public boolean feedbackAllowed() {
      return feedbackAllowed;
    }

    @Override
    public ExecutorViewer visibleToExecutors() {
      return visibleToExecutors;
    }

    @Override
    public VisionOtherPrescribers visionOtherPrescribers() {
      return visionOtherPrescribers;
    }

    @Override
    public Status status() {
      return status;
    }

    @Override
    public Actor.ID statusUpdater() {
      return statusUpdater;
    }

    @Override
    public Executor executor() {
      return executor;
    }

    @Override
    public Optional<Reservation> reservation() {
      return Optional.ofNullable(reservation);
    }

    public void setStatus(Status status) {
      this.status = status;
    }

    public void setExecutor(Executor executor) {
      this.executor = executor;
    }
  }

  class Wrapper<T extends Prescription> implements Prescription {
    protected final T target;

    public Wrapper(T target) {
      this.target = target;
    }

    @Override
    public RID rid() {
      return target.rid();
    }

    @Override
    public Prescriber prescriber() {
      return target.prescriber();
    }

    @Override
    public Patient patient() {
      return target.patient();
    }

    @Override
    public Type type() {
      return target.type();
    }

    @Override
    public LocalDateTime creationDate() {
      return target.creationDate();
    }

    @Override
    public LocalDate expirationDate() {
      return target.expirationDate();
    }

    @Override
    public boolean feedbackAllowed() {
      return target.feedbackAllowed();
    }

    @Override
    public ExecutorViewer visibleToExecutors() {
      return target.visibleToExecutors();
    }

    @Override
    public VisionOtherPrescribers visionOtherPrescribers() {
      return target.visionOtherPrescribers();
    }

    @Override
    public Status status() {
      return target.status();
    }

    @Override
    public Actor.ID statusUpdater() {
      return target.statusUpdater();
    }

    @Override
    public Executor executor() {
      return target.executor();
    }

    @Override
    public Optional<Reservation> reservation() {
      return target.reservation();
    }
  }

  // tag::class[]
}
// end::class[]
