package be.recipe.api;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;

import be.recipe.api.executor.Executor;
import be.recipe.api.executor.GetProfile;
import be.recipe.api.patient.GetExecutorProfile;
import java.lang.annotation.Documented;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;
import javax.validation.Constraint;
import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import javax.validation.Payload;

// tag::feature[]
public interface ExecutorProfile
    // end::feature[]
    extends GetExecutorProfile.Response, GetProfile.Response
// tag::feature[]
{
  String illegal_access = "illegal.access.executor.profile";

  // end::feature[]
  // tag::feature[]

  enum ReservationFeature {
    disabled,
    enabled
  // end::feature[]
  ;

    @Inherited
    @Documented
    @Retention(RUNTIME)
    @Target(FIELD)
    @Constraint(validatedBy = Equals.Validator.class)
    public @interface Equals {
      String message() default "status.mismatch";

      ReservationFeature expected();

      Class<?>[] groups() default {};

      Class<? extends Payload>[] payload() default {};

      class Validator implements ConstraintValidator<Equals, Object> {
        private Object expected;

        @Override
        public void initialize(Equals constraintAnnotation) {
          expected = constraintAnnotation.expected();
        }

        @Override
        public boolean isValid(Object actual, ConstraintValidatorContext context) {
          return expected.equals(actual);
        }
      }
    }
    // tag::feature[]
  }

  // tag::communication-channel-specification[]
  enum CommunicationChannelSpecification {
    ANY,
    PHONE_NUMBER,
    EMAIL
  }
  // end::communication-channel-specification[]

  Executor.ID executorId();

  ReservationFeature reservationFeature();

  CommunicationChannelSpecification communicationChannelSpecification();

  // end::feature[]

  class Default implements ExecutorProfile {
    private final Executor.ID executorId;

    public static Default get(Executor.ID executorId) {
      return new Default(executorId);
    }

    public Default(Executor.ID executorId) {
      this.executorId = executorId;
    }

    @Override
    public Executor.ID executorId() {
      return executorId;
    }

    @Override
    public ReservationFeature reservationFeature() {
      return ReservationFeature.enabled;
    }

    @Override
    public CommunicationChannelSpecification communicationChannelSpecification() {
      return CommunicationChannelSpecification.ANY;
    }
  }

  class Simple implements ExecutorProfile {
    private CommunicationChannelSpecification communicationChannelSpecification;
    private Executor.ID executorId;
    private ReservationFeature reservationFeature;

    public static ExecutorProfile.Simple profile() {
      return new ExecutorProfile.Simple();
    }

    @Override
    public Executor.ID executorId() {
      return executorId;
    }

    @Override
    public ReservationFeature reservationFeature() {
      return reservationFeature;
    }

    @Override
    public CommunicationChannelSpecification communicationChannelSpecification() {
      return communicationChannelSpecification;
    }

    public ExecutorProfile.Simple communicationChannelSpecification(
        CommunicationChannelSpecification spec) {
      communicationChannelSpecification = spec;
      return this;
    }

    public ExecutorProfile.Simple reservationFeature(ReservationFeature reservationFeature) {
      this.reservationFeature = reservationFeature;
      return this;
    }

    public ExecutorProfile.Simple executorId(Executor.ID executorId) {
      this.executorId = executorId;
      return this;
    }
  }

  class Wrapper implements ExecutorProfile {
    private final ExecutorProfile target;

    public Wrapper(ExecutorProfile target) {
      this.target = target;
    }

    @Override
    public Executor.ID executorId() {
      return target.executorId();
    }

    @Override
    public ReservationFeature reservationFeature() {
      return target.reservationFeature();
    }

    @Override
    public CommunicationChannelSpecification communicationChannelSpecification() {
      return target.communicationChannelSpecification();
    }
  }
  // tag::feature[]
}
// end::feature[]
