package be.recipe.api.series;

import static java.util.Arrays.asList;

import java.util.Collections;
import java.util.List;
import java8.util.function.Function;
import java8.util.function.Predicate;
import java8.util.function.Supplier;
import java8.util.stream.Collector;
import java8.util.stream.Collectors;
import java8.util.stream.Stream;
import java8.util.stream.StreamSupport;

// tag::documented[]
public interface PartialResult<T> {

  Stream<T> stream();

  List<T> list();

  boolean hasMore();

  PartialResult<T> filter(Predicate<? super T> predicate);

  <C> PartialResult<C> map(Function<? super T, ? extends C> transformer);

  <R, A> R collect(Collector<? super T, A, R> collector);
  // end::documented[]

  class Simple<T> implements PartialResult<T> {
    private final Supplier<Stream<T>> supplier;
    private boolean hasMore;

    public static <T> Simple<T> of(Supplier<Stream<T>> supplier) {
      return new Simple<>(supplier);
    }

    @SafeVarargs
    public static <T> Simple<T> of(final T... it) {
      return new Simple<>(
          new Supplier<Stream<T>>() {
            @Override
            public Stream<T> get() {
              return StreamSupport.stream(asList(it));
            }
          });
    }

    public static <T> PartialResult<T> empty() {
      return new Simple<>(
          new Supplier<Stream<T>>() {
            @Override
            public Stream<T> get() {
              return StreamSupport.stream(Collections.<T>emptyList());
            }
          });
    }

    public static <T> PartialResult<T> buffered(final PartialResult<T> it) {
      final List<T> buffer = it.stream().collect(Collectors.<T>toList());
      return of(new Supplier<Stream<T>>() {
            @Override
            public Stream<T> get() {
              return StreamSupport.stream(buffer);
            }
          })
          .hasMore(it.hasMore());
    }

    public Simple(Supplier<Stream<T>> supplier) {
      this.supplier = supplier;
    }

    public PartialResult<T> hasMore(boolean hasMore) {
      this.hasMore = hasMore;
      return this;
    }

    public Stream<T> stream() {
      return supplier.get();
    }

    @Override
    public List<T> list() {
      return supplier.get().collect(Collectors.<T>toList());
    }

    @Override
    public boolean hasMore() {
      return hasMore;
    }

    @Override
    public <C> PartialResult<C> map(final Function<? super T, ? extends C> transformer) {
      return of(new Supplier<Stream<C>>() {
            @Override
            public Stream<C> get() {
              return Simple.this.stream().map(transformer);
            }
          })
          .hasMore(hasMore);
    }

    @Override
    public PartialResult<T> filter(final Predicate<? super T> predicate) {
      return of(new Supplier<Stream<T>>() {
            @Override
            public Stream<T> get() {
              return Simple.this.stream().filter(predicate);
            }
          })
          .hasMore(hasMore);
    }

    @Override
    public <R, A> R collect(Collector<? super T, A, R> collector) {
      return stream().collect(collector);
    }
  }

  class Wrapper<T> implements PartialResult<T> {
    private final PartialResult<T> target;

    public Wrapper(PartialResult<T> target) {
      this.target = target;
    }

    public Stream<T> stream() {
      return target.stream();
    }

    @Override
    public List<T> list() {
      return target.stream().collect(Collectors.<T>toList());
    }

    @Override
    public boolean hasMore() {
      return target.hasMore();
    }

    @Override
    public PartialResult<T> filter(final Predicate<? super T> predicate) {
      return Simple.of(
              new Supplier<Stream<T>>() {
                @Override
                public Stream<T> get() {
                  return Wrapper.this.stream().filter(predicate);
                }
              })
          .hasMore(hasMore());
    }

    @Override
    @SuppressWarnings("unchecked")
    public <C> PartialResult<C> map(final Function<? super T, ? extends C> transformer) {
      return (PartialResult<C>)
          Simple.of(
                  new Supplier<Stream<C>>() {
                    @Override
                    public Stream<C> get() {
                      return Wrapper.this.stream().map(transformer);
                    }
                  })
              .hasMore(hasMore());
    }

    @Override
    public <R, A> R collect(Collector<? super T, A, R> collector) {
      return stream().collect(collector);
    }
  }
  // tag::documented[]
}
// end::documented[]
