package be.recipe.api.reservation;

import static java.lang.annotation.ElementType.FIELD;
import static java.lang.annotation.RetentionPolicy.RUNTIME;
import static java.util.Arrays.asList;

import be.recipe.api.Prescription;
import be.recipe.api.executor.Executor;
import java.lang.annotation.Documented;
import java.lang.annotation.Inherited;
import java.lang.annotation.Retention;
import java.lang.annotation.Target;
import java.util.Comparator;
import javax.validation.Constraint;
import javax.validation.ConstraintValidator;
import javax.validation.ConstraintValidatorContext;
import javax.validation.Payload;
import org.threeten.bp.LocalDateTime;

// tag::class[]
public interface Reservation {
  // end::class[]
  String status_should_be_x = "reservation.status.should.be.%s";
  String status_should_not_be_x = "reservation.status.should.not.be.%s";
  String not_found = "reservation.not.found";
  String supports_reservations = "supports.reservations";
  String invalid_executor_id_format = "error.validation.vision.invalid.message";
  String prescription_status_invalid = "prescription.status.invalid.for.reservation";

  Reservation NULL = new Simple();

  // tag::methods[]
  Prescription.Encrypted prescription();

  Executor.ID executor();

  LocalDateTime creationDateTime();

  boolean newInfoForPatient();

  boolean newInfoForExecutor();

  Status status();

  String feedbackToPatient();

  LocalDateTime updateDateTime();

  String email();

  String phoneNumber();

  ContactPreference contactPreference();
  // end::methods[]
  // tag::types[]

  enum Status {
    REQUESTED_WITHOUT_COMMITMENT("requested without commitment", 0),
    REQUESTED("requested", 2),
    ACCEPTED("accepted", 2),
    REJECTED("rejected", 1),
    CANCELLATION_REQUESTED("cancellation requested", 2),
    CANCELLED("cancelled", 0),
    FULFILLED("fulfilled", 0);
    // end::types[]

    private String humanReadable;
    private int rank;

    Status(String humanReadable, int rank) {
      this.humanReadable = humanReadable;
      this.rank = rank;
    }

    public static java.util.Comparator<Status> comparator() {
      return new Comparator();
    }

    public String humanReadable() {
      return humanReadable;
    }

    @Inherited
    @Documented
    @Retention(RUNTIME)
    @Target(FIELD)
    @Constraint(validatedBy = ShouldBeOneOf.Validator.class)
    public @interface ShouldBeOneOf {
      String message() default status_should_not_be_x;

      Status[] status();

      Class<?>[] groups() default {};

      Class<? extends Payload>[] payload() default {};

      class Validator implements ConstraintValidator<ShouldBeOneOf, Status> {
        private Status[] expected;

        @Override
        public void initialize(ShouldBeOneOf constraintAnnotation) {
          expected = constraintAnnotation.status();
        }

        @Override
        public boolean isValid(Status status, ConstraintValidatorContext context) {
          return status == null || asList(expected).contains(status);
        }
      }
    }

    private static class Comparator implements java.util.Comparator<Status> {
      @Override
      public int compare(Status x, Status y) {
        return Integer.compare(x.rank, y.rank);
      }
    }
    // tag::types[]
  }
  // end::types[]

  // tag::attributes[]
  enum Attribute {
    modificationDate
  }
  // end::attributes[]

  class Simple implements Reservation {
    private LocalDateTime creationDateTime, updateDateTime;
    private String feedbackToPatient, email, phoneNumber;
    private ContactPreference contactPreference;
    private Prescription.Encrypted prescription;
    private boolean newInfoForPatient;
    private boolean newInfoForExecutor;
    private Executor.ID executor;
    private Status status;

    public static Simple reservation(Prescription.Encrypted prescription) {
      return new Simple().prescription(prescription);
    }

    public static Simple reservation(Executor.ID executorId) {
      return new Simple().executor(executorId);
    }

    @Override
    public Prescription.Encrypted prescription() {
      return prescription;
    }

    @Override
    public Executor.ID executor() {
      return executor;
    }

    @Override
    public LocalDateTime creationDateTime() {
      return creationDateTime;
    }

    @Override
    public boolean newInfoForPatient() {
      return newInfoForPatient;
    }

    @Override
    public boolean newInfoForExecutor() {
      return newInfoForExecutor;
    }

    @Override
    public Status status() {
      return status;
    }

    @Override
    public String feedbackToPatient() {
      return feedbackToPatient;
    }

    @Override
    public LocalDateTime updateDateTime() {
      return updateDateTime;
    }

    @Override
    public String email() {
      return email;
    }

    @Override
    public String phoneNumber() {
      return phoneNumber;
    }

    @Override
    public ContactPreference contactPreference() {
      return contactPreference;
    }

    public Simple creationDateTime(LocalDateTime creationDateTime) {
      this.creationDateTime = creationDateTime;
      return this;
    }

    public Simple executor(Executor.ID executorId) {
      this.executor = executorId;
      return this;
    }

    public Simple prescription(Prescription.Encrypted prescription) {
      this.prescription = prescription;
      return this;
    }

    public Simple status(Status status) {
      this.status = status;
      return this;
    }

    public Simple feedbackToPatient(String feedbackToPatient) {
      this.feedbackToPatient = feedbackToPatient;
      return this;
    }

    public Simple newInfoForPatient(boolean newInfoForPatient) {
      this.newInfoForPatient = newInfoForPatient;
      return this;
    }

    public Simple newInfoForExecutor(boolean newInfoForExecutor) {
      this.newInfoForExecutor = newInfoForExecutor;
      return this;
    }

    public Simple email(String email) {
      this.email = email;
      return this;
    }

    public Simple phoneNumber(String phoneNumber) {
      this.phoneNumber = phoneNumber;
      return this;
    }

    public Simple contactPreference(ContactPreference contactPreference) {
      this.contactPreference = contactPreference;
      return this;
    }
  }
  // tag::class[]
}
// end::class[]
