package be.recipe.api.patient;

import static be.recipe.api.constraints.Constraint.required;
import static be.recipe.api.patient.ListPrescriptions.*;
import static be.recipe.api.prescriber.VisionOtherPrescribers.OPEN;
import static be.recipe.api.viewer.AllViewer.all;

import be.recipe.api.Prescription;
import be.recipe.api.PrescriptionContent;
import be.recipe.api.executor.Executor;
import be.recipe.api.executor.ExecutorViewer;
import be.recipe.api.prescriber.Prescriber;
import be.recipe.api.prescriber.VisionOtherPrescribers;
import be.recipe.api.reservation.ContactPreference;
import be.recipe.api.reservation.Reservation;
import be.recipe.api.reservation.Reservation.Status;
import be.recipe.api.reservation.Reserved;
import be.recipe.api.series.Page;
import be.recipe.api.series.PartialResult;
import be.recipe.api.series.SortedBy;
import java8.util.Optional;
import java8.util.function.Function;

import javax.validation.Valid;
import javax.validation.constraints.NotNull;

import org.threeten.bp.LocalDate;
import org.threeten.bp.LocalDateTime;

// tag::class[]
// tag::class-response[]
public class ListPrescriptions {
  // end::class[]
  // end::class-response[]
  // tag::attributes[]

  public Reserved reserved;

  @Valid
  @NotNull(message = required)
  public Page page = Page.first();

  public SortedBy<Prescription.Attribute>[] sortedBy;

  // end::attributes[]

  public ListPrescriptions() {}

  public ListPrescriptions(Page page) {
    this.page = page;
  }

  public ListPrescriptions(Reserved reserved) {
    this.reserved = reserved;
  }

  public ListPrescriptions(SortedBy<Prescription.Attribute>[] sortedBy) {
    this.sortedBy = sortedBy.length == 0 ? null : sortedBy;
  }

  public ListPrescriptions(
      Page page, Reserved reserved, SortedBy<Prescription.Attribute>[] sortedBy) {
    this.reserved = reserved;
    this.page = page;
    this.sortedBy = (sortedBy == null || sortedBy.length == 0) ? null : sortedBy;
  }

  //  class Request {

  //    public static Request request() {
  //      return new Request();
  //    }

  //    @SafeVarargs
  //    public static Request sortedBy(SortedBy<Prescription.Attribute>... sortedBy) {
  //      final Request request = new Request();
  //      StreamSupport.stream(asList(sortedBy))
  //          .forEach(
  //              new Consumer<SortedBy<Prescription.Attribute>>() {
  //                @Override
  //                public void accept(SortedBy<Prescription.Attribute> it) {
  //                  request.sortedBy(it);
  //                }
  //              });
  //      return request;
  //    }

  //    public Request reserved(Reserved reserved) {
  //      this.reserved = reserved;
  //      return this;
  //    }
  //
  //    public Request reserved() {
  //      return reserved(yes);
  //    }
  //
  //    public Request notReserved() {
  //      return reserved(no);
  //    }
  //
  //    @SuppressWarnings({"unchecked", "UnusedReturnValue"})
  //    public Request sortedBy(SortedBy<Prescription.Attribute> sortedBy) {
  //      this.sortedBy =
  //          StreamSupport.stream(
  //                  asList(
  //                      StreamSupport.stream(Collections.singleton(sortedBy)),
  //                      this.sortedBy == null
  //                          ? StreamSupport.stream(
  //                              Collections.<SortedBy<Prescription.Attribute>>emptyList())
  //                          : StreamSupport.stream(asList(this.sortedBy))))
  //              .flatMap(
  //                  new Function<
  //                      Stream<SortedBy<Prescription.Attribute>>,
  //                      Stream<SortedBy<Prescription.Attribute>>>() {
  //                    @Override
  //                    public Stream<SortedBy<Prescription.Attribute>> apply(
  //                        Stream<SortedBy<Prescription.Attribute>> stream) {
  //                      return stream;
  //                    }
  //                  })
  //              .collect(Collectors.<SortedBy<Prescription.Attribute>>toList())
  //              .toArray(new SortedBy[0]);
  //      return this;
  //    }
  //  }

  // tag::class[]
  public interface Command<
      Request extends ListPrescriptions, Response extends ListPrescriptions.Response> {
    // end::class[]
    // tag::methods[]
    PartialResult<Response> list(Request request);
    // end::methods[]

    // tag::class[]
  }

  // tag::class-response[]
  public interface Response extends be.recipe.api.Registered {
    // end::class-response[]
    // end::class[]
    // tag::methods[]
    // tag::response-methods[]

    Patient.ID patient();

    Prescriber.ID prescriber();

    Prescription.Type type();

    LocalDate expirationDate();

    boolean feedbackAllowed();

    ExecutorViewer visibleToExecutors();

    VisionOtherPrescribers visionOtherPrescribers();

    Prescription.Status status();

    LocalDateTime creationDate();

    Optional<Reservation> reservation();

    // end::methods[]
    // tag::class[]

    interface Reservation {
      // end::class[]
      // tag::methods[]
      Executor.ID executorId();

      Status status();

      LocalDateTime creationDateTime();

      boolean hasUpdates();

      String feedbackToPatient();

      String email();

      String phoneNumber();

      ContactPreference contactPreference();

      // end::methods[]

      class Simple implements Reservation {
        public Executor.ID executorId;
        public LocalDateTime creationDateTime;
        public ContactPreference contactPreference;
        public String feedbackToPatient, email, phoneNumber;
        public boolean hasUpdates;
        public Status status;

        @Override
        public Executor.ID executorId() {
          return executorId;
        }

        @Override
        public Status status() {
          return status;
        }

        @Override
        public LocalDateTime creationDateTime() {
          return creationDateTime;
        }

        @Override
        public boolean hasUpdates() {
          return hasUpdates;
        }

        @Override
        public String feedbackToPatient() {
          return feedbackToPatient;
        }

        @Override
        public String email() {
          return email;
        }

        @Override
        public String phoneNumber() {
          return phoneNumber;
        }

        @Override
        public ContactPreference contactPreference() {
          return contactPreference;
        }
      }
      // tag::class[]
    }

    // end::methods[]
    // end::response-methods[]
    // tag::class-response[]

    interface PlainText extends Response {
      // end::class-response[]
      // tag::response-methods[]
      String prescriptionContent();

      // end::response-methods[]

      class Simple extends AbstractSimpleResponse implements PlainText {
        public String prescriptionContent;

        public Simple(Encrypted from) {
          super(from);
        }

        @Override
        public String prescriptionContent() {
          return prescriptionContent;
        }

        public PlainText.Simple prescriptionContent(String prescriptionContent) {
          this.prescriptionContent = prescriptionContent;
          return this;
        }

        public static Function<PlainText, String> toPrescriptionContent() {
          return new Function<PlainText, String>() {
            @Override
            public String apply(PlainText it) {
              return it.prescriptionContent();
            }
          };
        }
      }

      class Wrapper extends AbstractWrapper<PlainText> implements PlainText {
        public Wrapper(PlainText target) {
          super(target);
        }

        @Override
        public String prescriptionContent() {
          return target.prescriptionContent();
        }
      }
      // tag::class-response[]
    }

    // end::class-response[]

    interface Encrypted extends Response {
      byte[] encryptedPrescriptionContent();

      String encryptionKey();

      class Simple extends AbstractSimpleResponse implements Encrypted {
        public byte[] encryptedPrescriptionContent;
        public String encryptionKey;

        public static Encrypted.Simple response(Prescription.RID rid) {
          Encrypted.Simple it = new Encrypted.Simple();
          it.rid = rid;
          return it;
        }

        public static Encrypted response(Prescription.Encrypted encrypted) {
          return new PrescriptionWrapper(encrypted);
        }

        @Override
        public byte[] encryptedPrescriptionContent() {
          return encryptedPrescriptionContent;
        }

        @Override
        public String encryptionKey() {
          return encryptionKey;
        }

        public Encrypted.Simple prescriptionContent(PrescriptionContent.Encrypted it) {
          encryptionKey = it.key().toString();
          encryptedPrescriptionContent = it.bytes();
          return this;
        }

        public PlainText.Simple prescriptionContent(String prescriptionContent) {
          return new PlainText.Simple(this).prescriptionContent(prescriptionContent);
        }
      }

      class Wrapper extends AbstractWrapper<Encrypted> implements Encrypted {
        public Wrapper(Encrypted target) {
          super(target);
        }

        @Override
        public byte[] encryptedPrescriptionContent() {
          return target.encryptedPrescriptionContent();
        }

        @Override
        public String encryptionKey() {
          return target.encryptionKey();
        }
      }
    }

    abstract class AbstractWrapper<T extends Response> implements Response {
      final T target;

      public AbstractWrapper(T target) {
        this.target = target;
      }

      @Override
      public Prescription.RID rid() {
        return target.rid();
      }

      @Override
      public Prescription.Type type() {
        return target.type();
      }

      @Override
      public Patient.ID patient() {
        return target.patient();
      }

      @Override
      public Prescriber.ID prescriber() {
        return target.prescriber();
      }

      @Override
      public LocalDateTime creationDate() {
        return target.creationDate();
      }

      @Override
      public LocalDate expirationDate() {
        return target.expirationDate();
      }

      @Override
      public ExecutorViewer visibleToExecutors() {
        return target.visibleToExecutors();
      }

      @Override
      public VisionOtherPrescribers visionOtherPrescribers() {
        return target.visionOtherPrescribers();
      }

      @Override
      public Prescription.Status status() {
        return target.status();
      }

      @Override
      public boolean feedbackAllowed() {
        return target.feedbackAllowed();
      }

      @Override
      public Optional<Reservation> reservation() {
        return target.reservation();
      }
      // tag::class[]
      // tag::class-response[]
    }
  }
}

// end::class[]
// end::class-response[]

class AbstractSimpleResponse implements ListPrescriptions.Response {
  public VisionOtherPrescribers visionOtherPrescribers = OPEN;
  public ExecutorViewer visibleToExecutors = all();
  public Prescription.Status status;
  public LocalDateTime creationDate;
  public Prescription.RID rid;
  public Patient.ID patient;
  public Prescriber.ID prescriber;
  public Prescription.Type type;
  public LocalDate expirationDate;
  public boolean feedbackAllowed;
  public Optional<Reservation> reservation;

  public AbstractSimpleResponse() {}

  public AbstractSimpleResponse(ListPrescriptions.Response from) {
    rid = from.rid();
    patient = from.patient();
    creationDate = from.creationDate();
    prescriber = from.prescriber();
    status = from.status();
    type = from.type();
    expirationDate = from.expirationDate();
    visibleToExecutors = from.visibleToExecutors();
    visionOtherPrescribers = from.visionOtherPrescribers();
    reservation = from.reservation();
  }

  @Override
  public Prescription.RID rid() {
    return rid;
  }

  @Override
  public Patient.ID patient() {
    return patient;
  }

  @Override
  public Prescriber.ID prescriber() {
    return prescriber;
  }

  @Override
  public Prescription.Type type() {
    return type;
  }

  @Override
  public LocalDate expirationDate() {
    return expirationDate;
  }

  @Override
  public boolean feedbackAllowed() {
    return feedbackAllowed;
  }

  @Override
  public ExecutorViewer visibleToExecutors() {
    return visibleToExecutors;
  }

  @Override
  public VisionOtherPrescribers visionOtherPrescribers() {
    return visionOtherPrescribers;
  }

  @Override
  public Prescription.Status status() {
    return status;
  }

  @Override
  public LocalDateTime creationDate() {
    return creationDate;
  }

  @Override
  public Optional<Reservation> reservation() {
    return reservation;
  }
}

class PrescriptionWrapper implements ListPrescriptions.Response.Encrypted {
  private final Prescription.Encrypted target;

  public PrescriptionWrapper(Prescription.Encrypted target) {
    this.target = target;
  }

  @Override
  public Prescription.RID rid() {
    return target.rid();
  }

  @Override
  public Patient.ID patient() {
    return target.patient().id();
  }

  @Override
  public Prescriber.ID prescriber() {
    return target.prescriber().id();
  }

  @Override
  public Prescription.Type type() {
    return target.type();
  }

  @Override
  public LocalDate expirationDate() {
    return target.expirationDate();
  }

  @Override
  public byte[] encryptedPrescriptionContent() {
    return target.encryptedContent();
  }

  @Override
  public String encryptionKey() {
    return target.encryptionKey();
  }

  @Override
  public boolean feedbackAllowed() {
    return target.feedbackAllowed();
  }

  @Override
  public ExecutorViewer visibleToExecutors() {
    return target.visibleToExecutors();
  }

  @Override
  public VisionOtherPrescribers visionOtherPrescribers() {
    return target.visionOtherPrescribers();
  }

  @Override
  public Prescription.Status status() {
    return target.status();
  }

  @Override
  public LocalDateTime creationDate() {
    return target.creationDate();
  }

  @Override
  public Optional<Reservation> reservation() {
    return target
        .reservation()
        .map(
            new Function<be.recipe.api.reservation.Reservation, Reservation>() {
              @Override
              public Reservation apply(be.recipe.api.reservation.Reservation reservation) {
                return new ReservationWrapper(reservation);
              }
            });
  }
}

class ReservationWrapper implements Response.Reservation {
  private final Reservation target;

  ReservationWrapper(Reservation target) {
    this.target = target;
  }

  @Override
  public Executor.ID executorId() {
    return target.executor();
  }

  @Override
  public Status status() {
    return target.status();
  }

  @Override
  public LocalDateTime creationDateTime() {
    return target.creationDateTime();
  }

  @Override
  public boolean hasUpdates() {
    return target.newInfoForPatient();
  }

  @Override
  public String feedbackToPatient() {
    return target.feedbackToPatient();
  }

  @Override
  public String email() {
    return target.email();
  }

  @Override
  public String phoneNumber() {
    return target.phoneNumber();
  }

  @Override
  public ContactPreference contactPreference() {
    return target.contactPreference();
  }
}
