package be.recipe.api.crypto;

import static java.nio.charset.StandardCharsets.UTF_8;

import be.recipe.api.Prescription;
import be.recipe.api.crypto.Message.Cipher.Key;
import be.recipe.api.crypto.Message.Encrypted.Timestamped;
import be.recipe.api.text.ExtendableString;
import java.util.Arrays;
import java8.util.function.Function;
import java8.util.function.Predicate;
import javax.crypto.SecretKey;

public class Message {
  protected final Cipher cipher;
  protected final byte[] bytes;

  public Message(Cipher cipher, byte[] bytes) {
    this.cipher = cipher;
    this.bytes = bytes;
  }

  public byte[] bytes() {
    return bytes;
  }

  public interface Cipher extends be.recipe.api.crypto.Cipher<Cipher.Key> {
    interface Key {
      ID id();

      SecretKey secret();

      class ID extends ExtendableString {
        private ID(String value) {
          super(value);
        }

        public static ID keyID(String value) {
          return construct(value, new Function<String, ID>() {
            @Override
            public ID apply(String it) {
              return new ID(it);
            }
          });
        }
      }

      class Simple implements Key {
        private final ID id;
        private final SecretKey spec;

        public static Key key(ID id, SecretKey spec) {
          return new Simple(id, spec);
        }

        public Simple(ID id, SecretKey spec) {
          this.id = id;
          this.spec = spec;
        }

        @Override
        public ID id() {
          return id;
        }

        @Override
        public SecretKey secret() {
          return spec;
        }
      }

      interface DB<V> {
        @SuppressWarnings("unchecked")
        Key create(String patientId);

        Key get(ID id);
      }

      interface Owner<V> {
        void process(V processor);

        class Simple<V> extends ExtendableString implements Owner<V> {
          static <V> Owner<V> owner(String it) {
            return new Simple<>(it);
          }

          static <V> Owner<V> owner(ExtendableString it) {
            return owner(it.toString());
          }

          static <V> Owner<V> owner(byte[] it) {
            return owner(new String(it, UTF_8));
          }

          private Simple(String value) {
            super(value);
          }

          @Override
          public void process(V processor) {}
        }
      }

      class NotFound extends IllegalArgumentException {}

      class ByID implements Predicate<Key> {
        private final ID id;

        static ByID byID(ID id) {
          return new ByID(id);
        }

        public ByID(ID id) {
          this.id = id;
        }

        @Override
        public boolean test(Key key) {
          return key.id().equals(id);
        }
      }
    }
  }

  public static class Factory {
    private final Cipher cipher;
    private final Timestamped.Extractor unmarshaller;

    public Factory(Cipher cipher, Timestamped.Extractor unmarshaller) {
      this.cipher = cipher;
      this.unmarshaller = unmarshaller;
    }

    public PlainText plain(byte[] it) {
      return new PlainText(it, cipher);
    }

    public Encrypted encrypted(byte[] bytes) {
      return new Encrypted(bytes, cipher);
    }

    public Timestamped timestamped(byte[] bytes) {
      return new Timestamped(bytes, cipher, unmarshaller);
    }
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (!(o instanceof Message)) return false;
    Message that = (Message) o;
    return Arrays.equals(bytes, that.bytes);
  }

  @Override
  public int hashCode() {
    return Arrays.hashCode(bytes);
  }

  public static class Encrypted extends Message {
    public Encrypted(byte[] bytes, Cipher cipher) {
      super(cipher, bytes);
    }

    public PlainText decrypt(Key key) {
      return decrypt(key, bytes);
    }

    private PlainText decrypt(Key key, byte[] bytes) {
      return new PlainText(bytes == null ? null : cipher.decrypt(bytes, key), cipher);
    }

    public static class Timestamped extends Encrypted {
      private final Extractor unmarshaller;

      public Timestamped(byte[] bytes, Cipher cipher, Extractor unmarshaller) {
        super(bytes, cipher);
        this.unmarshaller = unmarshaller;
      }

      @Override
      public PlainText decrypt(Key key) {
        try {
          return super.decrypt(key, unmarshaller.encrypted(bytes));
        } catch (ExtractionFailed e) {
          return super.decrypt(key, bytes);
        }
      }

      public interface Context extends Compactor, Extractor {}

      public interface Compactor {
        byte[] compact(byte[] it);
      }

      public interface Extractor {
        byte[] encrypted(byte[] it);
      }

      public static class ExtractionFailed extends IllegalArgumentException {
        public ExtractionFailed() {}

        public ExtractionFailed(Throwable cause) {
          super(cause);
        }
      }
    }
  }

  public static class PlainText extends Message {
    public PlainText(byte[] bytes, Cipher cipher) {
      super(cipher, bytes);
    }

    public Encrypted encipher(Key key) {
      return new Encrypted(cipher.encrypt(bytes, key), cipher);
    }

    @Override
    public String toString() {
      return new String(bytes, UTF_8);
    }
  }
}
