package be.recipe.api.constraints;

import be.recipe.api.text.LocalisedMessage;
import java8.util.function.Function;
import java8.util.stream.Stream;
import java8.util.stream.StreamSupport;

import java.util.ArrayList;
import java.util.List;

// tag::class[]
public interface Violation {
  String name();
  // end::class[]

  String propertyPath();

  String usecase();

  <T> T value();

  Stream<LocalisedMessage> translations();

  class Simple implements Violation {
    private final List<LocalisedMessage> translations = new ArrayList<>();
    private String name, propertyPath, usecase;
    private Object value;

    public static Simple.Builder violation(String name) {
      return new Simple.Builder().name(name);
    }

    @Override
    public String name() {
      return name;
    }

    @Override
    public String propertyPath() {
      return propertyPath;
    }

    @Override
    public String usecase() {
      return usecase;
    }

    @Override
    @SuppressWarnings("unchecked")
    public <T> T value() {
      return (T) value;
    }

    @Override
    public Stream<LocalisedMessage> translations() {
      return StreamSupport.stream(translations);
    }

    public static Function<Violation, String> toViolation() {
      return new Function<Violation, String>() {
        @Override
        public String apply(Violation violation) {
          return violation.name();
        }
      };
    }

    public static class Builder {
      private final Simple violation = new Simple();

      public Builder name(String name) {
        violation.name = name;
        return this;
      }

      public Violation build() {
        return violation;
      }

      public static Function<Builder, Violation> toViolation() {
        return new Function<Builder, Violation>() {
          @Override
          public Violation apply(Violation.Simple.Builder builder) {
            return builder.build();
          }
        };
      }

      public Builder on(String propertyPath) {
        violation.propertyPath = propertyPath;
        return this;
      }

      public Builder of(String usecase) {
        violation.usecase = usecase;
        return this;
      }

      public Builder value(Object value) {
        violation.value = value;
        return this;
      }

      public Builder translation(LocalisedMessage translation) {
        violation.translations.add(translation);
        return this;
      }
    }

    @Override
    public String toString() {
      return "Violation{" + "propertyPath='" + propertyPath + '\'' + "name='" + name + '\'' + '}';
    }
  }

  interface PropertyPathFactory {
    String propertyPath(String violation);
  }

  interface ValueFactory {
    Object violationValue(String violation);
  }

  class Wrapper implements Violation {
    private final Violation target;

    public Wrapper(Violation target) {
      this.target = target;
    }

    @Override
    public String name() {
      return target.name();
    }

    @Override
    public String propertyPath() {
      return target.propertyPath();
    }

    @Override
    public String usecase() {
      return target.usecase();
    }

    @Override
    public <T> T value() {
      return target.value();
    }

    @Override
    public Stream<LocalisedMessage> translations() {
      return target.translations();
    }
  }
  // tag::class[]
}
// end::class[]
