package be.recipe.api;

import static be.recipe.api.Prescription.Type.prescriptionType;
import static be.recipe.api.crypto.Message.Cipher.Key.ID.keyID;
import static java.nio.charset.StandardCharsets.UTF_8;

import be.recipe.api.crypto.Message;
import be.recipe.api.executor.ArchivePrescription;
import be.recipe.api.executor.GetPrescriptionAndPutInProcess;
import be.recipe.api.executor.GetProfile;
import be.recipe.api.executor.UpdateProfile;
import be.recipe.api.patient.GetExecutorProfile;
import be.recipe.api.patient.PutVisionExecutors;
import be.recipe.api.patient.PutVisionOtherPrescribers;
import be.recipe.api.prescriber.*;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.zip.GZIPInputStream;
import java.util.zip.GZIPOutputStream;

import be.recipe.api.series.PartialResult;
import java8.util.Optional;
import java8.util.function.Function;

public class CipheringPrescriptionService
    extends PrescriptionService.Ciphering.Simplified.Abstract {
  private final PrescriptionService.Simplified target;
  private final PrescriptionContent.Factory prescriptionContentFactory;

  private Message.Cipher.Key.Owner<Prescription.OnContent> prescriber;

  public CipheringPrescriptionService(
      PrescriptionService.Simplified target,
      PrescriptionContent.Factory prescriptionContentFactory) {
    this.target = target;
    this.prescriptionContentFactory = prescriptionContentFactory;
  }

  public CipheringPrescriptionService(
      PrescriptionService.Simplified target,
      PrescriptionContent.Factory prescriptionContentFactory,
      Message.Cipher.Key.Owner<Prescription.OnContent> prescriber) {
    this(target, prescriptionContentFactory);
    this.prescriber = prescriber;
  }

  @Override
  public ListPrescriptions.PartialResult<ListPrescriptions.Response.PlainText> list(
      ListPrescriptions request) {
    ListPrescriptions.PartialResult<ListPrescriptions.Response.Encrypted> source =
        target.list(request);
    return ListPrescriptions.PartialResult.Simple.wrap(
            source.map(
                new Function<
                    ListPrescriptions.Response.Encrypted, ListPrescriptions.Response.PlainText>() {
                  @Override
                  public ListPrescriptions.Response.PlainText apply(
                      ListPrescriptions.Response.Encrypted encrypted) {
                    return new DecryptingListPrescriptionsResponseForPrescriber(encrypted);
                  }
                }))
        .hasHidden(source.hasHidden());
  }

  @Override
  public be.recipe.api.series.PartialResult<
          be.recipe.api.executor.ListPrescriptions.Response.PlainText>
      list(be.recipe.api.executor.ListPrescriptions request) {
    return target
        .list(request)
        .map(
            new Function<
                be.recipe.api.executor.ListPrescriptions.Response.Encrypted,
                be.recipe.api.executor.ListPrescriptions.Response.PlainText>() {
              @Override
              public be.recipe.api.executor.ListPrescriptions.Response.PlainText apply(
                  be.recipe.api.executor.ListPrescriptions.Response.Encrypted encrypted) {
                return new DecryptingListPrescriptionsResponseForExecutor(encrypted);
              }
            });
  }

  @Override
  public Prescription.PlainText get(be.recipe.api.prescriber.GetPrescription request) {
    return new DecryptingPrescription(target.get(request));
  }

  @Override
  public Prescription.PlainText add(
      CreatePrescription<Prescription.PlainText.Specification> request) {
    return new DecryptingPrescription(target.add(request(request)));
  }

  @Override
  public Prescription.PlainText get(be.recipe.api.executor.GetPrescription request) {
    return new DecryptingPrescription(target.get(request));
  }

  @Override
  public Prescription.PlainText getAndPutInProcess(GetPrescriptionAndPutInProcess request) {
    return new DecryptingPrescription(target.getAndPutInProcess(request));
  }

  @Override
  public PartialResult<be.recipe.api.patient.ListPrescriptions.Response.PlainText> list(
      be.recipe.api.patient.ListPrescriptions request) {
    return target
        .list(request)
        .map(
            new Function<
                be.recipe.api.patient.ListPrescriptions.Response.Encrypted,
                be.recipe.api.patient.ListPrescriptions.Response.PlainText>() {
              @Override
              public be.recipe.api.patient.ListPrescriptions.Response.PlainText apply(
                  be.recipe.api.patient.ListPrescriptions.Response.Encrypted encrypted) {
                return new DecryptingListPrescriptionsResponseForPatient(encrypted);
              }
            });
  }

  @Override
  public void update(PutVisionExecutors.Request request) {
    target.update(request);
  }

  private CreatePrescription<Prescription.Encrypted.Specification> request(
      CreatePrescription<Prescription.PlainText.Specification> request) {
    // tag::create-plain-prescription-content[]
    Prescription.PlainText.Specification customizations = request.customizations;
    // end::create-plain-prescription-content[]
    // tag::new-prescription[]
    Prescription.Encrypted.Specification spec =
        new Prescription.Encrypted.Specification(customizations);
    // end::new-prescription[]
    if (customizations.content != null) {
      // tag::create-plain-prescription-content[]
      PrescriptionContent.PlainText plainText =
          prescriptionContentFactory.create(compress(customizations.content.getBytes(UTF_8)));
      // end::create-plain-prescription-content[]
      // tag::encipher-prescription-content[]
      PrescriptionContent.Encrypted encrypted =
          plainText.encrypt(customizations.patient.id(), prescriptionType("P0"));
      // end::encipher-prescription-content[]
      // tag::new-prescription[]
      spec.encryptionKeyId = encrypted.key().toString();
      spec.encryptedContent = encrypted.bytes();
      // end::new-prescription[]
    }
    return new CreatePrescription<>(spec);
  }

  // tag::compression[]
  public static byte[] compress(byte[] bytes) {
    try (ByteArrayOutputStream baos = new ByteArrayOutputStream()) {
      try (GZIPOutputStream out = new GZIPOutputStream(baos)) {
        out.write(bytes);
      }
      return baos.toByteArray();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  // end::compression[]

  // tag::decompression[]
  private byte[] decompress(byte[] bytes) {
    if (bytes == null) return bytes;
    try (ByteArrayOutputStream out = new ByteArrayOutputStream()) {
      try (ByteArrayInputStream bais = new ByteArrayInputStream(bytes)) {
        try (GZIPInputStream in = new GZIPInputStream(bais)) {
          int i;
          byte[] buffer = new byte[1024];
          while ((i = in.read(buffer)) > 0) {
            out.write(buffer, 0, i);
          }
        }
      }
      return out.toByteArray();
    } catch (IOException e) {
      throw new RuntimeException(e);
    }
  }

  // end::decompression[]

  @Override
  public void update(be.recipe.api.prescriber.PutVisionOtherPrescribers request) {
    target.update(request);
  }

  @Override
  public GetExecutorProfile.Response getProfile(GetExecutorProfile request) {
    return target.getProfile(request);
  }

  @Override
  public void updateProfile(UpdateProfile request) {
    target.updateProfile(request);
  }

  @Override
  public GetProfile.Response getProfile(GetProfile request) {
    return target.getProfile(request);
  }

  @Override
  public void update(be.recipe.api.prescriber.RevokePrescription request) {
    target.update(request);
  }

  @Override
  public void update(be.recipe.api.patient.RevokePrescription request) {
    target.update(request);
  }

  @Override
  public void update(ArchivePrescription request) {
    target.update(request);
  }

  @Override
  public void update(PutVisionOtherPrescribers request) {
    target.update(request);
  }

  @Override
  public GetPrescriptionStatusResponse get(GetPrescriptionStatus request) {
    return target.get(request);
  }

  @Override
  public Prescription.PlainText get(be.recipe.api.patient.GetPrescription request) {
    return new DecryptingPrescription(target.get(request));
  }

  @Override
  public GetPrescriptionStatusResponse get(
      be.recipe.api.executor.GetPrescriptionStatus getPrescriptionStatus) {
    return target.get(getPrescriptionStatus);
  }

  @Override
  public GetPrescriptionStatusResponse get(be.recipe.api.patient.GetPrescriptionStatus request) {
    return target.get(request);
  }

  private class DecryptingPrescription extends Prescription.Encrypted.Wrapper
      implements Prescription.PlainText {
    private final Encrypted target;

    public DecryptingPrescription(Encrypted target) {
      super(target);
      this.target = target;
    }

    @Override
    public String content() {
      return Optional.ofNullable(target.encryptionKey())
          .flatMap(
              new Function<String, Optional<String>>() {
                @Override
                public Optional<String> apply(final String encryptionKey) {
                  return Optional.ofNullable(target.encryptedContent())
                      .map(
                          new Function<byte[], String>() {
                            @Override
                            public String apply(byte[] encryptedContent) {
                              return decrypt(encryptedContent, encryptionKey);
                            }
                          });
                }
              })
          .orElse(null);
    }
  }

  private String decrypt(byte[] encryptedContent, String encryptionKey) {
    // tag::decipher-prescription-content[]
    PrescriptionContent.Encrypted encrypted =
        prescriptionContentFactory.create(encryptedContent, keyID(encryptionKey));
    String content =
        Optional.ofNullable(decompress(encrypted.decrypt().bytes()))
            .map(
                new Function<byte[], String>() {
                  @Override
                  public String apply(byte[] it) {
                    return new String(it, UTF_8);
                  }
                })
            .orElse("");
    // end::decipher-prescription-content[]
    return content;
  }

  private class DecryptingListPrescriptionsResponseForPrescriber
      extends ListPrescriptions.Response.Encrypted.Wrapper
      implements ListPrescriptions.Response.PlainText {
    public DecryptingListPrescriptionsResponseForPrescriber(Encrypted target) {
      super(target);
    }

    @Override
    public String prescriptionContent() {
      return decrypt(encryptedPrescriptionContent(), encryptionKey());
    }
  }

  private class DecryptingListPrescriptionsResponseForExecutor
      extends be.recipe.api.executor.ListPrescriptions.Response.Encrypted.Wrapper
      implements be.recipe.api.executor.ListPrescriptions.Response.PlainText {
    public DecryptingListPrescriptionsResponseForExecutor(Encrypted target) {
      super(target);
    }

    @Override
    public String prescriptionContent() {
      return decrypt(encryptedPrescriptionContent(), encryptionKey());
    }
  }

  private class DecryptingListPrescriptionsResponseForPatient
      extends be.recipe.api.patient.ListPrescriptions.Response.Encrypted.Wrapper
      implements be.recipe.api.patient.ListPrescriptions.Response.PlainText {
    public DecryptingListPrescriptionsResponseForPatient(Encrypted target) {
      super(target);
    }

    @Override
    public String prescriptionContent() {
      return decrypt(encryptedPrescriptionContent(), encryptionKey());
    }
  }

  public static class Factory {
    private final PrescriptionService.Simplified target;
    private final PrescriptionContent.Factory prescriptionContantFactory;

    public Factory(
        PrescriptionService.Simplified target,
        PrescriptionContent.Factory prescriptionContantFactory) {
      this.target = target;
      this.prescriptionContantFactory = prescriptionContantFactory;
    }

    public CipheringPrescriptionService create(
        Message.Cipher.Key.Owner<Prescription.OnContent> prescriber) {
      return new CipheringPrescriptionService(target, prescriptionContantFactory, prescriber);
    }
  }
}
