package be.business.connector.core.utils;

import be.business.connector.core.domain.KgssIdentifierType;
import be.business.connector.core.ehealth.services.KeyDepotService;
import be.business.connector.core.ehealth.services.KeyDepotServiceImpl;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.exceptions.IntegrationModuleRuntimeException;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionToken;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionTokenFactory;
import org.perf4j.aop.Profiled;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.InputStream;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.apache.commons.lang.StringUtils.isBlank;

public class ETKHelper {

  private static final Logger LOG = LoggerFactory.getLogger(ETKHelper.class);
  private static final String RIZIV_CBE = "0206653946";
  private static final String EHEALTH_CBE = "0809394427";
  private static final String MY_ETK_PROPERTY = "MY_ETK";

  private PropertyHandler propertyHandler;
  private EncryptionUtils encryptionUtils;

  private static Map<String, List<EncryptionToken>> etksCache = new HashMap<>();
  private KeyDepotService keyDepotService = KeyDepotServiceImpl.getInstance();

  public ETKHelper(PropertyHandler propertyHandler, EncryptionUtils encryptionUtils) {
    this.propertyHandler = propertyHandler;
    this.encryptionUtils = encryptionUtils;
  }

  @Profiled(
      logFailuresSeparately = true,
      tag = "0.ETKHelper#getKGSS_ETK",
      logger = "org.perf4j.TimingLogger_Common")
  public List<EncryptionToken> getKGSS_ETK() throws IntegrationModuleException {
    return getEtks(KgssIdentifierType.CBE, EHEALTH_CBE, "KGSS");
  }

  @Profiled(
      logFailuresSeparately = true,
      tag = "0.ETKHelper#getRecipe_ETK",
      logger = "org.perf4j.TimingLogger_Common")
  public List<EncryptionToken> getRecipe_ETK() throws IntegrationModuleException {
    String recipeCbe = propertyHandler.getProperty("etk-helper.recipe-cbe");
    if (isBlank(recipeCbe)) {
      recipeCbe = RIZIV_CBE;
    }
    return getEtks(KgssIdentifierType.CBE, recipeCbe, "RECIPE");
  }

  @Profiled(
      logFailuresSeparately = true,
      tag = "0.ETKHelper#getSystemETK",
      logger = "org.perf4j.TimingLogger_Common")
  public List<EncryptionToken> getSystemETK() throws IntegrationModuleException {
    String application = "";
    KgssIdentifierType identifierType = null;
    String identifierValue = "";

    if (propertyHandler.hasProperty(MY_ETK_PROPERTY)) {
      String myETK = propertyHandler.getProperty(MY_ETK_PROPERTY);
      if (myETK.indexOf(';') > -1) {
        String[] etk = myETK.split(";");
        if (etk.length > 1) {
          identifierType = KgssIdentifierType.lookup(etk[0].toUpperCase());
          identifierValue = etk[1];
          if (etk.length >= 3) {
            application = etk[2];
          }
        }
      } else {
        try {
          InputStream etkStream = IOUtils.getResourceAsStream(myETK);
          byte[] etk = IOUtils.getBytes(etkStream);
          List<EncryptionToken> encryptionTokens = new ArrayList<>();
          EncryptionToken etkToken = EncryptionTokenFactory.getInstance().create(etk);
          encryptionTokens.add(etkToken);
          return encryptionTokens;
        } catch (Exception e) {
          throw new IntegrationModuleException("Invalid ETK", e);
        }
      }
    } else {
      X509Certificate certificate = encryptionUtils.getCertificate();
      if (certificate != null) {
        try {
          certificate.checkValidity();
        } catch (CertificateExpiredException e) {
          throw new IntegrationModuleRuntimeException(
              I18nHelper.getLabel("error.expired.system.certificate"), e);
        } catch (CertificateNotYetValidException e) {
          throw new IntegrationModuleRuntimeException(
              I18nHelper.getLabel("error.invalid.system.certificate"), e);
        }
      } else {
        throw new IntegrationModuleRuntimeException(
            I18nHelper.getLabel("error.notfound.system.certificate"));
      }
      CertificateParser parser = new CertificateParser(certificate);
      identifierType = KgssIdentifierType.lookup(parser.getType());
      identifierValue = parser.getValue();
      application = parser.getApplication();
    }

    if (identifierType == null || identifierValue == null || "".equals(identifierValue)) {
      throw new IntegrationModuleException(I18nHelper.getLabel("error.invalid.system.certificate"));
    }
    return getEtks(identifierType, Long.valueOf(identifierValue), application);
  }

  public List<EncryptionToken> getEtks(KgssIdentifierType identifierType, String identifierValue)
      throws IntegrationModuleException {
    if (isBlank(identifierValue)) {
      throw new IntegrationModuleException(I18nHelper.getLabel("error.invalid.etk"));
    }
    return getEtks(identifierType, identifierValue, "");
  }

  public List<EncryptionToken> getEtks(
      KgssIdentifierType identifierType, Long identifierValue, String applicationid)
      throws IntegrationModuleException {
    if (null != identifierType) {
      switch (identifierType) {
        case CBE:
          return getEtks(KgssIdentifierType.CBE, longToString(identifierValue, 10), applicationid);
        case SSIN:
          return getEtks(KgssIdentifierType.SSIN, longToString(identifierValue, 11), applicationid);
        case NIHII_PHARMACY:
          return getEtks(
              KgssIdentifierType.NIHII_PHARMACY, longToString(identifierValue, 8), applicationid);
        case NIHII_HOSPITAL:
          return getEtks(
              KgssIdentifierType.NIHII_HOSPITAL, longToString(identifierValue, 8), applicationid);
      }
    }
    // ETK_ID_TYPE_NIHII
    return getEtks(
        KgssIdentifierType.NIHII, longToString(identifierValue / 1000, 8), applicationid);
  }

  private List<EncryptionToken> getEtks(
      KgssIdentifierType identifierType, String identifierValue, String application)
      throws IntegrationModuleException {
    String etkCacheId = identifierType + "/" + identifierValue + "/" + application;
    if (etksCache.containsKey(etkCacheId)) {
      LOG.info("ETK retrieved from the cache : " + etkCacheId);
      return etksCache.get(etkCacheId);
    }
    List<EncryptionToken> encryptionTokens =
        getEtksFromDepot(identifierType, identifierValue, application);
    etksCache.put(etkCacheId, encryptionTokens);
    return encryptionTokens;
  }

  private List<EncryptionToken> getEtksFromDepot(
      KgssIdentifierType identifierType, String identifierValue, String application)
      throws IntegrationModuleException {
    try {
      List<EncryptionToken> encryptiontokens =
          keyDepotService.retrieveEtk(identifierType, identifierValue, application);
      return encryptiontokens;
    } catch (Throwable t) {
      Exceptionutils.errorHandler(t);
    }
    return null;
  }

  public static String longToString(Long id, int numberOfDigits) {
    if (id == null) {
      return null;
    }

    StringBuilder buffer = new StringBuilder(Long.toString(id.longValue()));
    int delta = numberOfDigits - buffer.length();

    if (delta == 0) {
      return buffer.toString();
    }
    if (delta < 0) {
      throw new IllegalArgumentException("numberOfDigits < input length");
    }
    for (; delta > 0; --delta) {
      buffer.insert(0, "0");
    }
    return buffer.toString();
  }

  public static String subString(String id, int numberOfDigits) {
    if (isBlank(id)) {
      return null;
    }

    StringBuilder buffer = new StringBuilder(id);
    int delta = numberOfDigits - buffer.length();

    if (delta == 0) {
      return buffer.toString();
    }
    if (delta < 0) {
      throw new IllegalArgumentException("numberOfDigits < input length");
    }
    for (; delta > 0; --delta) {
      buffer.insert(0, "0");
    }
    return buffer.toString();
  }
}
