package be.business.connector.common.ehealth;

import be.business.connector.core.ehealth.services.KgssService;
import be.business.connector.core.ehealth.services.KgssServiceImpl;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.utils.ETKHelper;
import be.business.connector.core.utils.EncryptionUtils;
import be.business.connector.core.utils.PropertyHandler;
import be.ehealth.technicalconnector.service.kgss.domain.KeyResult;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionToken;
import be.recipe.api.Prescription;
import be.recipe.api.crypto.Message;
import be.recipe.api.executor.Executor;
import be.recipe.api.patient.Patient;
import be.recipe.api.prescriber.Prescriber;
import be.recipe.api.prescriber.PrescriberType;
import java8.util.function.Consumer;
import java8.util.function.Functions;
import java8.util.stream.RefStreams;
import java8.util.stream.Stream;

import java.util.ArrayList;
import java.util.List;

import static be.recipe.api.crypto.Message.Cipher.Key.ID.keyID;
import static be.recipe.api.prescriber.PrescriberType.*;
import static java.nio.charset.StandardCharsets.UTF_8;

public class EhealthKeyRegistry implements Message.Cipher.Key.DB<Prescription.OnContent> {
  // tag::prescription-content[]
  private static final String the_prescription_prescriber =
          "urn:be:fgov:identification-namespace,urn:be:fgov:person:ssin:ehealth:1.0:nihii:%TYPE%:nihii11,%PRESCRIBER_NIHII%";
  private static final String the_patient =
      "urn:be:fgov:identification-namespace,urn:be:fgov:person:ssin,%PATIENT_SSIN%";
  private static final String known_dentists =
      "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:dentist:nihii11,";
  private static final String known_physicians =
          "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:doctor:nihii11,";
  private static final String known_midwifes =
      "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:nihii:midwife:nihii11,";
  private static final String known_nurses =
      "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:nurse:nihii11,";
  private static final String known_hospitals =
      "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:ehealth:1.0:hospital:nihii-number,";
  private static final String known_executors =
      "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:ehealth:1.0:pharmacy:nihii-number:recognisedpharmacy:boolean,true";

  // end::prescription-content[]

  private final KgssService kgss = KgssServiceImpl.getInstance();
  private PropertyHandler configuration;
  private ETKHelper etkHelper;

  public EhealthKeyRegistry() {
    this(PropertyHandler.getInstance());
  }

  public EhealthKeyRegistry(PropertyHandler configuration) {
    this.configuration = configuration;
  }

  @Override
  public final Message.Cipher.Key create(String patientId) {
    List<Message.Cipher.Key.Owner<Prescription.OnContent>> owners = new ArrayList<>();
    owners.add(Executor.All.all());
    owners.add(Patient.ID.patientID(patientId));
    owners.add(DOCTOR);
    owners.add(HOSPITAL);
    owners.add(DENTIST);
    owners.add(MIDWIFE);
    return new CreateKeyCommand()
        .execute(owners.toArray(new Message.Cipher.Key.Owner[0]));
  }

  @Override
  public Message.Cipher.Key get(Message.Cipher.Key.ID id) {
    try {
      EncryptionToken token = etkHelper.getKGSS_ETK().get(0);
      EncryptionToken systemToken = etkHelper.getSystemETK().get(0);
      KeyResult result =
          kgss.retrieveKeyFromKgss(
              id.toString().getBytes(UTF_8), systemToken.getEncoded(), token.getEncoded());
      return key(result);
    } catch (IntegrationModuleException e) {
      throw new Message.Cipher.Key.NotFound();
    }
  }

  private Message.Cipher.Key key(KeyResult result) {
    return Message.Cipher.Key.Simple.key(keyID(result.getKeyId()), result.getSecretKey());
  }

  public void refresh(PropertyHandler props) {
    refresh(new ETKHelper(props, new EncryptionUtils(props)), props);
  }

  public void refresh(ETKHelper etkHelper, PropertyHandler props) {
    this.etkHelper = etkHelper;
    configuration = props;
    kgss.clear();
  }

  private class CreateKeyCommand implements Prescription.OnContent {
    private final List<String> credentialTypes = new ArrayList<>();

    @SafeVarargs
    @SuppressWarnings("unchecked")
    public final Message.Cipher.Key execute(Message.Cipher.Key.Owner<Prescription.OnContent>... owners) {
      RefStreams.of(
              RefStreams.<Message.Cipher.Key.Owner<Prescription.OnContent>>of(prescriber()),
              RefStreams.of(owners))
          .flatMap(Functions.<Stream<Message.Cipher.Key.Owner<Prescription.OnContent>>>identity())
          .forEach(
              new Consumer<Message.Cipher.Key.Owner<Prescription.OnContent>>() {
                @Override
                public void accept(Message.Cipher.Key.Owner<Prescription.OnContent> it) {
                  it.process(CreateKeyCommand.this);
                }
              });
      EncryptionToken token = etkHelper.getKGSS_ETK().get(0);
      EncryptionToken systemToken = etkHelper.getSystemETK().get(0);
      return key(
          kgss.retrieveNewKey(
              token.getEncoded(), credentialTypes, null, null, null, systemToken.getEncoded()));
    }

    @Override
    public void process(Patient.ID patient) {
      credentialTypes.add(the_patient.replaceAll("%PATIENT_SSIN%", patient.toString()));
    }

    @Override
    public void process(Executor.ID executor) {
      throw new UnsupportedOperationException();
    }

    @Override
    public void process(Executor.All executor) {
      credentialTypes.add(known_executors);
    }

    @Override
    public void process(PrescriberType type) {
      switch (type) {
        case DOCTOR:
          credentialTypes.add(known_physicians);
          break;
        case HOSPITAL:
          credentialTypes.add(known_hospitals);
          break;
        case DENTIST:
          credentialTypes.add(known_dentists);
          break;
        case MIDWIFE:
          credentialTypes.add(known_midwifes);
          break;
        default:
          throw new IllegalArgumentException("Type " + type + "!");
      }
    }

    @Override
    public void process(Prescriber prescriber) {
      if (prescriber.type() != HOSPITAL)
        credentialTypes.add(
            the_prescription_prescriber
                .replaceAll("%TYPE%", prescriber.type().name().toLowerCase())
                .replaceAll("%PRESCRIBER_NIHII%", prescriber.id().toString()));
    }
  }

  private Prescriber.Simple prescriber() {
    return Prescriber.Simple.prescriber(
        prescriberType(configuration.getProperty("sessionmanager.samlattributedesignator.3")),
        Prescriber.ID.prescriberID(configuration.getProperty("user.nihii")));
  }

  public static PrescriberType prescriberType(String type) {
    switch (type) {
      case "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:doctor:nihii11":
        return DOCTOR;
      case "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:dentist:nihii11":
        return DENTIST;
      case "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:nihii:dentist:nihii11":
        return DENTIST;
      case "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:nihii:midwife:nihii11":
        return MIDWIFE;
      case "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:nihii:nurse:nihii11":
        return NURSE;
      case "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:person:ssin:ehealth:1.0:nihii:physiotherapist:nihii11":
        return PHYSIOTHERAPIST;
      case "urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:ehealth:1.0:certificateholder:hospital:nihii-number":
        return HOSPITAL;
      default:
        throw new IllegalArgumentException("Unknown prescriber type! [" + type + "]");
    }
  }
}
