/*
 * (C) 2021 Recip-e. All rights reserved.
 */
package be.business.connector.recipe.utils;

import static be.recipe.api.Prescription.Type.prescriptionType;
import static be.recipe.api.prescriber.PrescriberType.DOCTOR;
import static be.recipe.api.prescriber.PrescriberType.HOSPITAL;

import be.business.connector.common.ehealth.EhealthKeyRegistry;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.utils.ETKHelper;
import be.business.connector.core.utils.EncryptionUtils;
import be.business.connector.core.utils.PropertyHandler;
import be.business.connector.recipe.prescriber.AbstractPrescriberIntegrationModule;
import be.business.connector.recipe.prescriber.PrescriberIntegrationModuleV4Impl;
import be.business.connector.recipe.prescriber.dto.CreatePrescriptionDTO;
import be.ehealth.technicalconnector.service.kgss.domain.KeyResult;
import be.recipe.api.Prescription;
import be.recipe.api.crypto.Message;
import be.recipe.api.executor.Executor;
import be.recipe.api.patient.Patient;
import be.recipe.services.executor.ListOpenPrescriptionsResult;
import java.security.Key;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Semaphore;
import org.perf4j.aop.Profiled;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PrescriberEncryptionUtils {

  /** The Constant LOG. */
  private static final Logger LOG = LoggerFactory.getLogger(PrescriberEncryptionUtils.class);

  /** The etk helper. */
  private final ETKHelper etkHelper;

  /** The key cache. */
  private final Map<String, KeyResult> keyCache;

  private EhealthKeyRegistry keyRegistry;

  private final PropertyHandler propertyHandler;

  private final Key symmKey;

  public PrescriberEncryptionUtils(
      final PropertyHandler propertyHandler,
      final EncryptionUtils encryptionUtils,
      final Map<String, KeyResult> keyCache) {
    this.propertyHandler = propertyHandler;
    this.etkHelper = new ETKHelper(propertyHandler, encryptionUtils);
    this.symmKey = EncryptionUtils.getInstance().generateSecretKey();
    this.keyCache = keyCache;
    keyRegistry = new EhealthKeyRegistry(propertyHandler);
    keyRegistry.refresh(etkHelper, propertyHandler);
  }

  /**
   * Encrypt a list of {@link CreatePrescriptionDTO}s.
   *
   * @param createPrescriptionDTOs the list of {@link CreatePrescriptionDTO}s
   * @param prescriberIntegrationModuleV4 the {@link AbstractPrescriberIntegrationModule}
   * @return the encrypted {@link ListOpenPrescriptionsResult}
   */
  public List<CreatePrescriptionDTO> doEncryptions(
      final List<CreatePrescriptionDTO> createPrescriptionDTOs,
      final AbstractPrescriberIntegrationModule prescriberIntegrationModuleV4) {
    final int threadLimit =
        PropertyHandler.getInstance().getIntegerProperty("encryption.thread.number", "50");
    final Semaphore semaphore = new Semaphore(threadLimit, true);
    final List<PrescriptionEncryptorThread> dataList = new ArrayList<>();
    for (final CreatePrescriptionDTO createPrescriptionDTO : createPrescriptionDTOs) {
      if (createPrescriptionDTO.getPrescription() != null) {
        final KeyResult key =
            getNewKey(
                createPrescriptionDTO.getPatientId(), createPrescriptionDTO.getPrescriptionType());
        final PrescriptionEncryptorThread encryptorThread =
            new PrescriptionEncryptorThread(
                semaphore,
                createPrescriptionDTO,
                key,
                symmKey,
                etkHelper,
                prescriberIntegrationModuleV4);
        dataList.add(encryptorThread);
      }
    }

    for (final PrescriptionEncryptorThread encryptorThread : dataList) {
      semaphore.acquireUninterruptibly();
      encryptorThread.start();
    }
    try {
      semaphore.acquireUninterruptibly(threadLimit);
    } catch (final IllegalArgumentException e) {
      LOG.debug("Incorrect Thread configuration : " + e);
      throw new IntegrationModuleException(e.getMessage(), e);
    }
    final List<CreatePrescriptionDTO> finalResult = new ArrayList<>();
    for (final PrescriptionEncryptorThread decryptorThread : dataList) {
      finalResult.add(decryptorThread.getCreatePrescriptionDTO());
    }

    for (final CreatePrescriptionDTO createPrescriptionDTO : createPrescriptionDTOs) {
      for (final CreatePrescriptionDTO finalDTO : finalResult) {
        if (createPrescriptionDTO.getSequenceNumber() == finalDTO.getSequenceNumber()) {
          createPrescriptionDTO.setRid(finalDTO.getRid());
          createPrescriptionDTO.setErrorOccured(finalDTO.isErrorOccured());
          createPrescriptionDTO.setException(finalDTO.getException());
          if (finalDTO.isErrorOccured()) {
            LOG.info(
                String.format(
                    "Error occured during doEncryptions-operations[%s]:[%s] ",
                    finalDTO.getSequenceNumber(), finalDTO.getException()));
          }
        }
      }
    }

    return createPrescriptionDTOs;
  }

  /**
   * Gets the new key.
   *
   * @param patientId the patient id
   * @param prescriptionType the prescription type
   * @return the new key @ the integration module exception
   */
  public KeyResult getNewKey(final String patientId, final String prescriptionType) {
    KeyResult key;
    if (keyCache.containsKey(patientId)) {
      key = keyCache.get(patientId);
    } else {
      key =
          getNewKeyFromKgss(patientId);
      keyCache.put(patientId, key);
    }
    return key;
  }

  @Profiled(logFailuresSeparately = true, tag = "0.PrescriberIntegrationModule#getNewKeyFromKgss")
  protected KeyResult getNewKeyFromKgss(final String patientId) {
    Message.Cipher.Key key =
        keyRegistry.create(patientId);
    return new KeyResult(key.secret(), key.id().toString());
  }
}
