package be.business.connector.recipe.patient;

import static be.business.connector.recipe.AbstractRecipeClient.programId;
import static be.business.connector.recipe.utils.RidValidator.validateRid;

import java.util.List;
import java.util.UUID;

import org.perf4j.aop.Profiled;

import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.handlers.InsurabilityHandler;
import be.business.connector.core.utils.Exceptionutils;
import be.business.connector.core.utils.I18nHelper;
import be.business.connector.core.utils.MarshallerHelper;
import be.business.connector.core.utils.PropertyHandler;
import be.business.connector.recipe.patient.services.RecipePatientServiceDevV4Impl;
import be.business.connector.recipe.utils.RidValidator;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionToken;
import be.recipe.services.patient.CreateReservation;
import be.recipe.services.patient.CreateReservationParam;
import be.recipe.services.patient.CreateReservationResponse;
import be.recipe.services.patient.CreateReservationResult;
import be.recipe.services.patient.GetPrescriptionForPatient;
import be.recipe.services.patient.GetPrescriptionForPatientResponse;
import be.recipe.services.patient.GetPrescriptionForPatientResult;
import be.recipe.services.patient.GetPrescriptionStatus;
import be.recipe.services.patient.GetPrescriptionStatusParam;
import be.recipe.services.patient.GetPrescriptionStatusResponse;
import be.recipe.services.patient.GetPrescriptionStatusResult;
import be.recipe.services.patient.GetVision;
import be.recipe.services.patient.GetVisionParam;
import be.recipe.services.patient.GetVisionResponse;
import be.recipe.services.patient.GetVisionResult;
import be.recipe.services.patient.ListOpenRids;
import be.recipe.services.patient.ListOpenRidsParam;
import be.recipe.services.patient.ListOpenRidsResponse;
import be.recipe.services.patient.ListOpenRidsResult;
import be.recipe.services.patient.ListPatientPrescription;
import be.recipe.services.patient.ListPatientPrescriptionResponse;
import be.recipe.services.patient.ListPatientPrescriptionsParam;
import be.recipe.services.patient.ListPatientPrescriptionsResult;
import be.recipe.services.patient.PutVision;
import be.recipe.services.patient.PutVisionParam;
import be.recipe.services.patient.PutVisionResponse;
import be.recipe.services.patient.PutVisionResult;
import be.recipe.services.patient.RevokePrescription;
import be.recipe.services.patient.RevokePrescriptionResponse;
import be.recipe.services.patient.RevokePrescriptionResult;

import javax.xml.ws.WebServiceException;

/**
 * The Class PatientIntegrationModuleV4Impl.
 * 
 * @author <a href="mailto:bruno.casneuf@healthconnect.be">Bruno Casneuf</a>
 */
public class PatientIntegrationModuleDevV4Impl extends AbstractPatientIntegrationModule implements PatientIntegrationModuleDevV4 {

	public PatientIntegrationModuleDevV4Impl() {
		super();
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public void revokePrescription(final String rid, final String reason) {
		RidValidator.validateRid(rid);
		// ApplicationConfig.getInstance().assertValidSession();
		try {
			final byte[] sealedRevokePrescriptionParam = getSealedRevokePrescriptionParam(rid, reason);

			final RevokePrescription request = new RevokePrescription();
			request.setRevokePrescriptionParamSealed(sealedRevokePrescriptionParam);
			request.setProgramIdentification(programId(getClass().getSimpleName()));
			request.setMguid(UUID.randomUUID().toString());

			try {
				final RevokePrescriptionResponse response = RecipePatientServiceDevV4Impl.getInstance().revokePrescription(request);
				final MarshallerHelper<RevokePrescriptionResult, RevokePrescriptionResult> helper = new MarshallerHelper<>(
						RevokePrescriptionResult.class, RevokePrescriptionResult.class);
				final RevokePrescriptionResult result = helper
						.unsealWithSymmKey(response.getRevokePrescriptionResultSealed(), getSymmKey());
				checkStatus(result);
			} catch (final WebServiceException cte) {
				throw new IntegrationModuleException(I18nHelper.getLabel("error.connection.executor"), cte);
			}

		} catch (final Throwable t) {
			Exceptionutils.errorHandler(t);

		}
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public GetPrescriptionForPatientResult getPrescription(final String rid) {
		RidValidator.validateRid(rid);
		// //ApplicationConfig.getInstance().assertValidSession();
		InsurabilityHandler.setInsurability(null);
		InsurabilityHandler.setMessageId(null);

		try {
			final byte[] sealedContent = getSealedGetPrescriptionForPatientParam(rid);

			final GetPrescriptionForPatient request = new GetPrescriptionForPatient();
			request.setGetPrescriptionForPatientParamSealed(sealedContent);
			request.setProgramIdentification(programId(getClass().getSimpleName()));
			request.setMguid(UUID.randomUUID().toString());

			GetPrescriptionForPatientResponse response = null;
			try {
				response = RecipePatientServiceDevV4Impl.getInstance().getPrescriptionForPatient(request);
			} catch (final WebServiceException cte) {
				throw new IntegrationModuleException(I18nHelper.getLabel("error.connection.executor"), cte);
			}

			final GetPrescriptionForPatientResult finalResult = unsealPrescription(response.getGetPrescriptionForPatientResultSealed());
			return finalResult;
		} catch (final Throwable t) {
			Exceptionutils.errorHandler(t);
		}
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Profiled(logFailuresSeparately = true, tag = "0.PatientIntegrationModuleV4#getData(GetVisionParam)")
	@Override
	public GetVisionResult getData(final GetVisionParam data) {
		RidValidator.validateRid(data.getRid());
		// ApplicationConfig.getInstance().assertValidSession();
		try {
			final GetVision request = getVisionRequest(data);
			try {
				final GetVisionResponse getDataResponse = RecipePatientServiceDevV4Impl.getInstance().getVision(request);
				final GetVisionResult getVisionResult = unsealGetVisionResponse(getDataResponse);
				checkStatus(getVisionResult);
				return getVisionResult;
			} catch (final WebServiceException cte) {
				throw new IntegrationModuleException(I18nHelper.getLabel("error.connection.executor"), cte);
			}
		} catch (final Throwable t) {
			Exceptionutils.errorHandler(t);
		}
		return null;
	}

	/**
	 * Unseal get data response.
	 *
	 * @param getVisionResponse
	 *            the get data response
	 * @return the gets the vision response @ the integration module exception
	 */
	private GetVisionResult unsealGetVisionResponse(final GetVisionResponse getVisionResponse) {
		final MarshallerHelper<GetVisionResult, Object> marshaller = new MarshallerHelper<>(GetVisionResult.class, Object.class);
		final GetVisionResult result = marshaller.unsealWithSymmKey(getVisionResponse.getGetVisionResultSealed(), getSymmKey());
		return result;
	}

	/**
	 * Unseal put vision.
	 *
	 * @param putVisionResponse
	 *            the data response
	 * @return the put vision response @ the integration module exception
	 */
	private PutVisionResult unsealPutVisionResponse(final PutVisionResponse putVisionResponse) {
		final MarshallerHelper<PutVisionResult, Object> marshaller = new MarshallerHelper<>(PutVisionResult.class, Object.class);
		final PutVisionResult result = marshaller.unsealWithSymmKey(putVisionResponse.getPutVisionResultSealed(), getSymmKey());
		return result;
	}




	/**
	 * Unseal put reservation.
	 *
	 * @param dataResponse
	 *            the data response
	 * @return the put reservation response @ the integration module exception
	 */
	private CreateReservationResult unsealCreateReservationResponse(final CreateReservationResponse dataResponse) {
		final MarshallerHelper<CreateReservationResult, Object> marshaller = new MarshallerHelper<>(CreateReservationResult.class, Object.class);
		final CreateReservationResult result = marshaller.unsealWithSymmKey(dataResponse.getCreateReservationResultSealed(), getSymmKey());
		return result;
	}

	/**
	 * {@inheritDoc}
	 */
	@Profiled(logFailuresSeparately = true, tag = "0.PatientIntegrationModuleV4#putVision")
	@Override
	public PutVisionResult putData(final PutVisionParam putVisionParam) {
		RidValidator.validateRid(putVisionParam.getRid());
		// ApplicationConfig.getInstance().assertValidSession();
		try {
			final PutVision request = putVision(putVisionParam);
			try {
				final PutVisionResponse response = RecipePatientServiceDevV4Impl.getInstance().putVision(request);
				final PutVisionResult result = unsealPutVisionResponse(response);
				checkStatus(result);
				return result;
			} catch (final WebServiceException cte) {
				throw new IntegrationModuleException(I18nHelper.getLabel("error.connection.executor"), cte);
			}
		} catch (final Throwable t) {
			Exceptionutils.errorHandler(t);
		}
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Profiled(logFailuresSeparately = true, tag = "0.PatientIntegrationModuleV4#putReservation")
	@Override
	public CreateReservationResult putData(final CreateReservationParam data) {
		RidValidator.validateRid(data.getRid());
		// ApplicationConfig.getInstance().assertValidSession();
		try {
			final CreateReservation putReservation = putReservationRequest(data);
			try {
				final CreateReservationResponse response = RecipePatientServiceDevV4Impl.getInstance().createReservation(putReservation);
				final CreateReservationResult result = unsealCreateReservationResponse(response);
				checkStatus(result);
				return result;
			} catch (final WebServiceException cte) {
				throw new IntegrationModuleException(I18nHelper.getLabel("error.connection.executor"), cte);
			}
		} catch (final Throwable t) {
			Exceptionutils.errorHandler(t);
		}
		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Override
	public ListPatientPrescriptionsResult listOpenPrescriptions(ListPatientPrescriptionsParam listPatientPrescriptionsParam) {
		// ApplicationConfig.getInstance().assertValidSession();
		try {
			// init helper
			final MarshallerHelper<ListPatientPrescriptionsResult, ListPatientPrescriptionsParam> helper = new MarshallerHelper<>(
					ListPatientPrescriptionsResult.class, ListPatientPrescriptionsParam.class);

			// get recipe etk
			final List<EncryptionToken> etkRecipes = getEtkHelper().getRecipe_ETK();

			// create param
			final ListPatientPrescriptionsParam param = new ListPatientPrescriptionsParam();
			param.setSymmKey(getSymmKey().getEncoded());

			// create request
			final ListPatientPrescription request = new ListPatientPrescription();
			request.setListPatientPrescriptionsParamSealed(sealRequest(etkRecipes.get(0), helper.toXMLByteArray(param)));
			request.setProgramIdentification(programId(getClass().getSimpleName()));
			request.setMguid(UUID.randomUUID().toString());

			// call sealed WS
			ListPatientPrescriptionResponse response = null;
			try {
				response = RecipePatientServiceDevV4Impl.getInstance().listOpenPrescriptions(request);
			} catch (final WebServiceException cte) {
				throw new IntegrationModuleException(I18nHelper.getLabel("error.connection.prescriber"), cte);
			}

			// unseal WS response
			final ListPatientPrescriptionsResult result = helper.unsealWithSymmKey(response.getListPatientPrescriptionsResultSealed(), getSymmKey());

			checkStatus(result);
			return result;

		} catch (final Throwable t) {
			Exceptionutils.errorHandler(t);
		}

		return null;
	}

	/**
	 * {@inheritDoc}
	 */
	@Profiled(logFailuresSeparately = true, tag = "0.PatientIntegrationModuleV4#getData(GetPrescriptionStatusParam)")
	@Override
	public GetPrescriptionStatusResult getData(final GetPrescriptionStatusParam data) {
		validateRid(data.getRid());
		// ApplicationConfig.getInstance().assertValidSession();
		try {

			final GetPrescriptionStatus getPrescriptionStatus = getGetPrescriptionStatusRequest(data);
			try {
				final GetPrescriptionStatusResponse response = RecipePatientServiceDevV4Impl.getInstance()
						.getPrescriptionStatus(getPrescriptionStatus);
				final GetPrescriptionStatusResult result = unsealGetPrescriptionStatusResponse(response);
				checkStatus(result);
				return result;
			} catch (final WebServiceException cte) {
				throw new IntegrationModuleException(I18nHelper.getLabel("error.connection.executor"), cte);
			}
		} catch (final Throwable t) {
			Exceptionutils.errorHandler(t);
		}
		return null;
	}

	/**
	 * Unseal get prescription status response.
	 *
	 * @param response
	 *            the get data response
	 * @return the gets the prescription status response @ the integration module exception
	 */
	private GetPrescriptionStatusResult unsealGetPrescriptionStatusResponse(final GetPrescriptionStatusResponse response) {
		final MarshallerHelper<GetPrescriptionStatusResult, Object> marshaller = new MarshallerHelper<>(GetPrescriptionStatusResult.class,
				Object.class);
		return marshaller.unsealWithSymmKey(response.getGetPrescriptionStatusResultSealed(), getSymmKey());
	}


	/**
	 * {@inheritDoc}
	 */
	@Profiled(logFailuresSeparately = true, tag = "0.PatientIntegrationModuleV4#getData(ListOpenPrescriptionsParam)")
	@Override
	public ListOpenRidsResult getData(final ListOpenRidsParam data) {
		// ApplicationConfig.getInstance().assertValidSession();
		try {

			final ListOpenRids request = getListOpenRids(data);
			try {
				final ListOpenRidsResponse getDataResponse = RecipePatientServiceDevV4Impl.getInstance().listOpenRids(request);
				final ListOpenRidsResult unsealedResponse = unsealListOpenPrescriptionResponse(getDataResponse);
				checkStatus(unsealedResponse);
				return unsealedResponse;
			} catch (final WebServiceException cte) {
				throw new IntegrationModuleException(I18nHelper.getLabel("error.connection.executor"), cte);
			}
		} catch (final Throwable t) {
			Exceptionutils.errorHandler(t);
		}
		return null;
	}

	/**
	 * Unseal list open prescriptions response.
	 *
	 * @param response
	 *            the response
	 * @return the list open prescriptions result @ the integration module exception
	 */
	private ListOpenRidsResult unsealListOpenPrescriptionResponse(final ListOpenRidsResponse response) {
		final MarshallerHelper<ListOpenRidsResult, Object> marshaller = new MarshallerHelper<>(ListOpenRidsResult.class, Object.class);
		return marshaller.unsealWithSymmKey(response.getListOpenRidsResultSealed(), getSymmKey());
	}

	/**
	 * Gets the vision request.
	 *
	 * @param data
	 *            the data
	 * @return the vision request @ the integration module exception
	 */
	protected GetVision getVisionRequest(final GetVisionParam data) {
		data.setSymmKey(getSymmKey().getEncoded());
		final GetVision getVision = new GetVision();
		getVision.setGetVisionParamSealed(getSealedData(data));
		getVision.setProgramIdentification(programId(getClass().getSimpleName()));
		getVision.setMguid(UUID.randomUUID().toString());
		return getVision;
	}

	/**
	 * Put vision request.
	 *
	 * @param data
	 *            the data
	 * @return the put data @ the integration module exception
	 */
	protected PutVision putVision(final PutVisionParam data) {
		data.setSymmKey(getSymmKey().getEncoded());
		final PutVision putVision = new PutVision();
		putVision.setPutVisionParamSealed(getSealedData(data));
		putVision.setProgramIdentification(programId(getClass().getSimpleName()));
		putVision.setMguid(UUID.randomUUID().toString());
		return putVision;
	}

	/**
	 * Put reservation request.
	 *
	 * @param data
	 *            the data
	 * @return the put data @ the integration module exception
	 */
	protected CreateReservation putReservationRequest(final CreateReservationParam data) {
		data.setSymmKey(getSymmKey().getEncoded());
		final CreateReservation putReservation = new CreateReservation();
		putReservation.setCreateReservationParamSealed(getSealedData(data));
		putReservation.setProgramIdentification(programId(getClass().getSimpleName()));
		putReservation.setMguid(UUID.randomUUID().toString());
		return putReservation;
	}

	/**
	 * Gets the gets the prescription status request.
	 *
	 * @param data
	 *            the data
	 * @return the gets the prescription status request @ the integration module exception
	 */
	protected GetPrescriptionStatus getGetPrescriptionStatusRequest(final GetPrescriptionStatusParam data) {
		data.setSymmKey(getSymmKey().getEncoded());
		final GetPrescriptionStatus getPrescriptionStatus = new GetPrescriptionStatus();
		getPrescriptionStatus.setGetPrescriptionStatusParamSealed(getSealedData(data));
		getPrescriptionStatus.setProgramIdentification(programId(getClass().getSimpleName()));
		getPrescriptionStatus.setMguid(UUID.randomUUID().toString());
		return getPrescriptionStatus;
	}


	/**
	 * Gets the list open prescriptions request.
	 *
	 * @param data
	 *            the data
	 * @return the list open prescriptions request @ the integration module exception
	 */
	protected ListOpenRids getListOpenRids(final ListOpenRidsParam data) {
		data.setSymmKey(getSymmKey().getEncoded());
		final ListOpenRids listOpenPrescription = new ListOpenRids();
		listOpenPrescription.setListOpenRidsParamSealed(getSealedData(data));
		listOpenPrescription.setProgramIdentification(programId(getClass().getSimpleName()));
		listOpenPrescription.setMguid(UUID.randomUUID().toString());
		return listOpenPrescription;
	}

	/**
	 * Gets the sealed data.
	 *
	 * @param request
	 *            the request
	 * @return the sealed data @ the integration module exception
	 */
	private byte[] getSealedData(final GetPrescriptionStatusParam request) {
		request.setSymmKey(getSymmKey().getEncoded());
		return sealForRecipe(request, GetPrescriptionStatusParam.class);
	}


	/**
	 * Gets the sealed data.
	 *
	 * @param request
	 *            the request
	 * @return the sealed data @ the integration module exception
	 */
	private byte[] getSealedData(final ListOpenRidsParam request) {
		request.setSymmKey(getSymmKey().getEncoded());
		return sealForRecipe(request, ListOpenRidsParam.class);
	}

	/**
	 * Gets the sealed data.
	 *
	 * @param request
	 *            the request
	 * @return the sealed data @ the integration module exception
	 */
	private byte[] getSealedData(final CreateReservationParam request) {
		request.setSymmKey(getSymmKey().getEncoded());
		return sealForRecipe(request, CreateReservationParam.class);
	}

	/**
	 * Gets the sealed data.
	 *
	 * @param request
	 *            the request
	 * @return the sealed data @ the integration module exception
	 */
	private byte[] getSealedData(final GetVisionParam request) {
		request.setSymmKey(getSymmKey().getEncoded());
		return sealForRecipe(request, GetVisionParam.class);
	}

	/**
	 * Gets the sealed data.
	 *
	 * @param request
	 *            the request
	 * @return the sealed data @ the integration module exception
	 */
	private byte[] getSealedData(final PutVisionParam request) {
		request.setSymmKey(getSymmKey().getEncoded());
		return sealForRecipe(request, PutVisionParam.class);
	}

}