package be.business.connector.recipe;

import be.business.connector.common.ehealth.EhealthCipher;
import be.business.connector.common.ehealth.EhealthKeyRegistry;
import be.business.connector.core.utils.PropertyHandler;
import be.business.connector.recipe.executor.RecipeExecutorClient;
import be.business.connector.recipe.executor.TimestampingContext;
import be.business.connector.recipe.patient.RecipePatientClient;
import be.business.connector.recipe.prescriber.RecipePrescriberClient;
import be.recipe.api.MandateService;
import be.recipe.api.PrescriptionService;
import be.recipe.api.Recipe;
import be.recipe.api.crypto.Message;
import be.recipe.api.reservation.ReservationService;
import java.util.UUID;
import java8.util.function.Supplier;
import javax.xml.datatype.DatatypeConfigurationException;

public class RecipeClientInfrastructure implements Recipe.Infrastructure {
  private final EhealthKeyRegistry ehealthKeyRegistry = new EhealthKeyRegistry();
  private final EhealthCipher cipher = new EhealthCipher();
  private final RecipePrescriberClient prescriberClient;
  private final RecipeExecutorClient executorClient;
  private final RecipePatientClient patientClient;

  public RecipeClientInfrastructure(final Dependencies deps) throws DatatypeConfigurationException {
    prescriberClient = new RecipePrescriberClient(ehealthKeyRegistry, cipher, traceId(deps));
    executorClient = new RecipeExecutorClient(ehealthKeyRegistry, cipher, traceId(deps));
    patientClient = new RecipePatientClient(ehealthKeyRegistry, cipher, traceId(deps));
  }

  private static Supplier<String> traceId(final Dependencies deps) {
    return new Supplier<String>() {
      @Override
      public String get() {
        return deps.traceId();
      }
    };
  }

  public RecipeClientInfrastructure(PropertyHandler props) throws DatatypeConfigurationException {
    this(
        new Dependencies() {
          @Override
          public String traceId() {
            return UUID.randomUUID().toString();
          }
        });
    refresh(props);
  }

  @Override
  public RoutingPrescriptionService prescriptionService() {
    return new RoutingPrescriptionService(prescriberClient, executorClient, patientClient);
  }

  public RecipePrescriberClient prescriberClient() {
    return prescriberClient;
  }

  public RecipeExecutorClient executorClient() {
    return executorClient;
  }

  public RecipePatientClient patientClient() {
    return patientClient;
  }

  @Override
  public EhealthCipher cipher() {
    return cipher;
  }

  @Override
  public EhealthKeyRegistry keyRegistry() {
    return ehealthKeyRegistry;
  }

  @Override
  public Message.Encrypted.Timestamped.Extractor timestampingContext() {
    return new TimestampingContext();
  }

  @Override
  public ReservationService.Simplified reservationService() {
    return new RoutingReservationService(executorClient, patientClient);
  }

  @Override
  public MandateService.Simplified mandateService() {
    return new RoutingMandateService(executorClient);
  }

  @SuppressWarnings({"rawtypes", "UnnecessaryLocalVariable"})
  public static PrescriptionService.Ciphering client() throws DatatypeConfigurationException {
    // tag::create-client[]
    RecipeClientInfrastructure infrastructure =
        new RecipeClientInfrastructure(PropertyHandler.getInstance());
    Recipe recipe = new Recipe(infrastructure);
    PrescriptionService.Ciphering client = recipe.cipheringPrescriptionService();
    // end::create-client[]
    return client;
  }

  private void refresh(PropertyHandler props) {
    prescriberClient.refresh(props);
  }

  public interface Dependencies {
    String traceId();

    //    default String traceId() {
    //      return UUID.randomUUID().toString();
    //    }
  }
}
