package be.apb.gfddpp.common.utils;

import java.io.ByteArrayInputStream;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBElement;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Marshaller;
import javax.xml.bind.Unmarshaller;
import javax.xml.bind.annotation.XmlRootElement;
import javax.xml.stream.XMLInputFactory;
import javax.xml.stream.XMLStreamException;

import be.recipe.common.exceptions.RecipeException;

public class JaxContextCentralizer {

	private static JaxContextCentralizer instance;

	private Map<Class<?>, JAXBContext> contextStore;

	private XMLInputFactory xmlInputFactory = XMLInputFactory.newInstance();

	private JaxContextCentralizer() {

		if (contextStore == null) {
			contextStore = new HashMap<>();
			// try {
			// getContext(TimestampedPrescription.class);
			// } catch (RecipeException e) {
			// throw new RecipeException(e.getMessage(), e);
			// }
		}
	}

	public static JaxContextCentralizer getInstance() {
		if (instance == null) {
			instance = new JaxContextCentralizer();
		}
		return instance;
	}

	public synchronized void addContext(Class<?> clazz) throws RecipeException {

		try {
			getContext(clazz);
		} catch (RecipeException e) {
			throw new RecipeException(e.getMessage(), e);
		}
	}

	public final JAXBContext getContext(Class<?> clazz) throws RecipeException {
		if (!contextStore.containsKey(clazz)) {
			try {
				contextStore.put(clazz, JAXBContext.newInstance(clazz));
			} catch (JAXBException e) {
				String message = processJAXBException(e);
				throw new RecipeException(message, e);
			}
		}
		return contextStore.get(clazz);
	}

	public Unmarshaller getUnmarshaller(Class<?> clazz) throws RecipeException {

		try {
			return getContext(clazz).createUnmarshaller();
		} catch (JAXBException e) {
			throw new RecipeException(e.getMessage(), e);
		}
	}

	public Marshaller getMarshaller(Class<?> clazz) throws RecipeException {

		try {
			return getContext(clazz).createMarshaller();
		} catch (JAXBException e) {
			throw new RecipeException(e.getMessage(), e);
		}
	}

	public <X> X toObject(Class<X> clazz, String data) throws RecipeException {
		try {
			return toObject(clazz, data.getBytes("UTF-8"));
		} catch (UnsupportedEncodingException e) {
			throw new RecipeException(e.getMessage(), e);
		}
	}

	@SuppressWarnings("unchecked")
	public <X> X toObject(Class<X> clazz, byte[] data) throws RecipeException {
		try {
			ByteArrayInputStream bis = new ByteArrayInputStream(data);
			X result;
			if (clazz.getAnnotation(XmlRootElement.class) != null) {
				result = (X) getUnmarshaller(clazz).unmarshal(bis);
			} else {
				try {
					JAXBElement<X> jax = getUnmarshaller(clazz).unmarshal(xmlInputFactory.createXMLStreamReader(bis, "UTF-8"), clazz);
					result = jax.getValue();
				} catch (XMLStreamException e) {
					throw new RecipeException(e.getMessage(), e);
				}
			}
			return result;
		} catch (JAXBException e) {
			String message = processJAXBException(e);
			throw new RecipeException(message, e);
		}
	}

	private String processJAXBException(JAXBException e) {
		if (e.getLinkedException() != null) {
			return e.getLinkedException().getMessage();
		} else {
			return e.getLocalizedMessage();
		}
	}

	public String toXml(Class<?> clazz, Object obj) throws RecipeException {
		StringWriter sw = new StringWriter();
		Marshaller marshaller = getMarshaller(clazz);
		try {
			marshaller.setProperty(Marshaller.JAXB_FORMATTED_OUTPUT, Boolean.TRUE);

			if (clazz.getAnnotation(XmlRootElement.class) != null) {
				marshaller.marshal(obj, sw);
			} else {
				javax.xml.bind.JAXB.marshal(obj, sw);
			}

		} catch (JAXBException e) {
			throw new RecipeException(e.getMessage(), e);
		}
		return sw.toString();
	}
}