package be.recipe.common.exceptions;

import java.text.MessageFormat;
import java.util.Locale;
import java.util.MissingResourceException;
import java.util.ResourceBundle;
import java.util.SortedMap;
import java.util.TreeMap;

import be.recipe.common.exceptions.RecipeExceptionDetails.ErrorMap;
import be.recipe.utils.UTF8Control;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * The Class RecipeExceptionHelper handles I18N of error messages. Set the following system property to false for non
 * verbose stack traces: * -Dcom.sun.xml.ws.fault.SOAPFaultBuilder.disableCaptureStackTrace=false
 */
public class RecipeExceptionFactory {

    /**
     * The Constant RESOURCE_BUNDLE.
     */
    public static final String RESOURCE_BUNDLE = "RecipeErrors";

    /**
     * The Constant DUTCH_LOCALE.
     */
    public static final Locale DUTCH_LOCALE = new Locale("nl", "BE");

    /**
     * The Constant FRENCH_LOCALE.
     */
    public static final Locale FRENCH_LOCALE = new Locale("fr", "FR");

    /**
     * The Constant ENGLISH_LOCALE.
     */
    public static final Locale ENGLISH_LOCALE = Locale.ENGLISH;

    /**
     * The Constant LOG.
     */
    private final static Logger LOG = LoggerFactory.getLogger(RecipeExceptionFactory.class);

    /**
     * The Constant SEPERATOR.
     */
    public final static String SEPERATOR = ";";

    /**
     * The Constant LOCALES.
     */
    public final static Locale[] LOCALES = new Locale[]{ENGLISH_LOCALE, DUTCH_LOCALE, FRENCH_LOCALE};

    /**
     * Gets the functional exception.
     *
     * @param errorCode the error code
     * @param context the context
     * @return the functional exception
     */
    public static RecipeException createException(final String errorCode, final Object[] context) {
        final RecipeExceptionDetails recipeDetails = getRecipeDetails(errorCode, context);
        String message = errorCode;
        try {
            final ErrorMap errorMap = recipeDetails.getErrorMap();
            for(final RecipeExceptionDetails.ErrorMap.Entry entry: errorMap.entries) {
            	if(entry.key.equals("en")) {
            		message = entry.value.getMessage();
            		break;
            	}
            }
        } finally {
            LOG.error("Error: " + message);
        }
        return new RecipeException(message, recipeDetails);
    }

    /**
     * Gets the functional exception.
     *
     * @param errorCode the error code
     * @param context the context
     * @return the functional exception
     */
    public static RecipeException createException(final Object[] context, final String errorCode) {

        final RecipeExceptionDetails recipeDetails = getRecipeDetails(errorCode, context);
        String message = errorCode;
        try {
            final ErrorMap errorMap = recipeDetails.getErrorMap();
            for(final RecipeExceptionDetails.ErrorMap.Entry entry: errorMap.entries) {
            	if(entry.key.equals("en")) {
            		message = entry.value.getMessage();
            		break;
            	}
            }
        } catch (final RuntimeException ex) {

        } finally {
            LOG.error("Erro: " + message);
        }
        return new RecipeException(message, recipeDetails);
    }

    /**
     * Gets the functional exception.
     *
     * @param errorCode the error code
     * @param context the context
     * @param cause the cause
     * @return the functional exception
     */
    public static RecipeException createException(final String errorCode, final Object[] context, final Throwable cause) {

        if (cause instanceof RecipeException) {
            return (RecipeException) cause;
        }
        final RecipeExceptionDetails recipeDetails = getRecipeDetails(errorCode, context);
        String message = errorCode;
        try {
        	final ErrorMap errorMap = recipeDetails.getErrorMap();
            for(final RecipeExceptionDetails.ErrorMap.Entry entry: errorMap.entries) {
            	if(entry.key.equals("en")) {
            		message = entry.value.getMessage();
            		break;
            	}
            }
        } catch (final RuntimeException ex) {

        } finally {
            LOG.error("Error: " + message);
        }
        return new RecipeException(message, recipeDetails);
    }

    /**
     * Gets the recipe details.
     *
     * @param errorCode the error code
     * @param context the context
     * @return the recipe details
     */
    private static RecipeExceptionDetails getRecipeDetails(final String errorCode, final Object[] context) {
        final ErrorMap errorMap = new ErrorMap();
        for (final Locale loc : LOCALES) {
        	final RecipeExceptionDetails.ErrorMap.Entry entry = new RecipeExceptionDetails.ErrorMap.Entry();
        	entry.setKey(loc.getLanguage());
        	final be.recipe.services.core.RecipeError re = new be.recipe.services.core.RecipeError();
        	re.setMessage(getI18nMessage(errorCode, loc, context).getMessage());
        	entry.setValue(re);
        	errorMap.entries.add(entry);
        }
        final RecipeExceptionDetails red=  new RecipeExceptionDetails();
        red.setCode(errorCode);
        red.setErrorMap(errorMap);
        return red;
    }

    /**
     * Gets the i18n message.
     *
     * @param errorCode the error code
     * @param locale the locale
     * @param context the context
     * @return the i18n message
     */
    private static RecipeError getI18nMessage(final String errorCode, Locale locale, final Object[] context) {

        if (locale == null) {
            // Set default locale to English if no locale is specified;
            locale = ENGLISH_LOCALE;
        }
        Locale.setDefault(locale);

        try {
            final ResourceBundle rb = ResourceBundle.getBundle(RESOURCE_BUNDLE, locale, new UTF8Control());
            if (rb != null && rb.containsKey(errorCode)) {
                final String msg = rb.getString(errorCode);
                return new RecipeError(MessageFormat.format(msg, context));
            }
        } catch (final MissingResourceException e) {
            LOG.error(e.getMessage(), e);
        }

        return new RecipeError("!!! " + errorCode + " !!!");
    }
}