/*
 * (C) 2021 Recip-e. All rights reserved.
 */
package be.business.connector.recipe.utils.prescribercreatebulkprescriptions;

import be.business.connector.common.ehealth.EhealthKeyRegistry;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.utils.ETKHelper;
import be.business.connector.core.utils.EncryptionUtils;
import be.business.connector.core.utils.PropertyHandler;
import be.business.connector.core.utils.SessionValidator;
import be.business.connector.recipe.prescriber.AbstractPrescriberIntegrationModule;
import be.business.connector.recipe.prescriber.dto.CreatePrescriptionDTO;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.service.kgss.domain.KeyResult;
import be.ehealth.technicalconnector.service.sts.security.impl.KeyPairCredential;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.session.SessionItem;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionToken;
import be.recipe.api.crypto.Message;
import org.perf4j.aop.Profiled;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import java.security.Key;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Semaphore;

public class PrescriberCreateBulkPrescriptionsUseCase {

  private static final Logger LOG = LoggerFactory.getLogger(PrescriberCreateBulkPrescriptionsUseCase.class);
  private final ETKHelper etkHelper;
  private final Map<String, KeyResult> keyCache;
  private final EhealthKeyRegistry keyRegistry;
  private final Key symmKey;

  public PrescriberCreateBulkPrescriptionsUseCase(
      final PropertyHandler propertyHandler,
      final EncryptionUtils encryptionUtils,
      final Map<String, KeyResult> keyCache) {
    this.etkHelper = new ETKHelper(propertyHandler, encryptionUtils);
    this.symmKey = EncryptionUtils.getInstance().generateSecretKey();
    this.keyCache = keyCache;
    this.keyRegistry = new EhealthKeyRegistry(propertyHandler);
    this.keyRegistry.refresh(etkHelper, propertyHandler);
  }

  public List<CreatePrescriptionDTO> execute(
      final List<CreatePrescriptionDTO> createPrescriptionDTOs,
      final AbstractPrescriberIntegrationModule prescriberIntegrationModuleV4) throws TechnicalConnectorException {
    final long startCreating = System.currentTimeMillis();
    final int threadLimit =
        PropertyHandler.getInstance().getIntegerProperty("encryption.thread.number", "50");
    final Semaphore semaphore = new Semaphore(threadLimit, true);
    final List<CreateEncryptedPrescriptionsThread> dataList = new ArrayList<>();
    final SessionItem sessionItem = Session.getInstance().getSession();
    SessionValidator.assertValidSession(sessionItem);
    final Element assertion = sessionItem.getSAMLToken().getAssertion();
    final KeyPairCredential keyPairCredential = getKeyPairCredentialFromSession(sessionItem);
    for (final CreatePrescriptionDTO createPrescriptionDTO : createPrescriptionDTOs) {
      if (createPrescriptionDTO.getPrescription() != null) {
        final KeyResult key =
            getNewKey(createPrescriptionDTO.getPatientId());
        final EncryptionToken recipeEtk = etkHelper.getRecipe_ETK().get(0);
        final CreateEncryptedPrescriptionsTask task = new CreateEncryptedPrescriptionsTask(
                semaphore,
                createPrescriptionDTO,
                key,
                symmKey,
                recipeEtk,
                prescriberIntegrationModuleV4,
                assertion,
                keyPairCredential);

        CreateEncryptedPrescriptionsThread encryptorThread = new CreateEncryptedPrescriptionsJavaThread(task);
        dataList.add(encryptorThread);
      }
    }

    for (final CreateEncryptedPrescriptionsThread task : dataList) {
      semaphore.acquireUninterruptibly();
      task.execute();
    }
    try {
      semaphore.acquireUninterruptibly(threadLimit);
    } catch (final IllegalArgumentException e) {
      LOG.debug("Incorrect Thread configuration : " + e);
      throw new IntegrationModuleException(e.getMessage(), e);
    }
    final List<CreatePrescriptionDTO> finalResult = new ArrayList<>();
    for (final CreateEncryptedPrescriptionsThread task : dataList) {
      try {
        LOG.info("Ending thread {} at {}", task.getName(), System.currentTimeMillis());
        task.join();
      } catch (InterruptedException e) {
        throw new IntegrationModuleException(e);
	  }
      finalResult.add(task.getData());
    }

    for (final CreatePrescriptionDTO createPrescriptionDTO : createPrescriptionDTOs) {
      for (final CreatePrescriptionDTO finalDTO : finalResult) {
        if (createPrescriptionDTO.getSequenceNumber() == finalDTO.getSequenceNumber()) {
          createPrescriptionDTO.setRid(finalDTO.getRid());
          createPrescriptionDTO.setErrorOccured(finalDTO.isErrorOccured());
          createPrescriptionDTO.setException(finalDTO.getException());
          if (finalDTO.isErrorOccured()) {
            LOG.info(
                String.format(
                    "Error occured during doEncryptions-operations[%s]:[%s] ",
                    finalDTO.getSequenceNumber(), finalDTO.getException()));
          }
        }
      }
    }

    final long stopCreating = System.currentTimeMillis();
    LOG.debug("Creating {} prescriptions in bulk took {} ms", createPrescriptionDTOs.size(), stopCreating - startCreating);

    return createPrescriptionDTOs;
  }

  private static KeyPairCredential getKeyPairCredentialFromSession(final SessionItem sessionItem) {
    try {
      return new KeyPairCredential(
              sessionItem.getHolderOfKeyCredential().getPrivateKey(),
              sessionItem.getHolderOfKeyCredential().getCertificate());
    } catch (TechnicalConnectorException e) {
      LOG.error("Error generic webservice", e);
      throw new IntegrationModuleException(e);
    }
  }

  private KeyResult getNewKey(final String patientId) {
    KeyResult key;
    if (keyCache.containsKey(patientId)) {
      key = keyCache.get(patientId);
    } else {
      key =
          getNewKeyFromKgss(patientId);
      keyCache.put(patientId, key);
    }
    return key;
  }

  @Profiled(logFailuresSeparately = true, tag = "0.PrescriberIntegrationModule#getNewKeyFromKgss")
  protected KeyResult getNewKeyFromKgss(final String patientId) {
    Message.Cipher.Key key =
        keyRegistry.create(patientId);
    return new KeyResult(key.secret(), key.id().toString());
  }
}
