package be.business.connector.recipe.utils.executorlistopenprescriptions;

import be.business.connector.common.StandaloneRequestorProvider;
import be.business.connector.core.domain.KgssIdentifierType;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.technical.connector.utils.Crypto;
import be.business.connector.core.utils.ETKHelper;
import be.business.connector.core.utils.PropertyHandler;
import be.business.connector.recipe.executor.dto.ListOpenPrescriptionsResultWithExceptions;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.service.kgss.impl.KgssServiceImpl;
import be.ehealth.technicalconnector.service.sts.security.impl.KeyPairCredential;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.session.SessionItem;
import be.fgov.ehealth.etee.kgss._1_0.protocol.GetKeyRequestContent;
import be.recipe.services.executor.GetOpenPrescriptionForExecutor;
import be.recipe.services.executor.ListOpenPrescriptionsResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import java.security.PrivateKey;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Semaphore;

import static org.apache.commons.collections.CollectionUtils.*;

public class MultiThreadedExecutorListOpenPrescriptionsUseCase {

	private final ETKHelper etkHelper;
	private static final Logger LOG = LoggerFactory.getLogger(MultiThreadedExecutorListOpenPrescriptionsUseCase.class);

	public MultiThreadedExecutorListOpenPrescriptionsUseCase(ETKHelper etkHelper) {
		this.etkHelper = etkHelper;
	}

	public ListOpenPrescriptionsResult execute(final ListOpenPrescriptionsResult listOpenPrescriptionsResult) throws IntegrationModuleException {
		if (listOpenPrescriptionsResult == null || isEmpty(listOpenPrescriptionsResult.getPrescriptions())) {
			return listOpenPrescriptionsResult;
		}
		return doDecryptions(listOpenPrescriptionsResult);
	}

	public ListOpenPrescriptionsResultWithExceptions executeWithExceptions(
			final ListOpenPrescriptionsResult listOpenPrescriptionsResult)
			throws IntegrationModuleException {
		if (listOpenPrescriptionsResult == null || isEmpty(listOpenPrescriptionsResult.getPrescriptions())) {
			return new ListOpenPrescriptionsResultWithExceptions();
		}
		final Map<GetOpenPrescriptionForExecutor, DecryptPrescriptionThread> dataMap =
				createThreads(listOpenPrescriptionsResult);
		final ListOpenPrescriptionsResultWithExceptions result =
				new ListOpenPrescriptionsResultWithExceptions();
		for (DecryptPrescriptionThread thread : dataMap.values()) {
			result.addPrescription(thread.getPrescription(), thread.getError());
		}
		result.setHasMoreResults(listOpenPrescriptionsResult.isHasMoreResults());
		result.setSession(listOpenPrescriptionsResult.getSession());
		return result;
	}

	private Map<GetOpenPrescriptionForExecutor, DecryptPrescriptionThread> createThreads(
			ListOpenPrescriptionsResult listOpenPrescriptionsResult) {
		final Map<GetOpenPrescriptionForExecutor, DecryptPrescriptionThread> dataMap = new HashMap<>();

		final int threadLimit = PropertyHandler.getInstance().getIntegerProperty("decryption.thread.number", "50");
		final Semaphore semaphore = new Semaphore(threadLimit, true);

		final byte[] pharmacyEtk = getPharmacyEtkAsBytes();
		final byte[] kgssEtk = etkHelper.getKGSS_ETK().get(0).getEncoded();

		final KgssServiceImpl kgssService = new KgssServiceImpl();
		final GetKeyRequestContent getKeyRequestContent = new GetKeyRequestContent();
		getKeyRequestContent.setETK(pharmacyEtk);
		final SessionItem sessionItem = Session.getInstance().getSession();
		KeyPairCredential encryptionCredential;
		KeyPairCredential holderOfKeyCredential;
		try {
			// Wrap in KeyPairCredential to improve performance (not doing this causes SAMLHolderOfKeyHandler to be slow when executing the requests
			encryptionCredential = new KeyPairCredential(
					sessionItem.getEncryptionCredential().getPrivateKey(),
					sessionItem.getEncryptionCredential().getCertificate());
			holderOfKeyCredential = new KeyPairCredential(
					sessionItem.getHolderOfKeyCredential().getPrivateKey(),
					sessionItem.getHolderOfKeyCredential().getCertificate());
		} catch (TechnicalConnectorException e) {
            throw new IntegrationModuleException(e);
		}
		final Element samlAssertion = sessionItem.getSAMLToken().getAssertion();
		final Map<String, PrivateKey> encryptionPrivateKeys = sessionItem.getEncryptionPrivateKeys();
		final Crypto crypto = new Crypto();
		kgssService.bootstrap();

		for (final GetOpenPrescriptionForExecutor getOpenPrescription : listOpenPrescriptionsResult.getPrescriptions()) {
			if (getOpenPrescription.getPrescription() != null) {
				final DecryptPrescriptionTask task = new DecryptPrescriptionTask(getOpenPrescription, semaphore, pharmacyEtk, kgssEtk, encryptionCredential, holderOfKeyCredential, samlAssertion, encryptionPrivateKeys, kgssService, crypto);
				dataMap.put(getOpenPrescription, new DecryptPrescriptionJavaThread(task));
			}
		}

		for (final DecryptPrescriptionThread decryptPrescriptionThread : dataMap.values()) {
			semaphore.acquireUninterruptibly();
			decryptPrescriptionThread.execute();
		}
		try {
			semaphore.acquireUninterruptibly(threadLimit);
		} catch (final IllegalArgumentException e) {
			LOG.debug("Incorrect Thread configuration : " + e);
		}
		return dataMap;
	}

	private byte[] getPharmacyEtkAsBytes() {
		return etkHelper
				.getEtks(KgssIdentifierType.NIHII_PHARMACY, StandaloneRequestorProvider.getRequestorIdInformation())
				.get(0)
				.getEncoded();
	}

	private ListOpenPrescriptionsResult doDecryptions(
			final ListOpenPrescriptionsResult prescriptionList) {
		final Map<GetOpenPrescriptionForExecutor, DecryptPrescriptionThread> dataMap = createThreads(prescriptionList);
		final ListOpenPrescriptionsResult finalResult = new ListOpenPrescriptionsResult();
		for (DecryptPrescriptionThread thread : dataMap.values()) {
			finalResult.getPrescriptions().add(thread.getPrescription());
		}
		finalResult.setHasMoreResults(prescriptionList.isHasMoreResults());
		finalResult.setSession(prescriptionList.getSession());

		Collections.sort(
				finalResult.getPrescriptions(),
				new Comparator<GetOpenPrescriptionForExecutor>() {
					@Override
					public int compare(GetOpenPrescriptionForExecutor f1, GetOpenPrescriptionForExecutor f2) {
						return f2.getCreationDate().compareTo(f1.getCreationDate());
					}
				});

		return finalResult;
	}

}
