/*
 * (C) 2021 Recip-e. All rights reserved.
 */
package be.business.connector.recipe.utils;

import be.business.connector.common.StandaloneRequestorProvider;
import be.business.connector.core.domain.KgssIdentifierType;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.utils.ETKHelper;
import be.business.connector.core.utils.EncryptionUtils;
import be.business.connector.core.utils.PropertyHandler;
import be.business.connector.recipe.executor.dto.ListOpenPrescriptionsResultWithExceptions;
import be.business.connector.recipe.executor.dto.ListReservationsResultWithExceptions;
import be.business.connector.recipe.utils.executorlistopenprescriptions.MultiThreadedExecutorListOpenPrescriptionsUseCase;
import be.business.connector.recipe.utils.executorlistopenprescriptions.SingleThreadedExecutorListOpenPrescriptionsUseCase;
import be.recipe.services.executor.ListOpenPrescriptionsResult;
import be.recipe.services.executor.ListReservationsResult;
import be.recipe.services.executor.ListReservationsResultItem;
import org.apache.commons.collections.CollectionUtils;
import org.perf4j.aop.Profiled;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.Semaphore;

/**
 * The Class DecryptionUtils.
 *
 * @author <a href="mailto:bruno.casneuf@recip-e.be">Bruno Casneuf</a>
 */
public class ExecutorDecryptionUtils {

  /** The Constant LOG. */
  private static final Logger LOG = LoggerFactory.getLogger(ExecutorDecryptionUtils.class);

  /** The etk helper. */
  private final ETKHelper etkHelper;


  /**
   * Instantiates a new {@link ExecutorDecryptionUtils}.
   *
   * @param propertyHandler the property handler
   * @param encryptionUtils the encryption utils
   */
  public ExecutorDecryptionUtils(
      final PropertyHandler propertyHandler, final EncryptionUtils encryptionUtils) {
    this.etkHelper = new ETKHelper(propertyHandler, encryptionUtils);
  }

  /**
   * Decrypt a {@link ListOpenPrescriptionsResult}.
   *
   * @param listOpenPrescriptionsResult the encrypted {@link ListOpenPrescriptionsResult}
   * @return the decrypted {@link ListOpenPrescriptionsResult}
   * @throws IntegrationModuleException the integration module exception
   */
  public ListOpenPrescriptionsResult decryptPrescriptions(
      final ListOpenPrescriptionsResult listOpenPrescriptionsResult) throws IntegrationModuleException {
    return new MultiThreadedExecutorListOpenPrescriptionsUseCase(etkHelper).execute(listOpenPrescriptionsResult);
  }

  public ListOpenPrescriptionsResult decryptPrescriptionsSingleThreaded(
          final ListOpenPrescriptionsResult listOpenPrescriptionsResult) throws IntegrationModuleException {
    return new SingleThreadedExecutorListOpenPrescriptionsUseCase(etkHelper).execute(listOpenPrescriptionsResult);
  }

  /**
   * Decrypt a {@link ListReservationsResult}.
   *
   * @param listReservationsResult the encrypted {@link ListReservationsResult}
   * @return the decrypted {@link ListReservationsResult}
   * @throws IntegrationModuleException the integration module exception
   */
  @Profiled(logFailuresSeparately = true, tag = "0.DecompressionUtil#decryptReservations")
  public ListReservationsResult decryptReservations(
      final ListReservationsResult listReservationsResult) throws IntegrationModuleException {
    if (listReservationsResult == null
        || listReservationsResult != null
            && CollectionUtils.isEmpty(listReservationsResult.getItems())) {
      return listReservationsResult;
    }
    return doDecryptions(listReservationsResult);
  }

  @Profiled(
      logFailuresSeparately = true,
      tag = "0.DecompressionUtil#decryptPrescriptionsWithException")
  public ListOpenPrescriptionsResultWithExceptions decryptPrescriptionsWithException(
      final ListOpenPrescriptionsResult listOpenPrescriptionsResult)
      throws IntegrationModuleException {

    return new MultiThreadedExecutorListOpenPrescriptionsUseCase(etkHelper)
            .executeWithExceptions(listOpenPrescriptionsResult);
  }

  @Profiled(
      logFailuresSeparately = true,
      tag = "0.DecompressionUtil#decryptReservationsWithExceptions")
  public ListReservationsResultWithExceptions decryptReservationsWithExceptions(
      final ListReservationsResult listReservationsResult) throws IntegrationModuleException {
    if (listReservationsResult == null
        || listReservationsResult != null
            && CollectionUtils.isEmpty(listReservationsResult.getItems())) {
      return new ListReservationsResultWithExceptions();
    }
    Map<ListReservationsResultItem, ReservationDecryptorThread> dataMap =
        createThreads(listReservationsResult);
    final ListReservationsResultWithExceptions finalResult =
        new ListReservationsResultWithExceptions();
    for (final ReservationDecryptorThread decryptorThread : dataMap.values()) {
      finalResult.addItem(
          decryptorThread.getListReservationsResultItem(), decryptorThread.getError());
    }
    finalResult.setHasMoreResults(listReservationsResult.isHasMoreResults());
    finalResult.setSession(listReservationsResult.getSession());
    return finalResult;
  }

  /**
   * Decrypt a {@link ListReservationsResult}.
   *
   * @param listReservationsResult the encrypted {@link ListReservationsResult}
   * @return the decrypted {@link ListReservationsResult}
   */
  private ListReservationsResult doDecryptions(
      final ListReservationsResult listReservationsResult) {
    Map<ListReservationsResultItem, ReservationDecryptorThread> dataMap =
        createThreads(listReservationsResult);

    final ListReservationsResult finalResult = new ListReservationsResult();
    for (ReservationDecryptorThread thread : dataMap.values())
      finalResult.getItems().add(thread.getListReservationsResultItem());
    finalResult.setHasMoreResults(listReservationsResult.isHasMoreResults());
    finalResult.setSession(listReservationsResult.getSession());

    Collections.sort(
        finalResult.getItems(),
        new Comparator<ListReservationsResultItem>() {
          @Override
          public int compare(ListReservationsResultItem f1, ListReservationsResultItem f2) {
            return f2.getCreationDate().compareTo(f1.getCreationDate());
          }
        });

    return finalResult;
  }

  private Map<ListReservationsResultItem, ReservationDecryptorThread> createThreads(
      ListReservationsResult listReservationsResult) {
    final int threadLimit =
        PropertyHandler.getInstance().getIntegerProperty("decryption.thread.number", "50");
    final Semaphore semaphore = new Semaphore(threadLimit, true);

    final Map<ListReservationsResultItem, ReservationDecryptorThread> dataMap = new HashMap<>();
    final String requestorIdInformation = StandaloneRequestorProvider.getRequestorIdInformation();
    final byte[] pharmacyEtk =
        etkHelper
            .getEtks(KgssIdentifierType.NIHII_PHARMACY, requestorIdInformation)
            .get(0)
            .getEncoded();
    final byte[] kgssEtk = etkHelper.getKGSS_ETK().get(0).getEncoded();
    for (final ListReservationsResultItem listReservationResultItem :
        listReservationsResult.getItems()) {
      if (listReservationResultItem.getPrescription() != null
          && listReservationResultItem.getPrescription().getPrescription() != null) {
        final ReservationDecryptorThread decryptorThread =
            new ReservationDecryptorThread(
                semaphore, listReservationResultItem, pharmacyEtk, kgssEtk);
        dataMap.put(listReservationResultItem, decryptorThread);
      }
    }

    for (final ReservationDecryptorThread decryptorThread : dataMap.values()) {
      semaphore.acquireUninterruptibly();
      decryptorThread.start();
    }
    try {
      semaphore.acquireUninterruptibly(threadLimit);
    } catch (final IllegalArgumentException e) {
      LOG.debug("Incorrect Thread configuration : " + e);
    }
    return dataMap;
  }
}
