package be.business.connector.recipe.prescriber;

import static be.recipe.api.Prescription.Type.prescriptionType;
import static be.recipe.api.Prescription.expirationdate_already_passed;
import static be.recipe.api.Prescription.expirationdate_too_far_in_future;
import static be.recipe.api.prescriber.PrescriberType.DOCTOR;
import static be.recipe.api.prescriber.PrescriberType.HOSPITAL;

import be.business.connector.common.StandaloneRequestorProvider;
import be.business.connector.common.ehealth.EhealthCipher;
import be.business.connector.common.ehealth.EhealthKeyRegistry;
import be.business.connector.common.module.AbstractIntegrationModule;
import be.business.connector.core.domain.KgssIdentifierType;
import be.business.connector.core.ehealth.services.KgssService;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.exceptions.IntegrationModuleValidationException;
import be.business.connector.core.technical.connector.utils.Crypto;
import be.business.connector.core.utils.*;
import be.business.connector.recipe.RoutingPrescriptionService;
import be.business.connector.recipe.prescriber.dto.CreatePrescriptionDTO;
import be.business.connector.recipe.utils.KmehrHelper;
import be.ehealth.technicalconnector.service.kgss.domain.KeyResult;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.session.SessionItem;
import be.fgov.ehealth.commons.core.v1.IdentifierType;
import be.fgov.ehealth.commons.core.v1.LocalisedString;
import be.fgov.ehealth.commons.core.v1.StatusType;
import be.fgov.ehealth.commons.protocol.v1.ResponseType;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionToken;
import be.fgov.ehealth.recipe.core.v4.SecuredContentType;
import be.recipe.api.CipheringPrescriptionService;
import be.recipe.api.PrefetchingPrescriptionService;
import be.recipe.api.Prescription;
import be.recipe.api.PrescriptionContent;
import be.recipe.api.crypto.Message;
import be.recipe.api.executor.Executor;
import be.recipe.api.patient.Patient;
import be.recipe.api.prescriber.*;
import be.recipe.api.prescriber.ListPrescriptions;
import be.recipe.common.JAXBTimestampingContext;
import be.recipe.services.prescriber.*;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java8.util.Optional;
import java8.util.function.Consumer;
import javax.xml.datatype.DatatypeConfigurationException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Validate;
import org.perf4j.aop.Profiled;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

/** The Class AbstractPrescriberIntegrationModule. */
public abstract class AbstractPrescriberIntegrationModule extends AbstractIntegrationModule
    implements ListPrescriptions.Command<ListPrescriptions, ListPrescriptions.Response.PlainText>,
        PutVisionOtherPrescribers.Command<PutVisionOtherPrescribers>,
        GetPrescription.Command<GetPrescription, Prescription.PlainText> {

  /** The Constant LOG. */
  private static final Logger LOG =
      LoggerFactory.getLogger(AbstractPrescriberIntegrationModule.class);

  /** The prescription cache. */
  private final Map<String, String> prescriptionCache = new HashMap<>();

  /** The kmehr helper. */
  private final KmehrHelper kmehrHelper = new KmehrHelper();

  public final PrescriptionService.Ciphering.Simplified target;

  /** The key cache. */
  protected Map<String, KeyResult> keyCache = new HashMap<>();

  /** The kgss service. */
  protected KgssService kgssService =
      be.business.connector.core.ehealth.services.KgssServiceImpl.getInstance();

  private Crypto crypto;
  private EhealthCipher cipher;
  private EhealthKeyRegistry keyRegistry;
  protected PrescriptionContent.Factory prescriptionContentFactory;

  /**
   * Instantiates a new abstract prescriber integration module. @ the integration module exception
   */
  public AbstractPrescriberIntegrationModule() {
    super();
    try {
      PropertyHandler props = PropertyHandler.getInstance();
      cipher = new EhealthCipher();
      keyRegistry = new EhealthKeyRegistry();
      crypto = new Crypto();
      keyRegistry.refresh(props);
      prescriptionContentFactory =
          new PrescriptionContent.Factory(
              new Message.Factory(cipher, new JAXBTimestampingContext()), keyRegistry);
      target =
          new PrefetchingPrescriptionService(
              new CipheringPrescriptionService(
                  new RoutingPrescriptionService(new RecipePrescriberClient(props), null, null),
                      prescriptionContentFactory));

      LOG.info("*************** Prescriber System module init correctly *******************");
    } catch (DatatypeConfigurationException e) {
      throw new RuntimeException(e);
    }
  }

  /**
   * Check status.
   *
   * @param response the response @ the integration module exception
   */
  public void checkStatus(final ResponseType response) {
    if (!EHEALTH_SUCCESS_CODE_100.equals(response.getStatus().getCode())
        && !EHEALTH_SUCCESS_CODE_200.equals(response.getStatus().getCode())) {
      LOG.error("Error Status received : " + response.getStatus().getCode());
      throw new IntegrationModuleException(getLocalisedMsg(response.getStatus()));
    }
  }

  /**
   * Check status.
   *
   * @param response the response @ the integration module exception
   */
  public void checkStatus(final be.recipe.services.core.ResponseType response) {
    if (!EHEALTH_SUCCESS_CODE_100.equals(response.getStatus().getCode())
        && !EHEALTH_SUCCESS_CODE_200.equals(response.getStatus().getCode())) {
      LOG.error("Error Status received : " + response.getStatus().getCode());
      Optional.ofNullable(response.getStatus().getMessageCode())
          .ifPresent(
              new Consumer<String>() {
                @Override
                public void accept(String code) {
                  if (code.equals(expirationdate_too_far_in_future))
                    throw validationException(
                        I18nHelper.getLabel(
                            code,
                            new Object[] {
                              extractLastValidExpirationDate(response.getStatus().getMessages())
                            }));
                  if (code.equals(expirationdate_already_passed))
                    throw validationException(I18nHelper.getLabel(code));
                }
              });
      throw new IntegrationModuleException(getLocalisedMsg(response.getStatus()), response);
    }
  }

  private static String extractLastValidExpirationDate(
      List<be.recipe.services.core.LocalisedString> messages) {
    for (be.recipe.services.core.LocalisedString msg : messages)
      if (msg.getLang().value().equals("EN")) return msg.getValue().substring(51, 61);
    throw new IllegalStateException(
        "Expected to be able to extract the last valid expiration date from the english message!");
  }

  private static IntegrationModuleValidationException validationException(String label) {
    return new IntegrationModuleValidationException(label, label);
  }

  /**
   * Gets the localised msg.
   *
   * @param status the status
   * @return the localised msg
   */
  private String getLocalisedMsg(final StatusType status) {
    final String locale = IntegrationModuleException.getUserLocale();
    for (final LocalisedString msg : status.getMessages()) {
      if (msg.getLang() != null && locale.equalsIgnoreCase(msg.getLang().value())) {
        return msg.getValue();
      }
    }
    if (!status.getMessages().isEmpty()) {
      return status.getMessages().get(0).getValue();
    }
    return status.getCode();
  }

  /**
   * Gets the localised msg.
   *
   * @param status the status
   * @return the localised msg
   */
  private String getLocalisedMsg(final be.recipe.services.core.StatusType status) {
    final String locale = IntegrationModuleException.getUserLocale();
    for (final be.recipe.services.core.LocalisedString msg : status.getMessages()) {
      if (msg.getLang() != null && locale.equalsIgnoreCase(msg.getLang().value())) {
        return msg.getValue();
      }
    }
    if (!status.getMessages().isEmpty()) {
      return status.getMessages().get(0).getValue();
    }
    return status.getCode();
  }

  /**
   * Creates the identifier type.
   *
   * @param id the id
   * @param type the type
   * @return the identifier type
   */
  protected IdentifierType createIdentifierType(final String id, final String type) {
    final IdentifierType ident = new IdentifierType();
    ident.setId(id);
    ident.setType(type);
    return ident;
  }

  /**
   * Sets the personal password.
   *
   * @param personalPassword the new personal password @ the integration module exception
   */
  public void setPersonalPassword(final String personalPassword) {
    final SessionItem sessionItem = Session.getInstance().getSession();
    SessionValidator.assertValidSession(sessionItem);

    try {
      final String niss = STSHelper.getNiss(sessionItem.getSAMLToken().getAssertion());
      final String nihii = STSHelper.getNihii(sessionItem.getSAMLToken().getAssertion());

      final EncryptionUtils encryptionUtils = EncryptionUtils.getInstance();
      encryptionUtils.unlockPersonalKey(
          StringUtils.isNotBlank(niss) ? niss : nihii, personalPassword);
      dataUnsealer = encryptionUtils.initUnsealing();
      final List<EncryptionToken> tokens =
          getEtkHelper()
              .getEtks(
                  getIdentifierType(), StandaloneRequestorProvider.getRequestorIdInformation());
      encryptionUtils.verifyDecryption(tokens.get(0));
    } catch (final Exception e) {
      throw new IntegrationModuleException(e);
    }
  }

  private KgssIdentifierType getIdentifierType() {
    final String type =
        STSHelper.getType(Session.getInstance().getSession().getSAMLToken().getAssertion());
    return type.equals("HOSPITAL") ? KgssIdentifierType.NIHII_HOSPITAL : KgssIdentifierType.NIHII;
  }

  /**
   * Gets the kmehr helper.
   *
   * @return the kmehr helper
   */
  protected KmehrHelper getKmehrHelper() {
    return kmehrHelper;
  }

  /**
   * Gets the sealed data.
   *
   * @param validationPropertiesParam the validation properties param
   * @return the sealed data @ the integration module exception
   */
  protected byte[] getSealedData(final ValidationPropertiesParam validationPropertiesParam) {
    return sealForRecipe(validationPropertiesParam, ValidationPropertiesParam.class);
  }

  /**
   * Seal for recipe.
   *
   * @param <T> the generic type
   * @param data the data
   * @param type the type
   * @return the byte[] @ the integration module exception
   */
  private <T> byte[] sealForRecipe(final T data, final Class<T> type) {
    final MarshallerHelper<Object, T> helper = new MarshallerHelper<>(Object.class, type);
    final EncryptionToken etkRecipe = getEtkHelper().getRecipe_ETK().get(0);
    return sealRequest(etkRecipe, helper.toXMLByteArray(data));
  }

  /**
   * Gets the new key.
   *
   * @param patientId the patient id
   * @return the new key @ the integration module exception
   */
  public KeyResult getNewKey(final String patientId) {
    if (keyCache.containsKey(patientId)) {
      return keyCache.get(patientId);
    } else {
      KeyResult key = getNewKeyFromKgss(patientId);
      keyCache.put(patientId, key);
      return key;
    }
  }

  @Profiled(logFailuresSeparately = true, tag = "0.PrescriberIntegrationModule#getNewKeyFromKgss")
  protected KeyResult getNewKeyFromKgss(final String patientId) {
    Message.Cipher.Key key = keyRegistry.create(patientId);
    return new KeyResult(key.secret(), key.id().toString());
  }

  /**
   * Seal prescription for unknown.
   *
   * @param key the key
   * @param messageToProtect the message to protect
   * @return the byte[] @ the integration module exception
   */
  @Profiled(
      logFailuresSeparately = true,
      tag = "0.PrescriberIntegrationModule#sealPrescriptionForUnknown")
  public byte[] sealPrescriptionForUnknown(final KeyResult key, final byte[] messageToProtect) {
    return crypto.seal(messageToProtect, key.getSecretKey(), key.getKeyId());
  }

  /**
   * Unseal feedback.
   *
   * @param message the message
   * @return the byte[] @ the integration module exception
   */
  @Profiled(logFailuresSeparately = true, tag = "0.PrescriberIntegrationModule#unsealFeedback")
  protected byte[] unsealFeedback(final byte[] message) {
    return unsealNotiffeed(message);
  }

  /**
   * Seal notification.
   *
   * @param paramEncryptionToken the param encryption token
   * @param paramArrayOfByte the param array of byte
   * @return the byte[] @ the integration module exception
   */
  @Profiled(logFailuresSeparately = true, tag = "0.PrescriberIntegrationModule#sealNotification")
  protected byte[] sealNotification(
      final EncryptionToken paramEncryptionToken, final byte[] paramArrayOfByte) {
    return crypto.seal(paramEncryptionToken, paramArrayOfByte);
  }

  /**
   * Marshall.
   *
   * @param <T> the generic type
   * @param data the data
   * @param type the type
   * @return the byte[] @ the integration module exception
   */
  private <T> byte[] marshall(final T data, final Class<T> type) {
    final MarshallerHelper<Object, T> helper = new MarshallerHelper<>(Object.class, type);
    return helper.toXMLByteArray(data);
  }

  /**
   * Gets the sealed data.
   *
   * @param listRidHistoryParam the request
   * @return the sealed data @ the integration module exception
   */
  protected byte[] getSealedData(final ListRidsHistoryParam listRidHistoryParam) {
    return sealForRecipe(listRidHistoryParam, ListRidsHistoryParam.class);
  }

  /**
   * Gets the sealed data.
   *
   * @param request the request
   * @return the sealed data @ the integration module exception
   */
  protected byte[] getSealedData(final GetPrescriptionStatusParam request) {
    return sealForRecipe(request, GetPrescriptionStatusParam.class);
  }

  /**
   * Gets the sealed data.
   *
   * @param putVisionParam the request
   * @return the sealed data @ the integration module exception
   */
  protected byte[] getSealedData(final PutVisionParam putVisionParam) {
    return sealForRecipe(putVisionParam, PutVisionParam.class);
  }

  /**
   * Gets the sealed data.
   *
   * @param listOpenRidsParam the request
   * @return the sealed data @ the integration module exception
   */
  protected byte[] getSealedData(final ListOpenRidsParam listOpenRidsParam) {
    return sealForRecipe(listOpenRidsParam, ListOpenRidsParam.class);
  }

  /**
   * @param dtos the dtos
   */
  protected void validateCreatePrescriptionDTOs(final List<CreatePrescriptionDTO> dtos) {
    try {
      Validate.notNull(dtos);
      Validate.notEmpty(dtos);
    } catch (Exception e) {
      throw new IntegrationModuleException(
          I18nHelper.getLabel("error.validation.list.empty.or.null"));
    }
    if (dtos.size() > 30) {
      throw new IntegrationModuleException(I18nHelper.getLabel("error.validation.too.many.items"));
    }
    final List<Integer> seqNbrs = new ArrayList<Integer>();
    for (final CreatePrescriptionDTO dto : dtos) {
      if (seqNbrs.contains(dto.getSequenceNumber())) {
        throw new IntegrationModuleException(
            I18nHelper.getLabel("error.validation.duplicate.sequencenumbers"));
      }
      seqNbrs.add(dto.getSequenceNumber());
    }
  }

  /**
   * Perform validation.
   *
   * @param prescription the prescription
   * @param prescriptionType the prescription type
   * @param expirationDateFromRequest the expiration date from request @ the integration module
   *     exception
   */
  @Profiled(
      logFailuresSeparately = true,
      tag = "0.PrescriberIntegrationModuleV4Impl#validateKmehr",
      logger = "org.perf4j.TimingLogger_Common")
  public void validateKmehr(
      final byte[] prescription,
      final String prescriptionType,
      final String expirationDateFromRequest) {
    final List<String> errors = new ArrayList<>();
    try {
      getKmehrHelper().assertValidKmehrPrescription(prescription, prescriptionType);
    } catch (final IntegrationModuleValidationException e) {
      errors.addAll(e.getValidationErrors());
    }
    validateExpirationDateFromKmehr(prescription, errors, expirationDateFromRequest);
    if (CollectionUtils.isNotEmpty(errors)) {
      LOG.info("******************************************************");
      for (final String error : errors) {
        LOG.info("Errors found in the kmehr:" + error);
      }
      LOG.info("******************************************************");
      throw new IntegrationModuleValidationException(
          I18nHelper.getLabel("error.xml.invalid"), errors);
    }
  }

  /**
   * Validate expiration date from kmehr.
   *
   * @param xmlDocument the xml document
   * @param errors the errors
   * @param expirationDateFromRequest the expiration date from request @ the integration module
   *     exception
   */
  private void validateExpirationDateFromKmehr(
      final byte[] xmlDocument, final List<String> errors, final String expirationDateFromRequest) {
    try {
      final DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
      factory.setNamespaceAware(false);
      final DocumentBuilder builder = factory.newDocumentBuilder();
      final Document kmehrDocument = builder.parse(new ByteArrayInputStream(xmlDocument));
      final PropertyHandler propertyHandler = PropertyHandler.getInstance();
      final XPath xpath = XPathFactory.newInstance().newXPath();
      final String xpathStr = propertyHandler.getProperty("expirationdate.xpath");
      final NodeList expirationDateNodeList =
          (NodeList) xpath.evaluate(xpathStr, kmehrDocument, XPathConstants.NODESET);
      if (expirationDateNodeList.item(0) != null) {
        final String expirationDateFromKmehr = expirationDateNodeList.item(0).getTextContent();
        if (!expirationDateFromKmehr.contentEquals(expirationDateFromRequest)) {
          errors.add(
              I18nHelper.getLabel(
                  "error.validation.expirationdate.different.message",
                  new Object[] {expirationDateFromRequest, expirationDateFromKmehr}));
        }
      } else {
        errors.add(I18nHelper.getLabel("error.validation.expirationdate.kmehr"));
      }
    } catch (XPathExpressionException
        | ParserConfigurationException
        | SAXException
        | IOException e) {
      Exceptionutils.errorHandler(e);
    }
  }

  /**
   * Creates the secured content type.
   *
   * @param content the content
   * @return the secured content type
   */
  @Profiled(
      logFailuresSeparately = true,
      tag = "0.PrescriberIntegrationModuleV4Impl#createSecuredContentType",
      logger = "org.perf4j.TimingLogger_Common")
  public SecuredContentType createSecuredContentType(final byte[] content) {
    final SecuredContentType secured = new SecuredContentType();
    secured.setSecuredContent(content);
    return secured;
  }

  @Override
  public Prescription.PlainText get(GetPrescription request) {
    return target.get(request);
  }

  @Override
  public be.recipe.api.prescriber.ListPrescriptions.PartialResult<
          ListPrescriptions.Response.PlainText>
      list(be.recipe.api.prescriber.ListPrescriptions request) {
    return target.list(request);
  }

  //TODO list in old way

  @Override
  public void update(PutVisionOtherPrescribers request) {
    target.update(request);
  }
}
