package be.business.connector.recipe.patient;

import static be.business.connector.recipe.utils.JAXB.*;
import static be.recipe.api.Prescription.RID.prescriptionID;
import static be.recipe.api.Prescription.not_found;
import static be.recipe.api.constraints.Violation.Simple.violation;
import static be.recipe.api.executor.Executor.ID.executorId;
import static be.recipe.api.patient.ListPrescriptions.Response.Encrypted.Simple.response;
import static be.recipe.api.patient.Patient.ID.patientID;
import static be.recipe.api.prescriber.Prescriber.ID.prescriberID;
import static be.recipe.services.core.PrescriptionAttribute.PRESCRIPTION_DATE;
import static be.recipe.services.core.PrescriptionAttribute.RESERVATION_MODIFICATION_DATE;
import static be.recipe.services.core.VisionType.PRESCRIBER;
import static be.recipe.services.patient.CreateReservationVersion.ONE;
import static be.recipe.services.patient.CreateReservationVersion.TWO;
import static java8.util.Spliterators.spliteratorUnknownSize;

import be.business.connector.common.ehealth.EhealthCipher;
import be.business.connector.core.services.GenericWebserviceCaller;
import be.business.connector.core.services.GenericWebserviceRequest;
import be.business.connector.core.utils.PropertyHandler;
import be.business.connector.recipe.AbstractRecipeClient;
import be.business.connector.recipe.utils.*;
import be.fgov.ehealth.recipe.protocol.v4.*;
import be.fgov.ehealth.recipe.protocol.v4.RevokePrescriptionResponse;
import be.fgov.ehealth.recipe.protocol.v4.UpdateReservationResponse;
import be.recipe.api.ExecutorProfile;
import be.recipe.api.GetPrescriptionStatusResponse;
import be.recipe.api.Prescription;
import be.recipe.api.constraints.Rejected;
import be.recipe.api.constraints.ValidationReport;
import be.recipe.api.constraints.Violation;
import be.recipe.api.crypto.Message;
import be.recipe.api.patient.*;
import be.recipe.api.patient.AddReservation;
import be.recipe.api.patient.CancelReservation;
import be.recipe.api.patient.GetExecutorProfile;
import be.recipe.api.patient.GetPrescriptionStatus;
import be.recipe.api.patient.RevokePrescription;
import be.recipe.api.prescriber.Prescriber;
import be.recipe.api.prescriber.PrescriberType;
import be.recipe.api.prescriber.VisionOtherPrescribers;
import be.recipe.api.reservation.Reservation;
import be.recipe.api.reservation.patient.ReservationService;
import be.recipe.api.series.PartialResult;
import be.recipe.api.series.SortedBy;
import be.recipe.services.core.*;
import be.recipe.services.patient.*;
import java.util.Calendar;
import java.util.List;
import java8.util.Optional;
import java8.util.Spliterator;
import java8.util.function.Consumer;
import java8.util.function.Function;
import java8.util.function.Supplier;
import java8.util.stream.RefStreams;
import java8.util.stream.Stream;
import java8.util.stream.StreamSupport;
import org.joda.time.DateTime;
import org.threeten.bp.Instant;
import org.threeten.bp.LocalDateTime;
import org.threeten.bp.ZoneId;

public class RecipePatientClient extends AbstractRecipeClient
    implements ReservationService.Simplified, PrescriptionService.Simplified {
  public static final String addressKey = "endpoint.patient.v4";

  public Supplier<CreateReservationVersion> legacyCreateReservationVersion =
      new Supplier<CreateReservationVersion>() {
        @Override
        public CreateReservationVersion get() {
          return ONE;
        }
      };

  public RecipePatientClient() {
    super(addressKey);
  }

  public RecipePatientClient(
      Message.Cipher.Key.DB<Prescription.OnContent> keyRegistry,
      EhealthCipher cipher,
      Supplier<String> traceId) {
    super(addressKey, keyRegistry, cipher, traceId);
  }

  public RecipePatientClient(PropertyHandler properties) {
    super(addressKey, properties);
  }

  @Override
  public void add(AddReservation request) {
    add(toJAXB(request));
  }

  @Override
  public void addLegacy(AddReservation request) {
    add(toJAXBLegacy(request));
  }

  private CreateReservationParam toJAXB(AddReservation from) {
    CreateReservationParam to = new CreateReservationParam();
    to.setRid(from.rid.toString());
    to.setExecutorId(Optional.ofNullable(from.executorId).map(toText()).orElse(null));
    to.setEmailAddress(from.email);
    to.setTelephoneNumber(from.phoneNumber);
    Optional.ofNullable(from.contactPreference)
        .map(toName())
        .map(toContactPreference())
        .ifPresent(setContactPreference(to));
    to.setVersion(TWO);
    return to;
  }

  private static Consumer<ContactPreference> setContactPreference(final CreateReservationParam to) {
    return new Consumer<ContactPreference>() {
      @Override
      public void accept(ContactPreference it) {
        to.setContactPreference(it);
      }
    };
  }

  private CreateReservationParam toJAXBLegacy(AddReservation from) {
    CreateReservationParam to = toJAXB(from);
    to.setVersion(legacyCreateReservationVersion.get());
    return to;
  }

  private void add(CreateReservationParam params) {
    params.setSymmKey(symmKey.getEncoded());

    CreateReservationRequest payload = new CreateReservationRequest();
    payload.setSecuredCreateReservationRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());
    payload.setIssueInstant(DateTime.now());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(AddReservation.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:createReservation\"");

    be.fgov.ehealth.recipe.protocol.v4.CreateReservationResponse response =
        GenericWebserviceCaller.callGenericWebservice(
            request, be.fgov.ehealth.recipe.protocol.v4.CreateReservationResponse.class);
    CreateReservationResult result =
        decrypt(
            response.getSecuredCreateReservationResponse().getSecuredContent(),
            CreateReservationResult.class);
    if (rejected(result.getStatus()))
      throw new RejectedWithResponse(
          from(result.getStatus().getValidationReport(), result), result);
  }

  @Override
  public PartialResult<ListPrescriptions.Response.Encrypted> list(ListPrescriptions request) {
    return fromJAXB(list(toJAXB(request)));
  }

  private ListPatientPrescriptionsParam toJAXB(ListPrescriptions from) {
    final ListPatientPrescriptionsParam to = new ListPatientPrescriptionsParam();
    to.setReserved(Optional.ofNullable(from.reserved).map(toName()).map(toReserved()).orElse(null));
    to.setPage(to(from.page));
    Optional.ofNullable(from.sortedBy).map(toPrescriptionSortedBy()).ifPresent(setSortOptions(to));
    return to;
  }

  private static Consumer<PrescriptionSortedBy> setSortOptions(
      final ListPatientPrescriptionsParam to) {
    return new Consumer<PrescriptionSortedBy>() {
      @Override
      public void accept(PrescriptionSortedBy it) {
        to.setSortOptions(it);
      }
    };
  }

  private Function<SortedBy<Prescription.Attribute>[], PrescriptionSortedBy>
      toPrescriptionSortedBy() {
    return new Function<SortedBy<Prescription.Attribute>[], PrescriptionSortedBy>() {
      @Override
      public PrescriptionSortedBy apply(SortedBy<Prescription.Attribute>[] it) {
        return toJAXB(it);
      }
    };
  }

  private static Function<String, Reserved> toReserved() {
    return new Function<String, Reserved>() {
      @Override
      public Reserved apply(String it) {
        return Reserved.fromValue(it);
      }
    };
  }

  private PrescriptionSortedBy toJAXB(SortedBy<Prescription.Attribute>[] from) {
    final PrescriptionSortedBy to = new PrescriptionSortedBy();
    RefStreams.of(from).map(toPrescriptionSortOrder()).forEach(addSortOrder(to));
    return to;
  }

  private static Consumer<PrescriptionSortedBy.SortedBy> addSortOrder(
      final PrescriptionSortedBy to) {
    return new Consumer<PrescriptionSortedBy.SortedBy>() {
      @Override
      public void accept(PrescriptionSortedBy.SortedBy it) {
        to.getSortedBies().add(it);
      }
    };
  }

  private Function<SortedBy<Prescription.Attribute>, PrescriptionSortedBy.SortedBy>
      toPrescriptionSortOrder() {
    return new Function<SortedBy<Prescription.Attribute>, PrescriptionSortedBy.SortedBy>() {
      @Override
      public PrescriptionSortedBy.SortedBy apply(SortedBy<Prescription.Attribute> it) {
        return toJAXB(it);
      }
    };
  }

  private PrescriptionSortedBy.SortedBy toJAXB(SortedBy<Prescription.Attribute> from) {
    PrescriptionSortedBy.SortedBy to = new PrescriptionSortedBy.SortedBy();
    to.setAttribute(toJAXB(from.attribute));
    to.setOrder(JAXB.toJAXB(from.order));
    return to;
  }

  private PrescriptionAttribute toJAXB(Prescription.Attribute from) {
    switch (from) {
      case prescriptionDate:
        return PRESCRIPTION_DATE;
      case reservationModificationDate:
        return RESERVATION_MODIFICATION_DATE;
      default:
        throw new IllegalArgumentException("Unknown attribute! [" + from + "]");
    }
  }

  private ListPatientPrescriptionsResult list(ListPatientPrescriptionsParam params) {
    params.setSymmKey(symmKey.getEncoded());

    ListOpenPrescriptionsRequest payload = new ListOpenPrescriptionsRequest();
    payload.setSecuredListOpenPrescriptionsRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());
    payload.setIssueInstant(DateTime.now());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(ListPrescriptions.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:listOpenPrescriptions\"");

    be.fgov.ehealth.recipe.protocol.v4.ListOpenPrescriptionsResponse response =
        GenericWebserviceCaller.callGenericWebservice(
            request, be.fgov.ehealth.recipe.protocol.v4.ListOpenPrescriptionsResponse.class);
    return decrypt(
        response.getSecuredListOpenPrescriptionsResponse().getSecuredContent(),
        ListPatientPrescriptionsResult.class);
  }

  private PartialResult<ListPrescriptions.Response.Encrypted> fromJAXB(final ListPatientPrescriptionsResult from) {
    if (rejected(from.getStatus()))
      if (from.getStatus().getMessageCode() != null
          && from.getStatus().getMessageCode().equals("GENERAL_ERROR"))
        throw new RuntimeExceptionWithResponse(from);
      else throw new RejectedWithResponse(from(from.getStatus().getValidationReport(), from), from);
    return PartialResult.Simple.of(
                    new Supplier<Stream<ListPrescriptions.Response.Encrypted>>() {
                      @Override
                      public Stream<ListPrescriptions.Response.Encrypted> get() {
                        return java8.util.stream.StreamSupport.stream(from.getPrescriptions()).map(toListPrescriptionsResponseEncrypted());
                      }
                    })
        .hasMore(from.isHasMoreResults());
  }

  private static Function<GetOpenPrescriptionForPatient, ListPrescriptions.Response.Encrypted>
  toListPrescriptionsResponseEncrypted() {
    return new Function<GetOpenPrescriptionForPatient, ListPrescriptions.Response.Encrypted>() {
      @Override
      public ListPrescriptions.Response.Encrypted apply(GetOpenPrescriptionForPatient it) {
        return RecipePatientClient.toResponse(it);
      }
    };
  }

  private static ListPrescriptions.Response.Encrypted toResponse(final GetOpenPrescriptionForPatient from) {
    ListPrescriptions.Response.Encrypted.Simple to = response(prescriptionID(from.getRid()));
    to.rid = prescriptionID(from.getRid());
    to.patient = patientID(from.getPatientId());
    to.prescriber = prescriberID(from.getPrescriberId());
    to.type = Prescription.Type.prescriptionType(from.getPrescriptionType());
    to.status = Prescription.Status.valueOf(from.getPrescriptionStatus().name());
    to.expirationDate = toLocalDate(from.getExpirationDate());
    to.creationDate = toLocalDateTime(from.getCreationDate());
    to.visibleToExecutors = PrescriptionViewerFormat.parse(from.getVision());
    to.visionOtherPrescribers = JAXB.from(from.getVisionOtherPrescribers());
    to.reservation =
            Optional.ofNullable(from.getReservedAtNihii())
                    .map(
                            new Function<String, ListPrescriptions.Response.Reservation>() {
                              @Override
                              public ListPrescriptions.Response.Reservation apply(String ignored) {
                                final ListPrescriptions.Response.Reservation.Simple reservation =
                                        new ListPrescriptions.Response.Reservation.Simple();
                                reservation.executorId = executorId(from.getReservedAtNihii());
                                Optional.ofNullable(from.getReservationDateTime())
                                        .ifPresent(
                                                new Consumer<Calendar>() {
                                                  @Override
                                                  public void accept(Calendar calendar) {
                                                    reservation.creationDateTime = fromJAXB(calendar);
                                                  }
                                                });
                                Optional.ofNullable(from.getReservationStatus())
                                        .ifPresent(
                                                new Consumer<ReservationStatus>() {
                                                  @Override
                                                  public void accept(ReservationStatus status) {
                                                    reservation.status = Reservation.Status.valueOf(status.name());
                                                  }
                                                });
                                Optional.ofNullable(from.isReservationHasUpdates())
                                        .ifPresent(
                                                new Consumer<Boolean>() {
                                                  @Override
                                                  public void accept(Boolean flag) {
                                                    reservation.hasUpdates = flag;
                                                  }
                                                });
                                reservation.contactPreference = JAXB.from(from.getContactPreference());
                                reservation.email = from.getEmailAddress();
                                reservation.phoneNumber = from.getTelephoneNumber();
                                reservation.feedbackToPatient = from.getFeedbackToPatient();
                                return reservation;
                              }
                            })
                    ;
    return to;
  }

  private static LocalDateTime fromJAXB(Calendar from) {
    return Instant.ofEpochMilli(from.getTimeInMillis())
        .atZone(ZoneId.systemDefault())
        .toLocalDateTime();
  }

  @Override
  public void update(be.recipe.api.patient.PutVisionExecutors.Request request) {
    update(toJAXB(request), be.recipe.api.patient.PutVisionExecutors.class);
  }

  private PutVisionParam toJAXB(be.recipe.api.patient.PutVisionExecutors.Request from) {
    PutVisionParam to = new PutVisionParam();
    if (from.prescription != null) to.setRid(from.prescription.toString());
    if (from.viewer != null) to.setVision(PrescriptionViewerFormat.toString(from.viewer));
    return to;
  }

  private void update(PutVisionParam params, Class<?> usecaseType) {
    params.setSymmKey(symmKey.getEncoded());

    PutVisionForPatientRequest payload = new PutVisionForPatientRequest();
    payload.setSecuredPutVisionForPatientRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(usecaseType.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:putVisionForPatient\"");
    PutVisionForPatientResponse response =
        GenericWebserviceCaller.callGenericWebservice(request, PutVisionForPatientResponse.class);

    PutVisionResult result =
        decrypt(
            response.getSecuredPutVisionForPatientResponse().getSecuredContent(),
            PutVisionResult.class);
    if (rejected(result.getStatus()))
      throw new RejectedWithResponse(
          from(result.getStatus().getValidationReport(), result), result);
  }

  @Override
  public GetExecutorProfile.Response getProfile(GetExecutorProfile request) {
    return fromJAXB(get(toJAXB(request)), request);
  }

  private GetExecutorProfileResult get(GetExecutorProfileParam params) {
    params.setSymmKey(symmKey.getEncoded());

    GetExecutorProfileRequest payload = new GetExecutorProfileRequest();
    payload.setSecuredGetExecutorProfileRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(GetExecutorProfile.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:getExecutorProfile\"");
    be.fgov.ehealth.recipe.protocol.v4.GetExecutorProfileResponse response =
        GenericWebserviceCaller.callGenericWebservice(request, be.fgov.ehealth.recipe.protocol.v4.GetExecutorProfileResponse.class);

    return decrypt(
        response.getSecuredGetExecutorProfileResponse().getSecuredContent(),
        GetExecutorProfileResult.class);
  }

  private GetExecutorProfileParam toJAXB(GetExecutorProfile from) {
    GetExecutorProfileParam to = new GetExecutorProfileParam();
    to.setExecutorId(Optional.ofNullable(from.executorId).map(toText()).orElse(null));
    return to;
  }

  private GetExecutorProfile.Response fromJAXB(
      GetExecutorProfileResult from, GetExecutorProfile request) {
    captureTraceId.accept(from.getId());
    if (rejected(from.getStatus()))
      throw new Rejected(from(from.getStatus().getValidationReport(), request));
    if (from.getProfile() == null) return null;
    return GetExecutorProfile.Response.Simple.profile()
        .reservationFeature(
            ExecutorProfile.ReservationFeature.valueOf(
                from.getProfile().getReservationFeature().value()))
        .communicationChannelSpecification(
            ExecutorProfile.CommunicationChannelSpecification.valueOf(
                from.getProfile().getCommunicationChannelSpecification().name()));
  }

  @Override
  public CancelReservation.Response update(CancelReservation request) {
    UpdateReservationParam params = updateReservationParams(request.rid);
    params.setCancel(toJAXB(request));
    return fromJAXB(update(params, CancelReservation.class).getCancelReservationResult(), request);
  }

  @Override
  public void update(CancelReservation.Legacy request) {
    add(toJAXB(request));
  }

  private CreateReservationParam toJAXB(CancelReservation.Legacy from) {
    CreateReservationParam to = new CreateReservationParam();
    to.setRid(from.rid.toString());
    return to;
  }

  private UpdateReservationParam updateReservationParams(Prescription.RID rid) {
    UpdateReservationParam params = new UpdateReservationParam();
    params.setRid(Optional.ofNullable(rid).map(toText()).orElse(null));
    params.setSymmKey(symmKey.getEncoded());
    return params;
  }

  private UpdateReservationResult update(UpdateReservationParam params, Class<?> usecaseType) {
    UpdateReservationRequest payload = new UpdateReservationRequest();
    payload.setSecuredUpdateReservationRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(usecaseType.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:updateReservation\"");
    UpdateReservationResponse response =
        GenericWebserviceCaller.callGenericWebservice(request, UpdateReservationResponse.class);

    return decrypt(
        response.getSecuredUpdateReservationResponse().getSecuredContent(),
        UpdateReservationResult.class);
  }

  private be.recipe.services.patient.CancelReservation toJAXB(CancelReservation from) {
    return new be.recipe.services.patient.CancelReservation();
  }

  private CancelReservation.Response fromJAXB(
      CancelReservationResult from, CancelReservation root) {
    if (rejected(from.getStatus()))
      throw new Rejected(from(from.getStatus().getValidationReport(), root));
    return new CancelReservation.Response() {};
  }

  @Override
  public UpdateContactDetails.Response  update(UpdateContactDetails request) {
    UpdateReservationParam params = updateReservationParams(request.rid);
    params.setContactDetails(toJAXB(request));
    return fromJAXB(update(params, UpdateContactDetails.class).getContactDetailsResult(), request);
  }



    private ContactDetails toJAXB(UpdateContactDetails from) {
      ContactDetails to = new ContactDetails();
      to.setEmailAddress(from.email);
      to.setTelephoneNumber(from.phoneNumber);
      to.setContactPreference(
              Optional.ofNullable(from.contactPreference)
                      .map(toName())
                      .map(toContactPreference())
                      .orElse(null));
      return to;
    }

    private static Function<String, ContactPreference> toContactPreference() {
      return new Function<String, ContactPreference>() {
        @Override
        public ContactPreference apply(String it) {
          return ContactPreference.valueOf(it);
        }
      };
    }

  private UpdateContactDetails.Response fromJAXB(
      ContactDetailsResult result, UpdateContactDetails request) {
    if (rejected(result.getStatus())) {
      throw new Rejected(from(result.getStatus().getValidationReport(), request));
    }
    return new UpdateContactDetails.Response() {};
  }

    @Override
    public Prescription.Encrypted get(GetPrescription request) {
      return get(toJAXB(request));
    }

    private Prescription.Encrypted fromJAXB(
            GetPrescriptionForPatientResult from, GetPrescriptionForPatientParam root) {
      if (rejected(from.getStatus()))
        throw new Rejected(fromJAXB(from.getStatus().getMessageCode(), root));
      final Prescription.Encrypted.Simple to = new Prescription.Encrypted.Simple();
      to.setRid(prescriptionID(from.getRid()));
      if (from.getPrescriberId() != null) {
        to.setPrescriber(fromJAXB(from.getPrescriberId()));
      }
      if (from.getPatientId() != null) {
        to.setPatient(new Patient.Simple(patientID(from.getPatientId().toString())));
      }
      to.setCreationDate(toLocalDateTime(from.getCreationDate()));
      to.setEncryptedContent(from.getPrescription());
      to.setEncryptionKey(from.getEncryptionKeyId());
      to.setFeedbackAllowed(from.isFeedbackAllowed());
      if (from.getExpirationDate() != null) {
        to.setExpirationDate(toLocalDate(from.getExpirationDate()));
      }
      to.setVisionOtherPrescribers(from(from.getVisionOtherPrescribers()));
      //    to.setVisibleToExecutors(PrescriptionViewerFormat.parse(from.getVision()));
      Optional.ofNullable(from.getReservation())
              .map(toGetPrescriptionReservationResponse(to))
              .ifPresent(
                      new Consumer<Reservation>() {
                        @Override
                        public void accept(Reservation reservation) {
                          to.setReservation(reservation);
                        }
                      });
      return to;
    }

    private Function<GetPrescriptionForPatientResult.Reservation, Reservation>
    toGetPrescriptionReservationResponse(final Prescription.Encrypted prescription) {
      return new Function<GetPrescriptionForPatientResult.Reservation, Reservation>() {
        @Override
        public Reservation apply(GetPrescriptionForPatientResult.Reservation reservation) {
          return fromJAXB(reservation, prescription);
        }
      };
    }

    private Reservation fromJAXB(
            GetPrescriptionForPatientResult.Reservation from, Prescription.Encrypted prescription) {
      return Reservation.Simple.reservation(prescription)
              .executor(executorId(from.getExecutorId()))
              .status(Reservation.Status.valueOf(from.getStatus().name()))
              .contactPreference(JAXB.from(from.getContactPreference()))
              .email(from.getEmail())
              .phoneNumber(from.getPhoneNumber());
    }

    static be.recipe.api.constraints.ValidationReport fromJAXB(
            String code, GetPrescriptionForPatientParam root) {
      Violation.Simple.Builder violation = violation(code);
      if (code.equals(not_found)) violation.on("prescriptionId").value(root.getRid());
      return ValidationReport.Simple.from(violation);
    }

    private Prescriber fromJAXB(long prescriberId) {
      return new Prescriber.Simple(prescriberID(String.valueOf(prescriberId)), PrescriberType.DOCTOR);
    }

    private Prescription.Encrypted get(GetPrescriptionForPatientParam params) {
      params.setSymmKey(symmKey.getEncoded());

      GetPrescriptionRequest payload = new GetPrescriptionRequest();
      payload.setSecuredGetPrescriptionRequest(content(encrypt(params)));
      payload.setProgramId(programId());
      payload.setId(traceId.get());

      GenericWebserviceRequest request = new GenericWebserviceRequest();
      request.setRequest(payload.unwrap());
      request.setRequestType(payload.getClass());
      request.setEndpoint(address);
      request.setServiceName(GetPrescription.class.getSimpleName());
      request.setAddLoggingHandler(true);
      request.setAddSoapFaultHandler(true);
      request.setAddInsurabilityHandler(true);
      request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:getPrescription\"");
      GetPrescriptionResponse response =
              GenericWebserviceCaller.callGenericWebservice(request, GetPrescriptionResponse.class);

      GetPrescriptionForPatientResult result =
              decrypt(
                      response.getSecuredGetPrescriptionResponse().getSecuredContent(),
                      GetPrescriptionForPatientResult.class);
      if (rejected(result.getStatus()))
        throw new RejectedWithResponse(
                from(result.getStatus().getValidationReport(), result), result);
      return fromJAXB(result, params);
    }

    private GetPrescriptionForPatientParam toJAXB(GetPrescription from) {
      GetPrescriptionForPatientParam to = new GetPrescriptionForPatientParam();
      to.setRid(from.rid.toString());
      return to;
    }

    @Override
    public void update(RevokePrescription request) {
      update(toJAXB(request));
    }

    private void update(RevokePrescriptionParam params) {
      params.setSymmKey(symmKey.getEncoded());

      RevokePrescriptionRequest payload = new RevokePrescriptionRequest();
      payload.setSecuredRevokePrescriptionRequest(content(encrypt(params)));
      payload.setProgramId(programId());
      payload.setId(traceId.get());

      GenericWebserviceRequest request = new GenericWebserviceRequest();
      request.setRequest(payload.unwrap());
      request.setRequestType(payload.getClass());
      request.setEndpoint(address);
      request.setServiceName(RevokePrescription.class.getSimpleName());
      request.setAddLoggingHandler(true);
      request.setAddSoapFaultHandler(true);
      request.setAddInsurabilityHandler(false);
      request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:revokePrescription\"");
      RevokePrescriptionResponse response =
              GenericWebserviceCaller.callGenericWebservice(request, RevokePrescriptionResponse.class);
      RevokePrescriptionResult result =
              decrypt(
                      response.getSecuredRevokePrescriptionResponse().getSecuredContent(),
                      RevokePrescriptionResult.class);
      if (rejected(result.getStatus()))
        throw new RejectedWithResponse(
                from(result.getStatus().getValidationReport(), result), result);
    }

    private RevokePrescriptionParam toJAXB(RevokePrescription from) {
      RevokePrescriptionParam to = new RevokePrescriptionParam();
      to.setRid(from.rid.toString());
      to.setReason("-");
      return to;
    }

    @Override
    public void lock(LockPrescription.Request from) {
      Dispatcher.lock(from, this);
    }

    @Override
    public void unlock(UnlockPrescription from) {
      Dispatcher.unlock(from, this);
    }

    @Override
    public void update(PutVisionOtherPrescribers request) {
      update(fromJAXB(request), PutVisionOtherPrescribers.class);
    }

    private PutVisionParam fromJAXB(final PutVisionOtherPrescribers from) {
      final PutVisionParam to = new PutVisionParam();
      Optional.ofNullable(from.rid)
              .ifPresent(
                      new Consumer<Prescription.RID>() {
                        @Override
                        public void accept(Prescription.RID rid) {
                          to.setRid(rid.toString());
                        }
                      });
      Optional.ofNullable(from.vision)
              .ifPresent(
                      new Consumer<VisionOtherPrescribers>() {
                        @Override
                        public void accept(VisionOtherPrescribers it) {
                          to.setVisionOtherPrescribers(
                                  be.recipe.services.core.VisionOtherPrescribers.valueOf(it.name()));
                        }
                      });
      to.setTarget(PRESCRIBER);
      return to;
    }

    @Override
    public GetPrescriptionStatusResponse get(GetPrescriptionStatus request) {
      return get(toJAXB(request));
    }

    private GetPrescriptionStatusResponse get(
            GetPrescriptionStatusParam params) {
      params.setSymmKey(symmKey.getEncoded());

      GetPrescriptionStatusRequest payload = new GetPrescriptionStatusRequest();
      payload.setSecuredGetPrescriptionStatusRequest(content(encrypt(params)));
      payload.setProgramId(programId());
      payload.setIssueInstant(DateTime.now());
      payload.setId(traceId.get());

      GenericWebserviceRequest request = new GenericWebserviceRequest();
      request.setRequest(payload.unwrap());
      request.setRequestType(payload.getClass());
      request.setEndpoint(address);
      request.setServiceName(GetPrescriptionStatus.class.getSimpleName());
      request.setAddLoggingHandler(true);
      request.setAddSoapFaultHandler(true);
      request.setAddInsurabilityHandler(true);
      request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:getPrescriptionStatus\"");

      be.fgov.ehealth.recipe.protocol.v4.GetPrescriptionStatusResponse response =
              GenericWebserviceCaller.callGenericWebservice(
                      request, be.fgov.ehealth.recipe.protocol.v4.GetPrescriptionStatusResponse.class);
      return fromJAXB(
              decrypt(
                      response.getSecuredGetPrescriptionStatusResponse().getSecuredContent(),
                      GetPrescriptionStatusResult.class), params);
    }

    private GetPrescriptionStatusResponse fromJAXB(
            GetPrescriptionStatusResult from, GetPrescriptionStatusParam root) {
      if (rejected(from.getStatus()))
        throw new RejectedWithResponse(fromJAXB(from.getStatus().getMessageCode(),root), from);
      return GetPrescriptionStatusResponse.Simple.response()
              .status(from(from.getPrescriptionStatus()))
              .executor(executor(from.getExecutorId()));
    }

    static ValidationReport fromJAXB(String code, GetPrescriptionStatusParam root) {
      Violation.Simple.Builder violation = violation(code);
      if (code.equals(not_found)) violation.on("prescriptionId").value(root.getRid());
      return ValidationReport.Simple.from(violation);
    }

    private GetPrescriptionStatusParam toJAXB(
            GetPrescriptionStatus from) {
      GetPrescriptionStatusParam to =
              new GetPrescriptionStatusParam();
      to.setRid(from.rid.toString());
      return to;
    }
  }


