package be.business.connector.recipe.executor;

import static be.business.connector.recipe.utils.JAXB.*;
import static be.business.connector.recipe.utils.Locales.getLocalisedMsg;
import static be.business.connector.recipe.utils.PrescriptionViewerFormat.parse;
import static be.recipe.api.Prescription.RID.prescriptionID;
import static be.recipe.api.Prescription.Type.prescriptionType;
import static be.recipe.api.Prescription.delivery_in_progress;
import static be.recipe.api.Prescription.not_found;
import static be.recipe.api.constraints.Violation.Simple.violation;
import static be.recipe.api.executor.GetProfile.Response.Simple.profile;
import static be.recipe.api.executor.ListPrescriptions.Response.Encrypted.Simple.response;
import static be.recipe.api.patient.Patient.ID.patientID;
import static be.recipe.api.prescriber.Prescriber.ID.prescriberID;
import static be.recipe.api.reservation.Reservation.Status.REQUESTED_WITHOUT_COMMITMENT;
import static be.recipe.services.core.ReservationAttribute.MODIFICATION_DATE;
import static java8.util.stream.StreamSupport.stream;

import be.business.connector.common.StandaloneRequestorProvider;
import be.business.connector.common.ehealth.EhealthCipher;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.services.GenericWebserviceCaller;
import be.business.connector.core.services.GenericWebserviceRequest;
import be.business.connector.core.utils.PropertyHandler;
import be.business.connector.recipe.AbstractRecipeClient;
import be.business.connector.recipe.utils.*;
import be.fgov.ehealth.recipe.protocol.v4.*;
import be.fgov.ehealth.recipe.protocol.v4.GetProfileResponse;
import be.fgov.ehealth.recipe.protocol.v4.ListRelationsResponse;
import be.fgov.ehealth.recipe.protocol.v4.MarkAsArchivedResponse;
import be.fgov.ehealth.recipe.protocol.v4.UpdateProfileResponse;
import be.fgov.ehealth.recipe.protocol.v4.UpdateReservationResponse;
import be.recipe.api.ExecutorProfile;
import be.recipe.api.GetPrescriptionStatusResponse;
import be.recipe.api.Prescription;
import be.recipe.api.constraints.Rejected;
import be.recipe.api.constraints.ValidationReport;
import be.recipe.api.constraints.Violation;
import be.recipe.api.crypto.Message;
import be.recipe.api.executor.*;
import be.recipe.api.executor.AcceptReservation;
import be.recipe.api.executor.FulfillReservation;
import be.recipe.api.executor.GetPrescriptionStatus;
import be.recipe.api.executor.GetProfile;
import be.recipe.api.executor.RejectReservation;
import be.recipe.api.executor.UpdateProfile;
import be.recipe.api.mandate.Mandate;
import be.recipe.api.mandate.MandateGiver;
import be.recipe.api.patient.Patient;
import be.recipe.api.prescriber.Prescriber;
import be.recipe.api.prescriber.PrescriberType;
import be.recipe.api.reservation.ContactPreference;
import be.recipe.api.reservation.Reservation;
import be.recipe.api.executor.ListReservations;
import be.recipe.api.reservation.executor.ReservationService;
import be.recipe.api.series.PartialResult;
import be.recipe.api.series.SortedBy;
import be.recipe.common.util.CalendarAdapter;
import be.recipe.services.core.*;
import be.recipe.services.executor.*;
import java.util.HashMap;
import java.util.Map;
import java8.util.Optional;
import java8.util.function.Consumer;
import java8.util.function.Function;
import java8.util.function.Supplier;
import java8.util.stream.RefStreams;
import java8.util.stream.Stream;
import javax.xml.datatype.DatatypeConfigurationException;
import org.apache.commons.collections.keyvalue.MultiKey;
import org.joda.time.DateTime;
import org.threeten.bp.LocalDate;

public class RecipeExecutorClient extends AbstractRecipeClient
    implements PrescriptionService.Simplified,
        ReservationService.Simplified,
        MandateService.Simplified {
  public static final String addressKey = "endpoint.executor.v4";
  private static final CalendarAdapter calendar = new CalendarAdapter();

  public Map<MultiKey, byte[]> sessionMap = new HashMap<>();

  public RecipeExecutorClient(PropertyHandler properties) throws DatatypeConfigurationException {
    super(addressKey, properties);
  }

  public RecipeExecutorClient(PropertyHandler properties, Map<MultiKey, byte[]> sessionMap)
      throws DatatypeConfigurationException {
    super(addressKey, properties);
    this.sessionMap = sessionMap;
  }

  public RecipeExecutorClient(
      Message.Cipher.Key.DB<Prescription.OnContent> keyRegistry,
      EhealthCipher cipher,
      Supplier<String> traceId)
      throws DatatypeConfigurationException {
    super(addressKey, keyRegistry, cipher, traceId);
  }

  @Override
  public Prescription.Encrypted getAndPutInProcess(GetPrescriptionAndPutInProcess request) {
    return get(toJAXB(request, false));
  }

  @Override
  public Prescription.Encrypted get(GetPrescription request) {
    return get(toJAXB(request, true));
  }

  @Override
  public PartialResult<ListPrescriptions.Response.Encrypted> list(ListPrescriptions request) {
    return list(toJAXB(request));
  }

  @Override
  public PartialResult<ListReservations.Response> list(ListReservations.Request request) {
    return list(toJAXB(request));
  }

  private GetPrescriptionForExecutorParam toJAXB(GetPrescription from, boolean alreadyDelivered) {
    GetPrescriptionForExecutorParam to = new GetPrescriptionForExecutorParam();
    to.setRid(from.rid.toString());
    to.setVersion(properties.getProperty("connector.version", "v2"));
    to.setAlreadyDelivered(alreadyDelivered);
    return to;
  }

  private Prescription.Encrypted get(GetPrescriptionForExecutorParam params) {
    params.setSymmKey(symmKey.getEncoded());

    GetPrescriptionForExecutorRequest payload = new GetPrescriptionForExecutorRequest();
    payload.setSecuredGetPrescriptionForExecutorRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setEndpoint(address);
    request.setServiceName(GetPrescription.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:getPrescriptionForExecutor\"");

    be.fgov.ehealth.recipe.protocol.v4.GetPrescriptionForExecutorResponse response =
        GenericWebserviceCaller.callGenericWebservice(
            request, be.fgov.ehealth.recipe.protocol.v4.GetPrescriptionForExecutorResponse.class);
    return fromJAXB(
        decrypt(
            response.getSecuredGetPrescriptionForExecutorResponse().getSecuredContent(),
            GetPrescriptionForExecutorResultSealed.class),
        params);
  }

  private Prescription.Encrypted fromJAXB(
      GetPrescriptionForExecutorResultSealed from, GetPrescriptionForExecutorParam root) {
    if (rejected(from.getStatus()))
      throw new RejectedWithResponse(fromJAXB(from.getStatus().getMessageCode(), root), from);
    Prescription.Encrypted.Simple to = new Prescription.Encrypted.Simple();
    to.setRid(prescriptionID(from.getRid()));
    to.setPrescriber(fromJAXB(from.getPrescriberId()));
    to.setPatient(new Patient.Simple(patientID(from.getPatientId())));
    to.setCreationDate(toLocalDateTime(from.getCreationDate()));
    to.setEncryptedContent(from.getPrescription());
    to.setEncryptionKey(from.getEncryptionKeyId());
    to.setFeedbackAllowed(from.isFeedbackAllowed());
    to.setExpirationDate(toLocalDate(from.getExpirationDate()));
    to.setVisibleToExecutors(PrescriptionViewerFormat.parse(from.getVision()));
    return to;
  }

  static ValidationReport fromJAXB(String code, GetPrescriptionForExecutorParam root) {
    Violation.Simple.Builder violation = violation(normalize(code));
    if (code.equals(not_found)) violation.on("prescriptionId").value(root.getRid());
    if (code.equals(delivery_in_progress))
      violation.on("").of(GetPrescriptionAndPutInProcess.class.getSimpleName());
    return ValidationReport.Simple.from(violation);
  }

  private static String normalize(String code) {
    switch (code) {
      case "error.updatePrescriptionStatus.locked":
        return delivery_in_progress;
      default:
        return code;
    }
  }

  private Prescriber fromJAXB(String id) {
    return new Prescriber.Simple(prescriberID(id), PrescriberType.DOCTOR);
  }

  private ListOpenPrescriptionsParam toJAXB(ListPrescriptions request) {
    ListOpenPrescriptionsParam params = new ListOpenPrescriptionsParam();
    params.setPatientId(request.patient == null ? null : request.patient.toString());
    params.setMandateHolderId(
        request.mandateHolder == null ? null : request.mandateHolder.toString());
    params.setPage(to(request.page));
    return params;
  }

  private ListReservationsParam toJAXB(ListReservations.Request from) {
    ListReservationsParam to = new ListReservationsParam();
    to.setStartDate(calendar.unmarshal(LocalDate.now().minusWeeks(2).toString()));
    to.setPage(to(from.page));
    Optional.ofNullable(from.sortedBy).map(toReservationSortedBy()).ifPresent(setSortOptions(to));
    return to;
  }

  private static Consumer<ReservationSortedBy> setSortOptions(final ListReservationsParam to) {
    return new Consumer<ReservationSortedBy>() {
      @Override
      public void accept(ReservationSortedBy it) {
        to.setSortOptions(it);
      }
    };
  }

  private Function<SortedBy<Reservation.Attribute>[], ReservationSortedBy> toReservationSortedBy() {
    return new Function<SortedBy<Reservation.Attribute>[], ReservationSortedBy>() {
      @Override
      public ReservationSortedBy apply(SortedBy<Reservation.Attribute>[] sortedBIES) {
        return toJAXB(sortedBIES);
      }
    };
  }

  private ReservationSortedBy toJAXB(SortedBy<Reservation.Attribute>[] from) {
    ReservationSortedBy to = new ReservationSortedBy();
    RefStreams.of(from).map(toReservationSortOrder()).forEach(setSortedBy(to));
    return to;
  }

  private static Consumer<ReservationSortedBy.SortedBy> setSortedBy(final ReservationSortedBy to) {
    return new Consumer<ReservationSortedBy.SortedBy>() {
      @Override
      public void accept(ReservationSortedBy.SortedBy it) {
        to.getSortedBies().add(it);
      }
    };
  }

  private Function<SortedBy<Reservation.Attribute>, ReservationSortedBy.SortedBy>
      toReservationSortOrder() {
    return new Function<SortedBy<Reservation.Attribute>, ReservationSortedBy.SortedBy>() {
      @Override
      public ReservationSortedBy.SortedBy apply(SortedBy<Reservation.Attribute> it) {
        return toJAXB(it);
      }
    };
  }

  private ReservationSortedBy.SortedBy toJAXB(SortedBy<Reservation.Attribute> from) {
    ReservationSortedBy.SortedBy to = new ReservationSortedBy.SortedBy();
    to.setAttribute(toJAXB(from.attribute));
    to.setOrder(JAXB.toJAXB(from.order));
    return to;
  }

  private ReservationAttribute toJAXB(Reservation.Attribute from) {
    switch (from) {
      case modificationDate:
        return MODIFICATION_DATE;
      default:
        throw new IllegalArgumentException("Unknown attribute! [" + from + "]");
    }
  }

  public PartialResult<ListPrescriptions.Response.Encrypted> list(
      ListOpenPrescriptionsParam params) {
    params.setSymmKey(symmKey.getEncoded());

    ListOpenPrescriptionsRequest payload = new ListOpenPrescriptionsRequest();
    payload.setSecuredListOpenPrescriptionsRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());
    payload.setIssueInstant(DateTime.now());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(ListPrescriptions.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:listOpenPrescriptions\"");

    be.fgov.ehealth.recipe.protocol.v4.ListOpenPrescriptionsResponse response =
        GenericWebserviceCaller.callGenericWebservice(
            request, be.fgov.ehealth.recipe.protocol.v4.ListOpenPrescriptionsResponse.class);
    return fromJAXB(
        decrypt(
            response.getSecuredListOpenPrescriptionsResponse().getSecuredContent(),
            ListOpenPrescriptionsResult.class),
        request);
  }

  public PartialResult<ListReservations.Response> list(ListReservationsParam params) {
    params.setSymmKey(symmKey.getEncoded());

    ListReservationsRequest payload = new ListReservationsRequest();
    payload.setSecuredListReservationsRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());
    payload.setIssueInstant(DateTime.now());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(ListReservations.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:listReservations\"");

    be.fgov.ehealth.recipe.protocol.v4.ListReservationsResponse response =
        GenericWebserviceCaller.callGenericWebservice(
            request, be.fgov.ehealth.recipe.protocol.v4.ListReservationsResponse.class);
    return fromJAXB(
        decrypt(
            response.getSecuredListReservationsResponse().getSecuredContent(),
            ListReservationsResult.class),
        request);
  }

  private PartialResult<ListPrescriptions.Response.Encrypted> fromJAXB(
      final ListOpenPrescriptionsResult from, Object root) {
    if (rejected(from.getStatus()))
      throw new RejectedWithResponse(from(from.getStatus().getValidationReport(), root), from);
    return PartialResult.Simple.of(
            new Supplier<Stream<ListPrescriptions.Response.Encrypted>>() {
              @Override
              public Stream<ListPrescriptions.Response.Encrypted> get() {
                return stream(from.getPrescriptions()).map(toLisPrescriptionsResponseEncrypted());
              }
            })
        .hasMore(from.isHasMoreResults());
  }

  private static Function<GetOpenPrescriptionForExecutor, ListPrescriptions.Response.Encrypted>
      toLisPrescriptionsResponseEncrypted() {
    return new Function<GetOpenPrescriptionForExecutor, ListPrescriptions.Response.Encrypted>() {
      @Override
      public ListPrescriptions.Response.Encrypted apply(GetOpenPrescriptionForExecutor it) {
        return RecipeExecutorClient.toResponse(it);
      }
    };
  }

  private PartialResult<ListReservations.Response> fromJAXB(
      final ListReservationsResult from, Object root) {
    if (rejected(from.getStatus()))
      throw new Rejected(from(from.getStatus().getValidationReport(), root));
    return PartialResult.Simple.of(
            new Supplier<Stream<Response>>() {
              @Override
              public Stream<Response> get() {
                return stream(from.getItems()).map(toListReservationsResponse());
              }
            })
        .hasMore(from.isHasMoreResults());
  }

  private static Function<ListReservationsResultItem, Response> toListReservationsResponse() {
    return new Function<ListReservationsResultItem, Response>() {
      @Override
      public Response apply(ListReservationsResultItem it) {
        return RecipeExecutorClient.toResponse(it);
      }
    };
  }

  private static ListPrescriptions.Response.Encrypted toResponse(
      GetOpenPrescriptionForExecutor from) {
    ListPrescriptions.Response.Encrypted.Simple to = response(prescriptionID(from.getRid()));
    to.type = prescriptionType(from.getPrescriptionType());
    to.patient = patientID(from.getPatientId());
    to.prescriber = prescriberID(from.getPrescriberId());
    to.creationDate = toLocalDateTime(from.getCreationDate());
    to.expirationDate = toLocalDate(from.getExpirationDate());
    to.encryptedPrescriptionContent = from.getPrescription();
    to.encryptionKey = from.getEncryptionKeyId();
    to.feedbackAllowed = from.isFeedbackAllowed();
    to.status = Prescription.Status.valueOf(from.getPrescriptionStatus().name());
    to.visibleTo = parse(from.getVision());
    return to;
  }

  private static ListReservations.Response toResponse(ListReservationsResultItem from) {
    ListReservations.Response.Simple to = new ListReservations.Response.Simple();
    to.rid = prescriptionID(from.getRid());
    to.visibleTo = parse(from.getVision());
    to.reservationStatus =
        from.getStatus() != null
            ? Reservation.Status.valueOf(from.getStatus().name())
            : REQUESTED_WITHOUT_COMMITMENT;
    to.createdOn = toLocalDateTime(from.getCreationDateReservation());
    to.updatedOn = toLocalDateTime(from.getLastUpdateReservation());
    if (from.getContactPreference() != null)
      to.contactPreference = ContactPreference.valueOf(from.getContactPreference().name());
    to.email = from.getEmailAddress();
    to.phoneNumber = from.getTelephoneNumber();
    to.prescription = toResponse(from.getPrescription());
    return to;
  }

  @Override
  public void update(AcceptReservation request) {
    UpdateReservationParam params = updateReservationParams(request.rid);
    params.setAccept(toJAXB(request));
    fromJAXB(update(params, AcceptReservation.class).getAcceptReservationResult(), request);
  }

  private void fromJAXB(VoidResult from, be.recipe.api.executor.AcceptReservation root) {
    if (rejected(from.getStatus())) {
      throw new Rejected(from(from.getStatus().getValidationReport(), root));
    }
  }

  private UpdateReservationParam updateReservationParams(Prescription.RID rid) {
    UpdateReservationParam params = new UpdateReservationParam();
    params.setRid(Optional.ofNullable(rid).map(toText()).orElse(null));
    params.setSymmKey(symmKey.getEncoded());
    return params;
  }

  private be.recipe.services.executor.AcceptReservation toJAXB(AcceptReservation from) {
    be.recipe.services.executor.AcceptReservation to =
        new be.recipe.services.executor.AcceptReservation();
    to.setMessage(from.message);
    return to;
  }

  private UpdateReservationResult update(UpdateReservationParam params, Class<?> usecaseType) {
    UpdateReservationRequest payload = new UpdateReservationRequest();
    payload.setSecuredUpdateReservationRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(usecaseType.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:updateReservation\"");
    be.fgov.ehealth.recipe.protocol.v4.UpdateReservationResponse response =
        GenericWebserviceCaller.callGenericWebservice(request, UpdateReservationResponse.class);

    return decrypt(
        response.getSecuredUpdateReservationResponse().getSecuredContent(),
        UpdateReservationResult.class);
  }

  @Override
  public void update(RejectReservation request) {
    UpdateReservationParam params = updateReservationParams(request.rid);
    params.setReject(toJAXB(request));
    update(params, RejectReservation.class);
  }

  private be.recipe.services.executor.RejectReservation toJAXB(RejectReservation from) {
    be.recipe.services.executor.RejectReservation to =
        new be.recipe.services.executor.RejectReservation();
    to.setMessage(from.message);
    return to;
  }

  @Override
  public void update(AcceptReservationCancellationRequest request) {
    UpdateReservationParam params = updateReservationParams(request.rid);
    params.setAcceptCancellationRequest(toJAXB(request));
    fromJAXB(
        update(params, AcceptReservationCancellationRequest.class)
            .getAcceptReservationCancellationResult(),
        request);
  }

  private void fromJAXB(VoidResult from, Object root) {
    if (rejected(from.getStatus()))
      throw new Rejected(from(from.getStatus().getValidationReport(), root));
  }

  private AcceptCancellationRequest toJAXB(AcceptReservationCancellationRequest from) {
    return new AcceptCancellationRequest();
  }

  @Override
  public void update(FulfillReservation request) {
    UpdateReservationParam params = updateReservationParams(request.rid);
    params.setFulfill(toJAXB(request));
    update(params, FulfillReservation.class);
  }

  private be.recipe.services.executor.FulfillReservation toJAXB(FulfillReservation from) {
    be.recipe.services.executor.FulfillReservation fulfillReservation =
        new be.recipe.services.executor.FulfillReservation();
    fulfillReservation.setMessage(from.message);
    return fulfillReservation;
  }

  @Override
  public void updateProfile(UpdateProfile request) {
    update(toJAXB(request));
  }

  private void update(UpdateProfileParam params) {
    UpdateProfileRequest payload = new UpdateProfileRequest();
    payload.setSecuredUpdateProfileRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(UpdateProfile.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:updateProfile\"");
    be.fgov.ehealth.recipe.protocol.v4.UpdateProfileResponse response =
        GenericWebserviceCaller.callGenericWebservice(request, be.fgov.ehealth.recipe.protocol.v4.UpdateProfileResponse.class);

    decrypt(
        response.getSecuredUpdateProfileResponse().getSecuredContent(), UpdateProfileResult.class);
  }

  private UpdateProfileParam toJAXB(UpdateProfile request) {
    UpdateProfileParam params = new UpdateProfileParam();
    params.setSymmKey(symmKey.getEncoded());
    if (request.reservationFeature() != null)
      params.setReservationFeature(
          ReservationFeature.valueOf(request.reservationFeature().name().toUpperCase()));
    if (request.communicationChannelSpecification() != null)
      params.setCommunicationChannelSpecification(
          CommunicationChannelSpecification.valueOf(
              request.communicationChannelSpecification().name()));
    return params;
  }

  @Override
  public GetProfile.Response getProfile(GetProfile request) {
    return fromJAXB(get(toJAXB(request)));
  }

  private GetProfileParam toJAXB(GetProfile request) {
    GetProfileParam to = new GetProfileParam();
    to.setSymmKey(symmKey.getEncoded());
    return to;
  }

  private GetProfile.Response fromJAXB(GetProfileResult from) {
    captureTraceId.accept(from.getId());
    if (from.getProfile() == null) return null;
    return profile()
        .reservationFeature(
            ExecutorProfile.ReservationFeature.valueOf(
                from.getProfile().getReservationFeature().value()))
        .communicationChannelSpecification(
            ExecutorProfile.CommunicationChannelSpecification.valueOf(
                from.getProfile().getCommunicationChannelSpecification().name()));
  }

  private GetProfileResult get(GetProfileParam params) {
    GetProfileRequest payload = new GetProfileRequest();
    payload.setSecuredGetProfileRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(GetProfile.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:getProfile\"");
    be.fgov.ehealth.recipe.protocol.v4.GetProfileResponse response =
        GenericWebserviceCaller.callGenericWebservice(request, be.fgov.ehealth.recipe.protocol.v4.GetProfileResponse.class);

    return decrypt(
        response.getSecuredGetProfileResponse().getSecuredContent(), GetProfileResult.class);
  }

  @Override
  public void update(ArchivePrescription request) {
    update(toJAXB(request));
  }

  private void update(MarkAsArchivedParam params) {
    MarkAsArchivedRequest payload = new MarkAsArchivedRequest();
    payload.setSecuredMarkAsArchivedRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setId(traceId.get());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(ArchivePrescription.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:markAsArchived\"");
    be.fgov.ehealth.recipe.protocol.v4.MarkAsArchivedResponse response =
        GenericWebserviceCaller.callGenericWebservice(request, be.fgov.ehealth.recipe.protocol.v4.MarkAsArchivedResponse.class);
    decrypt(
        response.getSecuredMarkAsArchivedResponse().getSecuredContent(),
        MarkAsArchivedResult.class);
  }

  private MarkAsArchivedParam toJAXB(ArchivePrescription from) {
    MarkAsArchivedParam to = new MarkAsArchivedParam();
    Optional.ofNullable(from.rid).map(toText()).ifPresent(setRID(to));
    to.setSymmKey(symmKey.getEncoded());
    return to;
  }

  private static Consumer<String> setRID(final MarkAsArchivedParam to) {
    return new Consumer<String>() {
      @Override
      public void accept(String it) {
        to.setRid(it);
      }
    };
  }

  @Override
  public Stream<ListMandatesByMandateHolder.Response> listMandates(
      ListMandatesByMandateHolder request) {
    return fromJAXB(list(toJAXB(request)), request);
  }

  private Stream<ListMandatesByMandateHolder.Response> fromJAXB(
      ListRelationsResult from, ListMandatesByMandateHolder request) {
    sessionMap.put(toSessionKey(request), from.getSession());
    return stream(from.getItems())
        .map(
            new Function<ListRelationsItem, ListMandatesByMandateHolder.Response>() {
              @Override
              public ListMandatesByMandateHolder.Response apply(final ListRelationsItem from) {
                return new ListMandatesByMandateHolder.Response() {
                  @Override
                  public MandateGiver mandateGiver() {
                    return MandateGiver.Simple.mandateGiver(patientID(from.getPatientId()))
                        .firstName(from.getPatientFirstname())
                        .lastName(from.getPatientLastname());
                  }

                  @Override
                  public Mandate.Type type() {
                    return Mandate.Type.valueOf(from.getMandateHolderType().name());
                  }
                };
              }
            });
  }

  private ListRelationsResult list(final ListRelationsParam params) {
    return withExceptionHandling(
        new Supplier<ListRelationsResult>() {
          @Override
          public ListRelationsResult get() {
            ListRelationsRequest payload = new ListRelationsRequest();
            payload.setSecuredListRelationsRequest(content(encrypt(params)));
            payload.setIssueInstant(DateTime.now());
            payload.setProgramId(programId());
            payload.setId(traceId.get());

            GenericWebserviceRequest request = new GenericWebserviceRequest();
            request.setRequest(payload.unwrap());
            request.setRequestType(payload.getClass());
            request.setEndpoint(address);
            request.setServiceName(ListMandatesByMandateHolder.class.getSimpleName());
            request.setAddLoggingHandler(true);
            request.setAddSoapFaultHandler(true);
            request.setAddInsurabilityHandler(true);
            request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:listRelations\"");
            ListRelationsResponse response =
                GenericWebserviceCaller.callGenericWebservice(request, ListRelationsResponse.class);
            ListRelationsResult result =
                decrypt(
                    response.getSecuredListRelationsResponse().getSecuredContent(),
                    ListRelationsResult.class);
            if (rejected(result.getStatus()))
              if (result.getStatus().getValidationReport() != null)
                throw new Rejected(from(result.getStatus().getValidationReport(), params));
              else
                throw new IntegrationModuleException(getLocalisedMsg(result.getStatus()), result);
            return result;
          }
        });
  }

  private ListRelationsParam toJAXB(ListMandatesByMandateHolder from) {
    ListRelationsParam to = new ListRelationsParam();
    if (from.mandateHolder != null) to.setMandateHolderId(from.mandateHolder.toString());
    to.setSymmKey(symmKey.getEncoded());
    to.setSession(sessionMap.get(toSessionKey(from)));
    return to;
  }

  public static MultiKey toSessionKey(ListMandatesByMandateHolder from) {
    return new MultiKey(
        StandaloneRequestorProvider.getRequestorIdInformation(),
        Optional.ofNullable(from.mandateHolder)
            .map(
                new Function<Patient.ID, Object>() {
                  @Override
                  public Object apply(Patient.ID it) {
                    return it.toString();
                  }
                })
            .orElse(null));
  }

  private <T> T withExceptionHandling(Supplier<T> task) {
    return withExceptionHandling("error.connection.executor", task);
  }

  @Override
  public GetPrescriptionStatusResponse get(GetPrescriptionStatus request) {
    return get(toJAXB(request));
  }

  private GetPrescriptionStatusResponse get(
      be.recipe.services.executor.GetPrescriptionStatusParam params) {
    params.setSymmKey(symmKey.getEncoded());

    GetPrescriptionStatusRequest payload = new GetPrescriptionStatusRequest();
    payload.setSecuredGetPrescriptionStatusRequest(content(encrypt(params)));
    payload.setProgramId(programId());
    payload.setIssueInstant(DateTime.now());
    payload.setId(traceId.get());

    GenericWebserviceRequest request = new GenericWebserviceRequest();
    request.setRequest(payload.unwrap());
    request.setRequestType(payload.getClass());
    request.setEndpoint(address);
    request.setServiceName(GetPrescriptionStatus.class.getSimpleName());
    request.setAddLoggingHandler(true);
    request.setAddSoapFaultHandler(true);
    request.setAddInsurabilityHandler(true);
    request.setSoapAction("\"urn:be:fgov:ehealth:recipe:protocol:v4:getPrescriptionStatus\"");

    be.fgov.ehealth.recipe.protocol.v4.GetPrescriptionStatusResponse response =
        GenericWebserviceCaller.callGenericWebservice(
            request, be.fgov.ehealth.recipe.protocol.v4.GetPrescriptionStatusResponse.class);
    return fromJAXB(
        decrypt(
            response.getSecuredGetPrescriptionStatusResponse().getSecuredContent(),
            be.recipe.services.executor.GetPrescriptionStatusResult.class), params);
  }

  private GetPrescriptionStatusResponse fromJAXB(
          GetPrescriptionStatusResult from, GetPrescriptionStatusParam root) {
    if (rejected(from.getStatus()))
      throw new RejectedWithResponse(fromJAXB(from.getStatus().getMessageCode(),root), from);
    return GetPrescriptionStatusResponse.Simple.response()
        .status(from(from.getPrescriptionStatus()))
        .executor(executor(from.getExecutorId()));
  }

  static ValidationReport fromJAXB(String code, GetPrescriptionStatusParam root) {
    Violation.Simple.Builder violation = violation(normalize(code));
    if (code.equals(not_found)) violation.on("prescriptionId").value(root.getRid());
    return ValidationReport.Simple.from(violation);
  }

  private be.recipe.services.executor.GetPrescriptionStatusParam toJAXB(
      GetPrescriptionStatus from) {
    be.recipe.services.executor.GetPrescriptionStatusParam to =
        new be.recipe.services.executor.GetPrescriptionStatusParam();
    to.setRid(from.rid.toString());
    return to;
  }
}
