package be.business.connector.recipe.executor;

import be.business.connector.recipe.executor.domain.GetAllReservationsParam;
import be.business.connector.recipe.executor.domain.GetPrescriptionForExecutorResult;
import be.business.connector.recipe.executor.dto.ListOpenPrescriptionsResultWithExceptions;
import be.business.connector.recipe.executor.dto.ListReservationsResultWithExceptions;
import be.recipe.api.executor.ListMandatesByMandateHolder;
import be.recipe.services.core.MandateHolderType;
import be.recipe.services.executor.GetOpenPrescriptionForExecutor;
import be.recipe.services.executor.GetPrescriptionStatusParam;
import be.recipe.services.executor.GetPrescriptionStatusResult;
import be.recipe.services.executor.ListNotificationsItem;
import be.recipe.services.executor.ListOpenPrescriptionsParam;
import be.recipe.services.executor.ListOpenPrescriptionsResult;
import be.recipe.services.executor.ListRelationsParam;
import be.recipe.services.executor.ListRelationsResult;
import be.recipe.services.executor.ListReservationsParam;
import be.recipe.services.executor.ListReservationsResult;
import be.recipe.services.executor.ListRidsHistoryParam;
import be.recipe.services.executor.ListRidsHistoryResult;
import be.recipe.services.executor.ListRidsInProcessParam;
import be.recipe.services.executor.ListRidsInProcessResult;
import be.recipe.services.executor.PutRidsInProcessParam;
import be.recipe.services.executor.PutRidsInProcessResult;
import java.util.List;

public interface ExecutorIntegrationModuleV4
    extends ListMandatesByMandateHolder.Command<
        ListMandatesByMandateHolder, ListMandatesByMandateHolder.Response> {

  /**
   * Gets the {@link ListOpenPrescriptionsResult}.
   *
   * @param request a {@link ListOpenPrescriptionsParam}
   * @return the {@link ListOpenPrescriptionsResult}
   */
  ListOpenPrescriptionsResult getData(ListOpenPrescriptionsParam request);

  /**
   * Gets the {@link GetPrescriptionStatusResult}.
   *
   * @param gpsp the gpsp
   * @return the {@link GetPrescriptionStatusResult}
   */
  GetPrescriptionStatusResult getData(GetPrescriptionStatusParam gpsp);

  /**
   * Gets the {@link ListRidsHistoryResult}.
   *
   * @param request the {@link ListRidsHistoryParam}
   * @return the {@link ListRidsHistoryResult}
   */
  ListRidsHistoryResult getData(ListRidsHistoryParam request);

  /**
   * Gets the {@link ListReservationsResult}.
   *
   * @param request the {@link GetAllReservationsParam}
   * @return the {@link ListReservationsResult}
   */
  ListReservationsResult getData(ListReservationsParam request);

  /**
   * Performs a getData(ListReservationsParam)- and decryptListReservationsResult-operation at once.
   *
   * @param request the {@link ListReservationsParam}
   * @return the decrypted {@link ListReservationsResult}
   */
  ListReservationsResult listReservationsDecrypted(ListReservationsParam request);

  ListReservationsResultWithExceptions listReservationsDecryptedWithExceptions(
      ListReservationsParam request);

  /**
   * Gets the {@link ListRidsInProcessResult}.
   *
   * @param request the {@link ListRidsInProcessParam}
   * @return the {@link ListRidsInProcessResult}
   */
  ListRidsInProcessResult getData(ListRidsInProcessParam request);

  /**
   * Puts the {@link PutRidsInProcessResult}.
   *
   * @param request the {@link PutRidsInProcessParam}
   * @return the {@link PutRidsInProcessResult}
   */
  PutRidsInProcessResult putData(PutRidsInProcessParam request);

  /**
   * Gets the {@link be.business.connector.recipe.executor.domain.GetPrescriptionForExecutorResult}
   * and marks it as delivered.
   *
   * @param rid the rid
   * @return the {@link
   *     be.business.connector.recipe.executor.domain.GetPrescriptionForExecutorResult}
   */
  GetPrescriptionForExecutorResult getAndMarkAsDelivered(String rid);

  List<GetPrescriptionForExecutorResult> getAndMarkAsDelivered(List<String> rids);

  /**
   * Gets the {@link be.business.connector.recipe.executor.domain.GetPrescriptionForExecutorResult}
   * and marks it as delivered.
   *
   * @param rid the rid
   * @param mandateHolderId the mandateHolderId
   * @param mandateHolderType the mandateHolderType
   * @return the {@link
   *     be.business.connector.recipe.executor.domain.GetPrescriptionForExecutorResult}
   */
  GetPrescriptionForExecutorResult getAndMarkAsDelivered(
      String rid, String mandateHolderId, MandateHolderType mandateHolderType);

  /**
   * Gets the {@link ListRelationsResult}.
   *
   * @param patientRelationParam the patient relation param
   * @return the {@link ListRelationsResult}
   */
  ListRelationsResult getData(ListRelationsParam patientRelationParam);

  /**
   * Decrypts a {@link GetOpenPrescriptionForExecutor}.
   *
   * @param getOpenPrescriptionForExecutor a the {@link GetOpenPrescriptionForExecutor}
   * @return a {@link GetOpenPrescriptionForExecutor}
   */
  GetOpenPrescriptionForExecutor decryptGetOpenPrescriptionForExecutor(
      final GetOpenPrescriptionForExecutor getOpenPrescriptionForExecutor);

  /**
   * Gets a prescription.
   *
   * @param rid the rid
   * @return the {@link GetPrescriptionForExecutorResult}
   */
  GetPrescriptionForExecutorResult getPrescription(String rid);

  GetPrescriptionForExecutorResult getPrescription(String rid, boolean alreadyDelivered);

  GetPrescriptionForExecutorResult getPrescription(
      String rid, String mandateHolderId, MandateHolderType mandateHolderType);

  /**
   * Gets a prescription.
   *
   * @param rid the rid
   * @param mandateHolderId the mandateHolderId
   * @param mandateHolderType the mandateHolderType
   * @return the {@link GetPrescriptionForExecutorResult}
   */
  GetPrescriptionForExecutorResult getPrescription(
      String rid,
      String mandateHolderId,
      MandateHolderType mandateHolderType,
      boolean alreadyDelivered);

  /**
   * Marks a rid as archived.
   *
   * @param rid the rid
   */
  void markAsArchived(String rid);

  /**
   * Marks a rid as archived.
   *
   * @param rid the rid
   * @param mandateHolderId the mandateHolderId
   * @param mandateHolderType the mandateHolderType
   */
  void markAsArchived(String rid, String mandateHolderId, MandateHolderType mandateHolderType);

  /**
   * Marks a rid as delivered.
   *
   * @param rid the rid
   */
  void markAsDelivered(String rid);

  /**
   * Marks a rid as delivered.
   *
   * @param rid the rid
   * @param mandateHolderId the mandateHolderId
   * @param mandateHolderType the mandateHolderType
   */
  void markAsDelivered(String rid, String mandateHolderId, MandateHolderType mandateHolderType);

  /**
   * Marks a rid as undelivered.
   *
   * @param rid the rid
   */
  void markAsUndelivered(String rid);

  /**
   * Marks a rid as undelivered.
   *
   * @param rid the rid
   * @param mandateHolderId the mandateHolderId
   * @param mandateHolderType the mandateHolderType
   */
  void markAsUndelivered(String rid, String mandateHolderId, MandateHolderType mandateHolderType);

  /**
   * Revokes a rid.
   *
   * @param rid the rid
   * @param reason the reason
   */
  void revokePrescription(String rid, String reason);

  /**
   * Revokes a rid.
   *
   * @param rid the rid
   * @param mandateHolderId the mandateHolderId
   * @param mandateHolderType the mandateHolderType
   */
  void revokePrescription(
      String rid, String reason, String mandateHolderId, MandateHolderType mandateHolderType);

  /**
   * Returns a list of {@link ListNotificationsItem}.
   *
   * @param readFlag the read flag
   * @return the list
   */
  List<ListNotificationsItem> listNotifications(boolean readFlag);

  /**
   * Returns a list of {@link ListNotificationsItem}.
   *
   * @param readFlag the readFlag
   * @param mandateHolderId the mandateHolderId
   * @param mandateHolderType the mandateHolderType
   * @return the list
   */
  List<ListNotificationsItem> listNotifications(
      boolean readFlag, String mandateHolderId, MandateHolderType mandateHolderType);

  /**
   * Creates a feedback.
   *
   * @param prescriberId the prescriber id
   * @param rid the rid
   * @param feedbackText the feedback text
   */
  void createFeedback(String prescriberId, String rid, byte[] feedbackText);

  /**
   * Creates a feedback.
   *
   * @param prescriberId the prescriberId
   * @param rid the rid
   * @param feedbackText the feedbackText
   * @param mandateHolderId the mandateHolderId
   * @param mandateHolderType the mandateHolderType
   */
  void createFeedback(
      String prescriberId,
      String rid,
      byte[] feedbackText,
      String mandateHolderId,
      MandateHolderType mandateHolderType);

  /**
   * Decrypts a {@link ListOpenPrescriptionsResult}.
   *
   * @param listOpenPrescriptionsResult the encrypted {@link ListOpenPrescriptionsResult}
   * @return the decrypted {@link ListOpenPrescriptionsResult}
   */
  ListOpenPrescriptionsResult decryptListOpenPrescriptionsResult(
      ListOpenPrescriptionsResult listOpenPrescriptionsResult);

  ListOpenPrescriptionsResult decryptListOpenPrescriptionsResultSingleThreaded(
          ListOpenPrescriptionsResult listOpenPrescriptionsResult);

  ListOpenPrescriptionsResultWithExceptions decryptListOpenPrescriptionsResultWithExceptions(
      ListOpenPrescriptionsResult listOpenPrescriptionsResult);

  ListOpenPrescriptionsResult listOpenPrescriptionsDecrypted(ListOpenPrescriptionsParam request);

  ListOpenPrescriptionsResult listOpenPrescriptionsDecryptedSingleThreaded(ListOpenPrescriptionsParam request);

  ListOpenPrescriptionsResultWithExceptions listOpenPrescriptionsWithExceptionsDecrypted(
      ListOpenPrescriptionsParam request);

  /**
   * Decrypts a {@link ListReservationsResult}.
   *
   * @param listReservationsResult the encrypted {@link ListReservationsResult}
   * @return the decrypted {@link ListReservationsResult}
   */
  ListReservationsResult decryptListReservationsResult(
      ListReservationsResult listReservationsResult);

  ListReservationsResultWithExceptions decryptListReservationsResultWithExceptions(
      ListReservationsResult listReservationsResult);

  /** Clears the session-map. */
  void clearSession();
}
