package be.business.connector.recipe;

import static be.business.connector.core.utils.I18nHelper.getLabel;
import static be.business.connector.core.utils.RecipeConstants.PROGRAM_IDENTIFICATION;

import be.business.connector.common.ehealth.EhealthCipher;
import be.business.connector.common.ehealth.EhealthKeyRegistry;
import be.business.connector.common.ehealth.EhealthResponseKey;
import be.business.connector.common.ehealth.EhealthSystemKey;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.utils.*;
import be.fgov.ehealth.recipe.core.v4.SecuredContentType;
import be.recipe.api.Prescription;
import be.recipe.api.constraints.Rejected;
import be.recipe.api.crypto.Message;
import be.recipe.services.core.StatusType;
import java.io.IOException;
import java.util.Calendar;
import java.util.Properties;
import java.util.UUID;
import java8.util.function.Consumer;
import java8.util.function.Function;
import java8.util.function.Supplier;
import javax.crypto.SecretKey;
import org.threeten.bp.Instant;
import org.threeten.bp.LocalDate;
import org.threeten.bp.LocalDateTime;
import org.threeten.bp.ZonedDateTime;

public class AbstractRecipeClient {
  private static final String CODE_RED = "300";
  static Properties metadata;

  protected final String addressKey;

  private Message.Cipher.Key.DB<Prescription.OnContent> keyRegistry;
  private RuntimeException exceptionToRaise;
  protected Message.Cipher.Key responseKey;
  protected Message.Cipher.Key systemKey;
  protected PropertyHandler properties;
  protected Message.Cipher cipher;
  private ETKHelper etkHelper;
  protected SecretKey symmKey;
  protected String address;

  public Supplier<String> traceId =
      new Supplier<String>() {
        @Override
        public String get() {
          return "id" + UUID.randomUUID();
        }
      };
  public Consumer<String> captureTraceId =
      new Consumer<String>() {
        @Override
        public void accept(String ignored) {}
      };

  public AbstractRecipeClient(String addressKey) {
    this.addressKey = addressKey;
  }

  public AbstractRecipeClient(
      String addressKey,
      Message.Cipher.Key.DB<Prescription.OnContent> keyRegistry,
      Message.Cipher cipher,
      Supplier<String> traceId) {
    this(addressKey);
    this.keyRegistry = keyRegistry;
    this.traceId = traceId;
    this.cipher = cipher;
  }

  public AbstractRecipeClient(String addressKey, PropertyHandler properties) {
    this(
        addressKey,
        null,
        new EhealthCipher(),
        new Supplier<String>() {
          @Override
          public String get() {
            return "id" + UUID.randomUUID();
          }
        });
    refresh(properties);
  }

  public AbstractRecipeClient(
      String addressKey,
      Message.Cipher.Key.DB<Prescription.OnContent> keyRegistry,
      Message.Cipher cipher,
      PropertyHandler properties) {
    this(
        addressKey,
        keyRegistry,
        cipher,
        new Supplier<String>() {
          @Override
          public String get() {
            return UUID.randomUUID().toString();
          }
        });
    refresh(properties);
  }

  protected static LocalDateTime toLocalDateTime(Calendar calendar) {
    return toZonedDateTime(calendar).toLocalDateTime();
  }

  protected static LocalDate toLocalDate(Calendar calendar) {
    return toZonedDateTime(calendar).toLocalDate();
  }

  private static ZonedDateTime toZonedDateTime(Calendar calendar) {
    return ZonedDateTime.ofInstant(
        Instant.ofEpochMilli(calendar.getTimeInMillis()), org.threeten.bp.ZoneId.systemDefault());
  }

  @SuppressWarnings("rawtypes")
  protected static Function<Enum, String> toName() {
    return new Function<Enum, String>() {
      @Override
      public String apply(Enum it) {
        return it.name();
      }
    };
  }

  protected static Function<String, String> toLowerCase() {
    return new Function<String, String>() {
      @Override
      public String apply(String it) {
        return it.toLowerCase();
      }
    };
  }

  protected static Function<Object, String> toText() {
    return new Function<Object, String>() {
      @Override
      public String apply(Object it) {
        return it.toString();
      }
    };
  }

  public void refresh(PropertyHandler properties) {
    EncryptionUtils encryptionUtils = new EncryptionUtils(PropertyHandler.getInstance());
    address = properties.getProperty(addressKey);
    etkHelper = new ETKHelper(properties, encryptionUtils);
    systemKey = new EhealthSystemKey(etkHelper);
    this.properties = properties;
    if (keyRegistry != null && keyRegistry instanceof EhealthKeyRegistry)
      ((EhealthKeyRegistry) keyRegistry).refresh(etkHelper, properties);
    initEncryption(encryptionUtils);
  }

  public void initEncryption(EncryptionUtils encryptionUtils) {
    ((EhealthCipher) cipher).crypto = null;
    symmKey = encryptionUtils.generateSecretKey();
    responseKey = new EhealthResponseKey(symmKey);
  }

  @SuppressWarnings("unchecked")
  protected <T> byte[] encrypt(T it) {
    final MarshallerHelper<Object, T> helper =
        new MarshallerHelper<>(Object.class, (Class<T>) it.getClass());
    return cipher.encrypt(helper.toXMLByteArray(it), systemKey);
  }

  protected <T> T decrypt(byte[] bytes, Class<T> type) {
    final MarshallerHelper<T, Object> marshaller = new MarshallerHelper<>(type, Object.class);
    return marshaller.toObject(cipher.decrypt(bytes, responseKey));
  }

  protected boolean rejected(StatusType status) {
    return status.getCode().equals(CODE_RED);
  }

  protected SecuredContentType content(byte[] encryptedBytes) {
    SecuredContentType content = new SecuredContentType();
    content.setSecuredContent(encryptedBytes);
    return content;
  }

  protected String programId() {
    return programId(getClass().getSimpleName());
  }

  public static String programId(String defaultPrefix) {
    return PropertyHandler.getInstance().getProperty(PROGRAM_IDENTIFICATION, defaultPrefix)
        + "-Recip-e-"
        + metadata().get("version");
  }

  public static Properties metadata() {
    if (metadata != null) return metadata;
    metadata = metadataWithoutCache();
    return metadata;
  }

  static Properties metadataWithoutCache() {
    Properties metadata = new Properties();
    metadata.setProperty("version", "5.5.10-SNAPSHOT");
    return metadata;
  }

  public void raise(RuntimeException e) {
    this.exceptionToRaise = e;
  }

  protected <T> T withExceptionHandling(String code, Supplier<T> task) {
    try {
      before();
      return task.get();
    } catch (Rejected | IntegrationModuleException e) {
      throw e;
    } catch (RuntimeException e) {
      throw new IntegrationModuleException(getLabel(code), e);
    }
  }

  private void before() {
    if (exceptionToRaise != null) throw exceptionToRaise;
  }
}
