/*
 * (C) 2021 Recip-e. All rights reserved.
 */
package be.business.connector.recipe.utils.prescribercreatebulkprescriptions;

import be.business.connector.core.utils.IOUtils;
import be.business.connector.core.utils.MarshallerHelper;
import be.business.connector.core.utils.PropertyHandler;
import be.business.connector.projects.common.utils.ValidationUtils;
import be.business.connector.recipe.prescriber.AbstractPrescriberIntegrationModule;
import be.business.connector.recipe.prescriber.dto.CreatePrescriptionDTO;
import be.business.connector.recipe.prescriber.services.RecipePrescriberServiceV4;
import be.business.connector.recipe.prescriber.services.RecipePrescriberServiceV4Impl;
import be.ehealth.technicalconnector.service.kgss.domain.KeyResult;
import be.ehealth.technicalconnector.service.sts.security.impl.KeyPairCredential;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionToken;
import be.fgov.ehealth.recipe.core.v4.CreatePrescriptionAdministrativeInformationType;
import be.fgov.ehealth.recipe.protocol.v4.CreatePrescriptionRequest;
import be.fgov.ehealth.recipe.protocol.v4.CreatePrescriptionResponse;
import be.recipe.services.prescriber.CreatePrescriptionParam;
import be.recipe.services.prescriber.CreatePrescriptionResult;
import org.joda.time.DateTime;
import org.w3c.dom.Element;

import java.security.Key;
import java.util.concurrent.Semaphore;

import static be.business.connector.projects.common.utils.ValidationUtils.validateExpirationDate;
import static be.business.connector.recipe.AbstractRecipeClient.programId;
import static be.business.connector.recipe.prescriber.RecipePrescriberClient.toJAXB;

public class CreateEncryptedPrescriptionsTask {

	private final CreatePrescriptionDTO createPrescriptionDTO;
	private final Semaphore semaphore;
	private final KeyResult key;
	private final Key symmKey;
	private final EncryptionToken recipeEtk;
	private final AbstractPrescriberIntegrationModule prescriptionIntegrationModule;
	private final Element assertion;
	private final KeyPairCredential keyPairCredential;

	public CreateEncryptedPrescriptionsTask(final Semaphore available, final CreatePrescriptionDTO createPrescriptionDTO, final KeyResult key,
											final Key symmKey, final EncryptionToken recipeEtk,
											final AbstractPrescriberIntegrationModule prescriptionIntegrationModule, final Element assertion,
											final KeyPairCredential keyPairCredential) {
		this.semaphore = available;
		this.createPrescriptionDTO = createPrescriptionDTO;
		this.key = key;
		this.symmKey = symmKey;
		this.recipeEtk = recipeEtk;
		this.prescriptionIntegrationModule = prescriptionIntegrationModule;
		this.assertion = assertion;
		this.keyPairCredential = keyPairCredential;
	}

	public void run() {
		try {
			ValidationUtils.validatePatientId(createPrescriptionDTO.getPatientId());
			ValidationUtils.validateVisi(createPrescriptionDTO.getVisi(), false);
			final String expirationDate = prescriptionIntegrationModule.extractExpirationDateFromKmehr(createPrescriptionDTO.getPrescription());
			validateExpirationDate(expirationDate);
			prescriptionIntegrationModule.validateKmehr(createPrescriptionDTO.getPrescription(), createPrescriptionDTO.getPrescriptionType(),
					expirationDate);
			final byte[] message = IOUtils.compress(createPrescriptionDTO.getPrescription());
			final byte[] sealedMessage = prescriptionIntegrationModule.sealPrescriptionForUnknown(key, message);
			// create sealed content
			final CreatePrescriptionParam params = new CreatePrescriptionParam();
			params.setPrescription(sealedMessage);
			params.setPrescriptionType(createPrescriptionDTO.getPrescriptionType());
			params.setFeedbackRequested(createPrescriptionDTO.isFeedbackRequested());
			params.setKeyId(key.getKeyId());
			params.setSymmKey(symmKey.getEncoded());
			params.setPatientId(createPrescriptionDTO.getPatientId());

			// New params for CreatePrescriptionParam in V4: begin
			params.setExpirationDate(expirationDate);
			params.setVision(createPrescriptionDTO.getVisi());
			params.setVisionOtherPrescribers(toJAXB(createPrescriptionDTO.getVisionOtherPrescribers()));
			// New params for CreatePrescriptionParam in V4: end

			// init helper
			final MarshallerHelper<CreatePrescriptionResult, CreatePrescriptionParam> helper = new MarshallerHelper<>(CreatePrescriptionResult.class,
					CreatePrescriptionParam.class);

			// create request
			final CreatePrescriptionRequest request = new CreatePrescriptionRequest();
			request.setSecuredCreatePrescriptionRequest(
					prescriptionIntegrationModule
							.createSecuredContentType(prescriptionIntegrationModule.sealRequest(recipeEtk, helper.toXMLByteArray(params))));
			request.setProgramId(programId(getClass().getSimpleName()));
			request.setId(prescriptionIntegrationModule.getId());
			request.setIssueInstant(new DateTime());

			final CreatePrescriptionAdministrativeInformationType adminValue = new CreatePrescriptionAdministrativeInformationType();
			adminValue.setKeyIdentifier(key.getKeyId().getBytes());
			adminValue.setPrescriptionVersion(PropertyHandler.getInstance().getProperty("prescription.version"));
			adminValue.setReferenceSourceVersion(
					prescriptionIntegrationModule.extractReferenceSourceVersionFromKmehr(createPrescriptionDTO.getPrescription()));
			adminValue.setPrescriptionType(createPrescriptionDTO.getPrescriptionType());
			request.setAdministrativeInformation(adminValue);

			// WS call
			final RecipePrescriberServiceV4 prescriberService = RecipePrescriberServiceV4Impl.getInstance();
			final CreatePrescriptionResponse response = prescriberService.createPrescription(request, assertion, keyPairCredential);
			final CreatePrescriptionResult result = helper.unsealWithSymmKey(response.getSecuredCreatePrescriptionResponse().getSecuredContent(),
					symmKey);
			prescriptionIntegrationModule.checkStatus(result);

			createPrescriptionDTO.setRid(result.getRid());
			createPrescriptionDTO.setErrorOccured(false);
		} catch (final Exception exception) {
			createPrescriptionDTO.setException(exception);
			createPrescriptionDTO.setErrorOccured(true);
		} finally {
			semaphore.release();
		}
	}

	public CreatePrescriptionDTO getData() {
		return this.createPrescriptionDTO;
	}

}