package be.business.connector.recipe.utils.executorlistopenprescriptions;

import be.business.connector.common.StandaloneRequestorProvider;
import be.business.connector.core.domain.KgssIdentifierType;
import be.business.connector.core.ehealth.services.OptimizedKgssServiceImpl;
import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.technical.connector.utils.Crypto;
import be.business.connector.core.utils.ETKHelper;
import be.business.connector.core.utils.IOUtils;
import be.ehealth.technicalconnector.service.kgss.domain.KeyResult;
import be.recipe.services.executor.GetOpenPrescriptionForExecutor;
import be.recipe.services.executor.ListOpenPrescriptionsResult;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.util.Map;
import java.util.Set;

import static java.util.stream.Collectors.toSet;
import static org.apache.commons.collections.CollectionUtils.isEmpty;

public class SingleThreadedExecutorListOpenPrescriptionsUseCase {
	private static final Logger LOG = LoggerFactory.getLogger(SingleThreadedExecutorListOpenPrescriptionsUseCase.class);
	private final ETKHelper etkHelper;
	private final Crypto crypto;

	public SingleThreadedExecutorListOpenPrescriptionsUseCase(ETKHelper etkHelper) {
		this.etkHelper = etkHelper;
		this.crypto = new Crypto();
	}

	public ListOpenPrescriptionsResult execute(final ListOpenPrescriptionsResult listOpenPrescriptionsResult) throws IntegrationModuleException {
		if (listOpenPrescriptionsResult == null || isEmpty(listOpenPrescriptionsResult.getPrescriptions())) {
			return listOpenPrescriptionsResult;
		}

		final long startList = System.currentTimeMillis();
		final Map<String, KeyResult> kgssKeys = retrieveKgssKeys(listOpenPrescriptionsResult);
		final long stopList = System.currentTimeMillis();
		LOG.debug("Retrieving KGSS keys took {} ms", stopList - startList);

		final long startDecrypt = System.currentTimeMillis();
		final ListOpenPrescriptionsResult decryptedPrescriptions = doDecryptions(listOpenPrescriptionsResult, kgssKeys);
		final long stopDecrypt = System.currentTimeMillis();
		LOG.debug("Unsealing all prescriptions took {} ms", stopDecrypt - startDecrypt);
		return decryptedPrescriptions;
	}

	private byte[] getPharmacyEtkAsBytes() {
		return etkHelper
				.getEtks(KgssIdentifierType.NIHII_PHARMACY, StandaloneRequestorProvider.getRequestorIdInformation())
				.get(0)
				.getEncoded();
	}

	private ListOpenPrescriptionsResult doDecryptions(
			final ListOpenPrescriptionsResult prescriptionList,
			final Map<String, KeyResult> kgssKeys) {

		final ListOpenPrescriptionsResult finalResult = new ListOpenPrescriptionsResult();

		prescriptionList.getPrescriptions().forEach(prescription -> {
			final KeyResult kgssKey = kgssKeys.get(prescription.getEncryptionKeyId());
			final GetOpenPrescriptionForExecutor decryptedPrescription = decrypt(prescription, kgssKey);
			finalResult.getPrescriptions().add(decryptedPrescription);
		});

		finalResult.setHasMoreResults(prescriptionList.isHasMoreResults());
		finalResult.setSession(prescriptionList.getSession());

		finalResult.getPrescriptions().sort((f1, f2) -> f2.getCreationDate().compareTo(f1.getCreationDate()));

		return finalResult;
	}

	private Map<String, KeyResult> retrieveKgssKeys(final ListOpenPrescriptionsResult prescriptionList) {
		final long startRetrieve = System.currentTimeMillis();

		final Set<String> keyIds = prescriptionList.getPrescriptions().stream()
				.map(GetOpenPrescriptionForExecutor::getEncryptionKeyId)
				.collect(toSet());

		final byte[] pharmacyEtkAsBytes = getPharmacyEtkAsBytes();
		final byte[] kgssEtkAsBytes = etkHelper.getKGSS_ETK().get(0).getEncoded();
		final Map<String, KeyResult> stringKeyResultMap = new OptimizedKgssServiceImpl().retrieveMultipleKeysWithoutCache(keyIds, pharmacyEtkAsBytes, kgssEtkAsBytes);

		final long stopRetrieve = System.currentTimeMillis();
		LOG.debug("Retrieving {} unique kgss keys took {} ms", keyIds.size(), stopRetrieve - startRetrieve);

		return stringKeyResultMap;
	}

	private GetOpenPrescriptionForExecutor decrypt(GetOpenPrescriptionForExecutor prescription, KeyResult kgssKey) {
		final byte[] unsealedPrescription = crypto.unsealForUnknown(kgssKey, prescription.getPrescription());
		try {
			prescription.setPrescription(IOUtils.decompress(unsealedPrescription));
		} catch (IOException e) {
			throw new IntegrationModuleException(e);
		}
		return prescription;
	}

}
