package be.business.connector.recipe.utils.executorlistopenprescriptions;

import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.technical.connector.utils.Crypto;
import be.business.connector.core.utils.I18nHelper;
import be.business.connector.core.utils.IOUtils;
import be.ehealth.technicalconnector.service.kgss.domain.KeyResult;
import be.ehealth.technicalconnector.service.sts.security.impl.KeyPairCredential;
import be.fgov.ehealth.etee.kgss._1_0.protocol.GetKeyRequestContent;
import be.fgov.ehealth.etee.kgss._1_0.protocol.GetKeyResponseContent;
import be.recipe.services.executor.GetOpenPrescriptionForExecutor;
import org.apache.commons.codec.binary.Base64;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Element;

import javax.crypto.spec.SecretKeySpec;
import java.security.PrivateKey;
import java.util.Map;
import java.util.concurrent.Semaphore;

public class DecryptPrescriptionTask {

	private static final Logger LOG = LoggerFactory.getLogger(DecryptPrescriptionTask.class);

	private GetOpenPrescriptionForExecutor prescription;
	private Semaphore semaphore;
	private byte[] pharmacyEtk;
	private byte[] kgssEtk;
	private final KeyPairCredential encryptionCredential;
	private final KeyPairCredential holderOfKeyCredential;
	private final Element samlAssertion;
	private final Map<String, PrivateKey> encryptionPrivateKeys;
	private final be.ehealth.technicalconnector.service.kgss.impl.KgssServiceImpl kgssService;
	private IntegrationModuleException error;
	private final Crypto crypto;

	public DecryptPrescriptionTask(final GetOpenPrescriptionForExecutor prescription, final Semaphore semaphore, final byte[] pharmacyEtk, final byte[] kgssEtk,
								   final KeyPairCredential encryptionCredential, final KeyPairCredential holderOfKeyCredential,
								   final Element samlAssertion, final Map<String, PrivateKey> encryptionPrivateKeys,
								   final be.ehealth.technicalconnector.service.kgss.impl.KgssServiceImpl kgssService, final Crypto crypto) {
		this.prescription = prescription;
		this.semaphore = semaphore;
		this.pharmacyEtk = pharmacyEtk;
		this.kgssEtk = kgssEtk;
		this.encryptionCredential = encryptionCredential;
		this.holderOfKeyCredential = holderOfKeyCredential;
		this.samlAssertion = samlAssertion;
		this.encryptionPrivateKeys = encryptionPrivateKeys;
		this.kgssService = kgssService;
		this.crypto = crypto;
	}

	public void run() {
		try {
			LOG.debug("Starting decrypt prescription task at {}", System.currentTimeMillis());
			final long startRetrieveKgssKey = System.currentTimeMillis();
			final GetKeyRequestContent getKeyRequestContent = new GetKeyRequestContent();
			getKeyRequestContent.setETK(pharmacyEtk);
			final String keyId = prescription.getEncryptionKeyId();
			getKeyRequestContent.setKeyIdentifier(Base64.decodeBase64(keyId));
			final GetKeyResponseContent keyResponse = kgssService.getKey(getKeyRequestContent, holderOfKeyCredential, encryptionCredential,
					samlAssertion, encryptionPrivateKeys, kgssEtk);
			final KeyResult keyResult = new KeyResult(new SecretKeySpec(keyResponse.getKey(), "AES"), keyId);
			final long endRetrieveKgssKey = System.currentTimeMillis();
			LOG.debug("Retrieve KGSS key took {} ms", endRetrieveKgssKey - startRetrieveKgssKey);
			final long startUnseal = System.currentTimeMillis();
			final byte[] unsealedPrescription = crypto.unsealForUnknown(keyResult, prescription.getPrescription());
			final long endUnseal = System.currentTimeMillis();
			LOG.debug("Unsealing prescription took {} ms", endUnseal - startUnseal);
			this.prescription.setPrescription(IOUtils.decompress(unsealedPrescription));
		} catch (final Exception e) {
			this.error = new IntegrationModuleException(I18nHelper.getLabel("technical.connector.error.retrieve.key"), e);
		} finally {
			semaphore.release();
			LOG.debug("Ending decrypt prescription task at {}", System.currentTimeMillis());
		}
	}

	public GetOpenPrescriptionForExecutor getPrescription() {
		return prescription;
	}

	public IntegrationModuleException getError() {
		return error;
	}
}
