package be.business.connector.projects.common.utils;

import be.business.connector.core.exceptions.IntegrationModuleException;
import be.business.connector.core.exceptions.IntegrationModuleValidationException;
import be.business.connector.core.utils.I18nHelper;
import be.recipe.api.prescriber.VisionOtherPrescribers;
import java8.util.Optional;
import java8.util.function.Function;
import org.apache.commons.lang.StringUtils;
import org.joda.time.LocalDate;
import org.joda.time.format.ISODateTimeFormat;
import org.perf4j.aop.Profiled;

/** The Class ValidationUtils. */
public class ValidationUtils {

  /** The Constant PHARMACY_NIHII. */
  private static final String PHARMACY_NIHII = "-PHARMACY.NIHII";

  /** The Constant LOCKED. */
  public static final String LOCKED = "locked";

  /** Private constructor for {@link ValidationUtils}. */
  private ValidationUtils() {}

  /**
   * Validate patient id not blank.
   *
   * @param patientId the patient id
   */
  public static void validatePatientIdNotBlank(final String patientId) {
    if (StringUtils.isBlank(patientId)) {
      throw new IntegrationModuleException("Patient ID is 0.");
    }
  }

  /**
   * Validate patient id.
   *
   * @param patientId the patient id
   */
  @Profiled(
      logFailuresSeparately = true,
      tag = "0.ValidationUtils#validatePatientId",
      logger = "org.perf4j.TimingLogger_Common")
  public static void validatePatientId(final String patientId) {
    if (!INSZUtils.isValidINSZ(patientId)) {
      throw new IntegrationModuleException(
          I18nHelper.getLabel("error.validation.patientid.incorrect"));
    }
  }

  /**
   * Validate mandate holder id.
   *
   * @param mandateHolderId the mandate holder id
   * @param optional the optional
   */
  public static void validateMandateHolderId(final String mandateHolderId, final boolean optional) {
    if (optional) {
      if (!StringUtils.isEmpty(mandateHolderId) && !INSZUtils.isValidINSZ(mandateHolderId)) {
        throw new IntegrationModuleException(
            I18nHelper.getLabel("error.validation.mandateholderid.incorrect"));
      }
    } else {
      if (StringUtils.isEmpty(mandateHolderId) || !INSZUtils.isValidINSZ(mandateHolderId)) {
        throw new IntegrationModuleException(
            I18nHelper.getLabel("error.validation.mandateholderid.incorrect"));
      }
    }
  }

  public static void validateExpirationDate(final String expirationDateText)
      throws IntegrationModuleValidationException {
    validateExpirationDate(
        Optional.ofNullable(expirationDateText)
            .map(
                new Function<String, LocalDate>() {
                  @Override
                  public LocalDate apply(String dateText) {
                    return parseDate(dateText, "error.validation.expirationdate4");
                  }
                })
            .orElse(null));
  }

  private static void validateExpirationDate(final LocalDate expirationDate)
      throws IntegrationModuleValidationException {
    validateExpirationDate(expirationDate, LocalDate.now());
  }

  public static void validateExpirationDate(LocalDate expirationDate, LocalDate today) {
    if (expirationDate == null) {
      final String label = I18nHelper.getLabel("error.validation.expirationdate1", null);
      throw new IntegrationModuleValidationException(label, label);
    }
  }

  private static LocalDate parseDate(String dateText, String errorKey) {
    try {
      return LocalDate.parse(dateText, ISODateTimeFormat.date());
    } catch (final Exception ex) {
      final String label = I18nHelper.getLabel(errorKey, new Object[] {dateText});
      throw new IntegrationModuleValidationException(label, label);
    }
  }

  /**
   * Validate visi.
   *
   * @param vision the vision
   * @param patient the patient
   */
  @Profiled(
      logFailuresSeparately = true,
      tag = "0.ValidationUtils#validateVisi",
      logger = "org.perf4j.TimingLogger_Common")
  public static void validateVisi(final String vision, final boolean patient) {
    if (patient) {
      if (vision == null
          || vision.equals("")
          || vision.equalsIgnoreCase("null")
          || vision.equalsIgnoreCase(LOCKED)
          || vision.endsWith(PHARMACY_NIHII)) {
        if (vision != null && vision.endsWith(PHARMACY_NIHII)) {
          final String executorNbr = vision.replaceAll(PHARMACY_NIHII, "");
          if (!StringUtils.isNumeric(executorNbr)) {
            // not numeric number
            throw new IntegrationModuleValidationException(
                I18nHelper.getLabel(
                    "error.validation.vision.invalid.message", new Object[] {vision}));
          }
        }
      } else {
        throw new IntegrationModuleValidationException(
            I18nHelper.getLabel("error.validation.vision.invalid.message", new Object[] {vision}));
      }
    } else {
      if (vision == null
          || vision.equals("")
          || vision.equalsIgnoreCase("null")
          || vision.equalsIgnoreCase(LOCKED)) {
        // ok
      } else {
        throw new IntegrationModuleValidationException(
            I18nHelper.getLabel(
                "error.validation.vision.invalid.prescriber.message", new Object[] {vision}));
      }
    }
  }
}
