/*
 * Copyright (c) eHealth
 */
package be.ehealth.technicalconnector.beid;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertNull;
import static org.junit.Assert.assertTrue;

import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import be.ehealth.technicalconnector.beid.domain.Address;
import be.ehealth.technicalconnector.beid.domain.Gender;
import be.ehealth.technicalconnector.beid.domain.Identity;
import be.ehealth.technicalconnector.beid.domain.SpecialStatus;
import be.ehealth.technicalconnector.utils.DateUtils;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.BeIDInfoRule;


/**
 * IntegrationTest for BeIDInfo
 *
 * @author EHP
 */
public class BeIDInfoTest {

    private static final Logger LOG = LoggerFactory.getLogger(BeIDInfoTest.class);

    @ClassRule
    public static BeIDInfoRule rule = BeIDInfoRule.activateEid().build();


    @Test
    public void validateIdentityMapping() throws Exception {
        rule.withLocation("/beid/Alice.xml");

        Identity identity = BeIDInfo.getInstance("validateMapping").getIdentity();
        LOG.debug(identity.toString());
        assertEquals("SPECIMEN", identity.getName());
        assertEquals("Alice Geldigekaart", identity.getFirstName());
        assertEquals("000000122157", identity.getCardNumber());
        assertTrue(DateUtils.parseDate("2009-07-23").isEqual(identity.getCardValidityDateBegin()));
        assertTrue(DateUtils.parseDate("2014-07-23").isEqual(identity.getCardValidityDateEnd()));
        assertEquals("Antwerpen", identity.getCardDeliveryMunicipality());
        assertEquals("71715100070", identity.getNationalNumber());
        assertEquals("A", identity.getMiddleName());
        assertEquals("BELG", identity.getNationality());
        assertEquals("Hamont-Achel", identity.getPlaceOfBirth());
        assertEquals(Gender.FEMALE, identity.getGender());
        assertNotNull(identity.getDateOfBirth());
        assertTrue(DateUtils.parseDate("1971-01-01").isEqual(identity.getDateOfBirth()));
        assertEquals(SpecialStatus.NO_STATUS, identity.getSpecialStatus());
        assertNull(identity.getSpecialOrganisation());
        assertNotNull(identity.getData());
    }

    @Test
    public void validateIdentityExtendedMinority() throws Exception {
        rule.withLocation("/beid/Caroline.xml");
        Identity identity = BeIDInfo.getInstance("validateMapping").getIdentity();
        assertEquals("SPECIMEN", identity.getName());
        assertEquals("Caroline Expirée2466", identity.getFirstName());
        assertEquals("000000246641", identity.getCardNumber());
        assertTrue(DateUtils.parseDate("2005-08-08").isEqual(identity.getCardValidityDateBegin()));
        assertTrue(DateUtils.parseDate("2010-08-08").isEqual(identity.getCardValidityDateEnd()));
        assertEquals("Certipost Specimen", identity.getCardDeliveryMunicipality());
        assertEquals("79797900036", identity.getNationalNumber());
        assertEquals("C", identity.getMiddleName());
        assertEquals("Belge", identity.getNationality());
        assertEquals("Hélécine", identity.getPlaceOfBirth());
        assertEquals(Gender.FEMALE, identity.getGender());
        assertNotNull(identity.getDateOfBirth());
        assertTrue(DateUtils.parseDate("1979-09-29").isEqual(identity.getDateOfBirth()));
        assertEquals(SpecialStatus.EXTENDED_MINORITY, identity.getSpecialStatus());
        assertNull(identity.getSpecialOrganisation());
        assertNotNull(identity.getData());
    }

    @Test
    public void validatePhoto() throws Exception {
        rule.withLocation("/beid/Alice.xml");
        byte[] photo = BeIDInfo.getInstance("validateMapping").getPhoto();
        assertNotNull(photo);
    }

    @Test
    public void validateAddressMapping() throws Exception {
        rule.withLocation("/beid/Alice.xml");
        Address actual = BeIDInfo.getInstance("validateMapping").getAddress();
        Assert.assertEquals("Meirplaats 1 bus 1", actual.getStreetAndNumber());
        Assert.assertEquals("2000", actual.getZip());
        Assert.assertEquals("Antwerpen", actual.getMunicipality());
    }


}
