/*
 * Decompiled with CFR 0.152.
 */
package be.fgov.ehealth.etee.crypto.encrypt;

import be.fgov.ehealth.etee.crypto.cert.CertPathChecker;
import be.fgov.ehealth.etee.crypto.cert.CertificateStatus;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionToken;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionTokenChecker;
import be.fgov.ehealth.etee.crypto.encrypt.EncryptionTokenResult;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPChecker;
import be.fgov.ehealth.etee.crypto.ocsp.OCSPData;
import be.fgov.ehealth.etee.crypto.status.CryptoResult;
import be.fgov.ehealth.etee.crypto.status.NotificationError;
import be.fgov.ehealth.etee.crypto.utils.KeyManager;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.NoSuchProviderException;
import java.security.SignatureException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateExpiredException;
import java.security.cert.CertificateNotYetValidException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import javax.naming.InvalidNameException;
import javax.naming.ldap.LdapName;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class EncryptionTokenCheckerImpl
implements EncryptionTokenChecker {
    private static final Logger LOGGER = LoggerFactory.getLogger(EncryptionTokenCheckerImpl.class);
    private OCSPChecker ocspChecker;
    private CertPathChecker certPathChecker;

    public EncryptionTokenCheckerImpl(OCSPChecker ocspChecker, CertPathChecker certPathChecker) {
        this.ocspChecker = ocspChecker;
        this.certPathChecker = certPathChecker;
    }

    @Override
    public CryptoResult<EncryptionToken> validate(EncryptionToken encryptionToken) {
        EncryptionTokenResult result = new EncryptionTokenResult(encryptionToken);
        this.verifySignature(encryptionToken, result);
        this.verifyValidity(encryptionToken, result);
        this.verifyKeyUsage(encryptionToken, result);
        this.verifyCertCorrelation(encryptionToken, result);
        this.verifyCertChain(encryptionToken, result);
        return result;
    }

    private void verifySignature(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        LOGGER.info("verifying whether the encryption certificate was signed by the private key that corresponds  with the included authentication certificate");
        try {
            encryptionToken.getCertificate().verify(encryptionToken.getAuthenticationCertificate().getPublicKey());
        }
        catch (SignatureException e) {
            LOGGER.warn("ETK encryption certificate signature is not valid", (Throwable)e);
            result.getErrors().add(NotificationError.ETK_ENCRYPTION_CERT_SIGNATURE_NOT_VALID);
        }
        catch (InvalidKeyException e) {
            LOGGER.warn("ETK authentication certificate public key is not valid", (Throwable)e);
            result.getErrors().add(NotificationError.ETK_AUTHENTICATION_CERT_HAS_INVALID_KEY);
        }
        catch (CertificateException e) {
            LOGGER.warn("ETK certificates are not valid", (Throwable)e);
            result.getErrors().add(NotificationError.ETK_CERTS_INVALID);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalArgumentException("Signature algorithm is not available in the runtime environment.", e);
        }
        catch (NoSuchProviderException e) {
            throw new IllegalArgumentException("The default security provider hasn't been configured yet. Please execute SecurityConfiguration.configure()", e);
        }
    }

    private void verifyValidity(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        this.verifyEncryptionCertValidity(encryptionToken, result);
        this.verifyAuthenticationCertValidity(encryptionToken, result);
    }

    private void verifyKeyUsage(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        this.verifyEncryptionCertKeyUsage(encryptionToken, result);
        this.verifyAuthenticationCertKeyUsage(encryptionToken, result);
    }

    private void verifyCertChain(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        this.verifyAuthenticationCertChain(encryptionToken, result);
    }

    private void verifyEncryptionCertValidity(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        LOGGER.info("Verify validity of ETK encryption certificate.");
        try {
            encryptionToken.getCertificate().checkValidity();
        }
        catch (CertificateExpiredException e) {
            LOGGER.warn("ETK encryption certificate has expired", (Throwable)e);
            result.getErrors().add(NotificationError.ETK_ENCRYPTION_CERT_EXPIRED);
        }
        catch (CertificateNotYetValidException e) {
            LOGGER.warn("ETK encryption certificate not yet valid", (Throwable)e);
            result.getErrors().add(NotificationError.ETK_ENCRYPTION_CERT_NOT_YET_VALID);
        }
    }

    private void verifyAuthenticationCertValidity(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        LOGGER.info("Verify validity of ETK authentication certificate.");
        try {
            encryptionToken.getAuthenticationCertificate().checkValidity();
        }
        catch (CertificateExpiredException e) {
            LOGGER.warn("ETK authentication certificate has expired", (Throwable)e);
            result.getErrors().add(NotificationError.ETK_AUTHENTICATION_CERT_EXPIRED);
        }
        catch (CertificateNotYetValidException e) {
            LOGGER.warn("ETK authentication certificate not yet valid", (Throwable)e);
            result.getErrors().add(NotificationError.ETK_AUTHENTICATION_CERT_NOT_YET_VALID);
        }
    }

    private void verifyEncryptionCertKeyUsage(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        LOGGER.info("verifying the key usage of the encryption certificate");
        if (!EncryptionTokenCheckerImpl.isEncryptionCertificate(encryptionToken.getCertificate())) {
            LOGGER.warn("The encryption certificate has a wrong key usage.");
            result.getErrors().add(NotificationError.ETK_ENCRYPTION_CERT_HAS_INVALID_KEYUSAGE);
        }
    }

    private void verifyAuthenticationCertKeyUsage(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        LOGGER.info("verifying the key usage of the authentication certificate");
        if (!EncryptionTokenCheckerImpl.isAuthenticationCertificate(encryptionToken.getAuthenticationCertificate())) {
            LOGGER.warn("The authentication certificate has a wrong key usage.");
            result.getErrors().add(NotificationError.ETK_AUTHENTICATION_CERT_HAS_INVALID_KEYUSAGE);
        }
    }

    private void verifyCertCorrelation(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        LOGGER.info("Verify correlation between ETK certificates.");
        try {
            LdapName authDN = new LdapName(encryptionToken.getAuthenticationCertificate().getSubjectX500Principal().getName("RFC1779"));
            LdapName encrDN = new LdapName(encryptionToken.getCertificate().getSubjectX500Principal().getName("RFC1779"));
            boolean ownerCheckOK = authDN.equals(encrDN);
            LOGGER.info("Owner of auth cert and encr cert is same entity ? " + ownerCheckOK);
            if (!ownerCheckOK) {
                LOGGER.warn("Certificate owner of encryption and authentication certificate is not one and the same");
                LOGGER.warn("auth cert owner : " + encryptionToken.getAuthenticationCertificate().getSubjectX500Principal().toString());
                LOGGER.warn("encr cert owner : " + encryptionToken.getCertificate().getSubjectX500Principal().toString());
                result.getErrors().add(NotificationError.ETK_CERTS_DO_NOT_MATCH);
            }
        }
        catch (InvalidNameException e) {
            LOGGER.error("SubjectDN of ETK Certificates DN could not be parsed", (Throwable)e);
            result.getErrors().add(NotificationError.ETK_CERTS_INVALID);
        }
    }

    private void verifyAuthenticationCertChain(EncryptionToken encryptionToken, EncryptionTokenResult result) {
        CertificateStatus certStatus;
        LOGGER.info("verifying the certificate chain of the authentication certificate in the ETK");
        ArrayList<X509Certificate> certChain = new ArrayList<X509Certificate>();
        certChain.add(encryptionToken.getAuthenticationCertificate());
        certChain.addAll(encryptionToken.getCaCertChain());
        CryptoResult<CertificateStatus> certPathResult = this.certPathChecker.validate(certChain);
        if (certPathResult.hasErrors()) {
            LOGGER.warn("CertPath Validation completed with errors: " + certPathResult);
            result.getErrors().addAll(certPathResult.getErrors());
        }
        if (certPathResult.hasWarnings()) {
            LOGGER.warn("CertPath Validation completed with warnings: " + certPathResult.getWarnings());
            result.getWarnings().addAll(certPathResult.getWarnings());
        }
        if ((certStatus = certPathResult.getData()).equals((Object)CertificateStatus.VALID)) {
            CryptoResult<OCSPData> ocspResult = this.ocspChecker.validate(certChain);
            if (ocspResult.hasErrors()) {
                LOGGER.warn("OCSP Validation completed with errors: " + ocspResult);
                result.getErrors().addAll(ocspResult.getErrors());
                result.getErrors().add(NotificationError.ETK_AUTHENTICATION_CERT_STATUS_UNKNOWN);
                return;
            }
            if (ocspResult.hasWarnings()) {
                LOGGER.warn("OCSP Validation completed with warnings: " + ocspResult.getWarnings());
                result.getWarnings().addAll(ocspResult.getWarnings());
            }
            certStatus = ocspResult.getData().getCertStatus();
        }
        switch (certStatus) {
            case VALID: {
                LOGGER.info("ETK authentication cert chain trusted and not revoked");
                break;
            }
            case EXPIRED: {
                LOGGER.warn("ETK Authentication cert is expired");
                result.getErrors().add(NotificationError.ETK_AUTHENTICATION_CERT_EXPIRED);
                break;
            }
            case REVOKED: {
                LOGGER.warn("ETK Authentication cert is revoked");
                result.getErrors().add(NotificationError.ETK_AUTHENTICATION_CERT_REVOKED);
                break;
            }
            default: {
                LOGGER.warn("Authentication certificate chain in ETK is not valid");
                LOGGER.warn("Authentication certificate : " + ((Object)((X509Certificate)certChain.get(0)).getSubjectDN()).toString());
                result.getErrors().add(NotificationError.ETK_AUTHENTICATION_CERT_STATUS_UNKNOWN);
            }
        }
    }

    private static boolean isAuthenticationCertificate(X509Certificate cert) {
        if (cert.getBasicConstraints() >= 0) {
            return false;
        }
        int keyUsage = KeyManager.getKeyUsage(cert);
        return (keyUsage & 0x80) == 128;
    }

    private static boolean isEncryptionCertificate(X509Certificate cert) {
        if (cert.getBasicConstraints() >= 0) {
            return false;
        }
        int keyUsage = KeyManager.getKeyUsage(cert);
        return (keyUsage & 0x30) == 48;
    }
}

