/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.test.civicsv2.session;

import java.io.InputStream;

import javax.xml.ws.soap.SOAPFaultException;

import be.ehealth.businessconnector.civicsv2.session.CivicsSessionService;
import be.ehealth.businessconnector.civicsv2.session.CivicsSessionServiceFactory;
import org.junit.Assert;
import org.junit.BeforeClass;
import org.junit.ClassRule;
import org.junit.Test;

import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.samcivics.schemas.v2.FindCNKRequest;
import be.fgov.ehealth.samcivics.schemas.v2.FindCNKResponse;
import be.fgov.ehealth.samcivics.schemas.v2.FindParagraphRequest;
import be.fgov.ehealth.samcivics.schemas.v2.FindParagraphResponse;
import be.fgov.ehealth.samcivics.schemas.v2.FindParagraphTextRequest;
import be.fgov.ehealth.samcivics.schemas.v2.FindParagraphTextResponse;
import be.fgov.ehealth.samcivics.schemas.v2.FindPublicCNKRequest;
import be.fgov.ehealth.samcivics.schemas.v2.FindPublicCNKResponse;
import be.fgov.ehealth.samcivics.schemas.v2.FindReimbursementConditionsRequest;
import be.fgov.ehealth.samcivics.schemas.v2.FindReimbursementConditionsResponse;
import be.fgov.ehealth.samcivics.schemas.v2.GetAddedDocumentsRequest;
import be.fgov.ehealth.samcivics.schemas.v2.GetAddedDocumentsResponse;
import be.fgov.ehealth.samcivics.schemas.v2.GetParagraphExclusionsRequest;
import be.fgov.ehealth.samcivics.schemas.v2.GetParagraphExclusionsResponse;
import be.fgov.ehealth.samcivics.schemas.v2.GetParagraphIncludedSpecialtiesRequest;
import be.fgov.ehealth.samcivics.schemas.v2.GetParagraphIncludedSpecialtiesResponse;
import be.fgov.ehealth.samcivics.schemas.v2.GetProfessionalAuthorizationsRequest;
import be.fgov.ehealth.samcivics.schemas.v2.GetProfessionalAuthorizationsResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;

/**
 * Integration tests for tarification.
 * 
 * @author eHealth Platform
 * 
 */
public final class CivicsIntegrationTest {

    @ClassRule
    public static SessionRule sessionRule = SessionRule.withActiveSession().baseOn("/be.ehealth.businessconnector.civicsv2.test.properties").build();

    private static CivicsSessionService service;

    /**
     * @param oa
     */
    public CivicsIntegrationTest() {
        super();
    }

    @BeforeClass
    public static void init() throws Exception {
        service = CivicsSessionServiceFactory.getCivicsSession();
    }

    private <T> T transformLocationToObject(String location, Class<T> classT) throws Exception {
        InputStream is = ConnectorIOUtils.getResourceAsStream(location);
        return ConnectorXmlUtils.toObject(is, classT);
    }

    @Test(expected = SOAPFaultException.class)
    public void findCNKTestXmlNoDataFound() throws Exception {
        findCNKTest("/examples/FindCNK-NoDataFound.xml");
    }

    @Test
    public void findCNKTestXml() throws Exception {
        FindCNKResponse response = findCNKTest("/examples/FindCNK.xml");
        Assert.assertTrue("No products found", response.getProducts().size() >= 1);
    }

    private FindCNKResponse findCNKTest(String location) throws Exception {
        FindCNKRequest request = transformLocationToObject(location, FindCNKRequest.class);
        return service.findCNK(request);
    }

    @Test(expected = SOAPFaultException.class)
    public void findParagraphXmlNoDataFound() throws Exception {
        findParagraphClassification("/examples/FindParagraph-NoDataFound.xml");
    }

    @Test
    public void findParagraphXml() throws Exception {
        FindParagraphResponse response = findParagraphClassification("/examples/FindParagraph.xml");
        Assert.assertTrue("No paragraphs found", response.getFindParagraphs().size() >= 1);
    }

    private FindParagraphResponse findParagraphClassification(String location) throws Exception {
        FindParagraphRequest request = transformLocationToObject(location, FindParagraphRequest.class);
        return service.findParagraph(request);
    }

    @Test(expected = SOAPFaultException.class)
    public void findParagraphTextXmlNoDataFound() throws Exception {
        findParagraphText("/examples/FindParagraphText-NoDataFound.xml");
    }

    @Test
    public void findParagraphTextXml() throws Exception {
        FindParagraphTextResponse response = findParagraphText("/examples/FindParagraphText.xml");
        Assert.assertNotNull("No paragraph text found", response.getParagraph());
    }

    private FindParagraphTextResponse findParagraphText(String location) throws Exception {
        FindParagraphTextRequest request = transformLocationToObject(location, FindParagraphTextRequest.class);
        return service.findParagraphText(request);
    }

    @Test(expected = SOAPFaultException.class)
    public void findPublicCNKXmlNoDataFound() throws Exception {
        findPublicCNK("/examples/FindPublicCNK-NoDataFound.xml");
    }

    @Test
    public void findPublicCNKXml() throws Exception {
        FindPublicCNKResponse response = findPublicCNK("/examples/FindPublicCNK.xml");
        Assert.assertTrue("No ampps found", response.getAmpps().size() >= 1);
    }

    private FindPublicCNKResponse findPublicCNK(String location) throws Exception {
        FindPublicCNKRequest request = transformLocationToObject(location, FindPublicCNKRequest.class);
        return service.findPublicCNK(request);
    }

    @Test(expected = SOAPFaultException.class)
    public void findReimbursementConditionsXmlNoDataFound() throws Exception {
        findReimbursementConditions("/examples/FindReimboursementConditions-NoDataFound.xml");
    }

    @Test
    public void findReimbursementConditionsXml() throws Exception {
        FindReimbursementConditionsResponse response = findReimbursementConditions("/examples/FindReimboursementConditions.xml");
        Assert.assertTrue("No paragraph and therapies found", response.getParagraphAndTherapies().size() >= 1);
    }

    private FindReimbursementConditionsResponse findReimbursementConditions(String location) throws Exception {
        FindReimbursementConditionsRequest request = transformLocationToObject(location, FindReimbursementConditionsRequest.class);
        return service.findReimbursementConditions(request);
    }

    @Test(expected = SOAPFaultException.class)
    public void getAddedDocumentsXmlNoDataFound() throws Exception {
        getAddedDocuments("/examples/GetAddedDocuments-NoDataFound.xml");
    }

    @Test
    public void getAddedDocumentsXml() throws Exception {
        GetAddedDocumentsResponse response = getAddedDocuments("/examples/GetAddedDocuments.xml");
        Assert.assertTrue("No added documents found", response.getAddedDocuments().size() >= 1);
    }

    private GetAddedDocumentsResponse getAddedDocuments(String location) throws Exception {
        GetAddedDocumentsRequest request = transformLocationToObject(location, GetAddedDocumentsRequest.class);
        return service.getAddedDocuments(request);
    }

    @Test(expected = SOAPFaultException.class)
    public void getParagraphExclusionXmlNoDataFound() throws Exception {
        getParagraphExclusion("/examples/GetParagraphExclusion-NoDataFound.xml");
    }

    @Test
    public void getParagraphExclusionXml() throws Exception {
        GetParagraphExclusionsResponse response = getParagraphExclusion("/examples/GetParagraphExclusion.xml");
        Assert.assertTrue("No excluded paragraphs found", response.getExcludedParagraphs().size() >= 1);
    }

    private GetParagraphExclusionsResponse getParagraphExclusion(String location) throws Exception {
        GetParagraphExclusionsRequest request = transformLocationToObject(location, GetParagraphExclusionsRequest.class);
        return service.getParagraphExclusions(request);
    }

    @Test(expected = SOAPFaultException.class)
    public void getParagraphIncludedSpecialitiesXmlNoDataFound() throws Exception {
        getParagraphIncludedSpecialities("/examples/GetParagraphIncludedSpecialties-NoDataFound.xml");
    }

    @Test
    public void getParagraphIncludedSpecialitiesXml() throws Exception {
        GetParagraphIncludedSpecialtiesResponse response = getParagraphIncludedSpecialities("/examples/GetParagraphIncludedSpecialties.xml");
        Assert.assertTrue("No atm and childrens found", response.getAtmAndChildrens().size() >= 1);
    }

    private GetParagraphIncludedSpecialtiesResponse getParagraphIncludedSpecialities(String location) throws Exception {
        GetParagraphIncludedSpecialtiesRequest request = transformLocationToObject(location, GetParagraphIncludedSpecialtiesRequest.class);
        return CivicsSessionServiceFactory.getCivicsSession().getParagraphIncludedSpecialities(request);
    }

    @Test(expected = SOAPFaultException.class)
    public void getProfessionalAuthorizationsXmlNoDataFound() throws Exception {
        getProfessionalAuthorizations("/examples/GetProfessionalAuthorizations-NoDataFound.xml");
    }

    @Test
    public void getProfessionalAuthorizationsXml() throws Exception {
        GetProfessionalAuthorizationsResponse response = getProfessionalAuthorizations("/examples/GetProfessionalAuthorizations.xml");
        Assert.assertTrue("No professional authorizations found", response.getProfessionalAuthorizations().size() >= 1);
    }

    private GetProfessionalAuthorizationsResponse getProfessionalAuthorizations(String location) throws Exception {
        GetProfessionalAuthorizationsRequest request = transformLocationToObject(location, GetProfessionalAuthorizationsRequest.class);
        return CivicsSessionServiceFactory.getCivicsSession().getProfessionalAuthorizations(request);
    }
}