﻿/*
 * Copyright (c) eHealth
 */
using System;
using ikvm.extensions;
using NUnit.Framework;
using java.io;
using java.util;
using org.joda.time;
using be.ehealth.technicalconnector.config;
using be.ehealth.technicalconnector.exception;
using be.ehealth.technicalconnector.session;
using be.ehealth.technicalconnector.utils;
using be.fgov.ehealth.mediprima.core.v1;
using be.fgov.ehealth.mediprima.protocol.v1;
using be.fgov.ehealth.technicalconnector.tests.utils;
using be.fgov.ehealth.technicalconnector.tests.session;
using be.ehealth.technicalconnector.idgenerator;
using be.ehealth.businessconnector.mediprima.builder;

namespace be.ehealth.businessconnector.mediprima.session
{

/**
 * Integration tests for consultation mediprima. The scenarios numbering match those of the MediPrima specification.
 *
 * @author EHP
 */
public class MediprimaConsultationIntegrationTest {
		
		 
        [TestFixture]
        public class WithSoaErrorIntegrationTest {

            static object[] testCase =
        {
            new Object[]{"scenario01", "Invalid ssin", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("85000132773").withPeriod().build()},
            new Object[]{"scenario02", "Consult with physician and period", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("86081523513").withPeriod().build()},
            new Object[]{"scenario03", "Consultation date > today", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("86081523513").withReferenceDate(new org.joda.time.DateTime().plusDays(10)).build()},
            new Object[]{"scenario04", "Consultation date > today", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("86081523513").withReferenceDate(new org.joda.time.DateTime().minusMonths(3)).build()}
        };



            [SetUp]
            public static  void setupsession() {
                SessionInitializer.init(".\\be.ehealth.businessconnector.mediprima.test.properties", SupportedLanguages.NET.getAbbreviation());
            }

            [TearDown]
            public static void teardownSession()  {
                Session.getInstance().unloadSession();
            }

            [Test, TestCaseSource("testCase")]
            public void testScenario(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
               try {
                   MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);
                   Assert.Fail("should have thrown SoaErrorException");
               } catch (SoaErrorException e) {
                   ConsultCarmedInterventionResponse response = (ConsultCarmedInterventionResponse) e.getResponseTypeV2();
                   String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
                   MarshallerHelper helper = new MarshallerHelper(typeof(ConsultCarmedInterventionResponse), typeof(ConsultCarmedInterventionResponse));
                   XmlAsserter.assertSimilar(expected, helper.toString((ConsultCarmedInterventionResponse) response));
               }
            }
        }


        /**
         * Integration tests for consultation mediprima.
         *
         * @author EHP
         *
         */
        [TestFixture]
        public class WithoutSoaErrorIntegrationTest {

            static object[] testCase =
        {
            new Object[]{"scenario05", "Unknown ssin", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("86081523513").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario06", "Known ssin but dates nok", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("35061540420").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario07a", "Decision without required rights - missing coverage", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("35061540420").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario07b", "Decision without required rights - missing coverage", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("76091051964").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario08", "Decision without required rights - coverage exists but nok", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("61011060097").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario09", "Decision without required rights - coverage ok but care provider nok", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("74052543021").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario10", "Decision pending", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("11030441141").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario11", "Decision is active - full support", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("92010442932").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario12", "Decision is active - partial support", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("75052657066").withReferenceDate(new org.joda.time.DateTime()).build()}
        };



            [SetUp]
            public static  void setupsession() {
                SessionInitializer.init(".\\be.ehealth.businessconnector.mediprima.test.properties", SupportedLanguages.NET.getAbbreviation());
            }

            [TearDown]
            public static void teardownSession()  {
                Session.getInstance().unloadSession();
            }

            [Test, TestCaseSource("testCase")]
            public void testScenario(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
                   ConsultCarmedInterventionResponse response = MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);

            String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
            MarshallerHelper helper = new MarshallerHelper(typeof(ConsultCarmedInterventionResponse), typeof(ConsultCarmedInterventionResponse));
            XmlAsserter.assertSimilar(expected, helper.toString((ConsultCarmedInterventionResponse) response));
            }
        }

        private static String expectedResponse(org.joda.time.DateTime consultationDate, String requestId, String scenarioId) {
            Map velocityContext = new HashMap();
            velocityContext.put("requestId", requestId);
            velocityContext.put("consultationDate", DateUtils.printDate(consultationDate));
            String commentDate = org.joda.time.format.DateTimeFormat.forPattern("dd/MM/yyyy").print(new org.joda.time.DateTime());
            velocityContext.put("commentStartDate", commentDate);
            velocityContext.put("commentEndDate", commentDate);
            velocityContext.put("startDate", DateUtils.printDate(new org.joda.time.DateTime()));
            velocityContext.put("endDate", DateUtils.printDate(new org.joda.time.DateTime()));
            return TemplateEngineUtils.generate(velocityContext, ".\\consultation\\expected\\" + scenarioId + "Response.xml");
        }
    }
}