﻿/*
 * Copyright (c) eHealth
 */
using System;
using ikvm.extensions;
using NUnit.Framework;
using java.io;
using java.util;
using org.joda.time;
using be.ehealth.technicalconnector.config;
using be.ehealth.technicalconnector.exception;
using be.ehealth.technicalconnector.session;
using be.ehealth.technicalconnector.utils;
using be.fgov.ehealth.mediprima.core.v2;
using be.fgov.ehealth.mediprima.protocol.v2;
using be.fgov.ehealth.businessconnector.tests.utils;
using be.fgov.ehealth.technicalconnector.tests.utils;
using be.fgov.ehealth.technicalconnector.tests.session;
using be.ehealth.technicalconnector.idgenerator;
using be.ehealth.businessconnector.mediprimav2.builder;

namespace be.ehealth.businessconnector.mediprimav2.session
{

/**
 * Integration tests for consultation mediprima. The scenarios numbering match those of the MediPrima specification.
 *
 * @author EHP
 */
public class MediprimaConsultationIntegrationTest {
		
		 
        [TestFixture]
        public class WithSoaErrorIntegrationTest {

            static object[] testCase =
        {
            new Object[]{"scenario01", "SSIN invalid", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("85000132773").withPeriod().build()},
            new Object[]{"scenario02", "call doctor on period", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("70410100375").withPeriod().build()},
            new Object[]{"scenario03", "Consultation with date > today", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("70410100375").withReferenceDate(new org.joda.time.DateTime().plusDays(10)).build()},
            new Object[]{"scenario04", "Consultation with date < today - 2 months", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("70410100375").withReferenceDate(new org.joda.time.DateTime().minusMonths(3)).build()}
        };



            [SetUp]
            public static  void setupsession() {
                SessionInitializer.init(".\\be.ehealth.businessconnector.mediprima.v2.test.properties", SupportedLanguages.NET.getAbbreviation());
            }

            [TearDown]
            public static void teardownSession()  {
                Session.getInstance().unloadSession();
            }

            [Test, TestCaseSource("testCase")]
            public void testScenario(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
               try {
                   MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);
                   Assert.Fail("should have thrown SoaErrorException");
               } catch (SoaErrorException e) {
                   ConsultCarmedInterventionResponse response = (ConsultCarmedInterventionResponse) e.getResponseTypeV2();
                   String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
                   MarshallerHelper helper = new MarshallerHelper(typeof(ConsultCarmedInterventionResponse), typeof(ConsultCarmedInterventionResponse));
                   be.fgov.ehealth.businessconnector.tests.utils.XmlAsserter.assertSimilar(expected, helper.toString((ConsultCarmedInterventionResponse) response));
               }
            }
        }


        /**
         * Integration tests for consultation mediprima.
         *
         * @author EHP
         *
         */
        [TestFixture]
        public class WithoutSoaErrorIntegrationTest {

            static object[] testCase =
        {
            new Object[]{"scenario05", "No decision (SSIN unknown)", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("70410100375").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario06", "No more cart (SSIN known, dates NOK)", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("60061861748").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario07", "Decision without rights, reason no cover", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("68050555876").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario08", "Decision without rights, reason cover present but nok", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("95090238957").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario09", "Decision without right, reason cover ok but recipient nok", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("88081964510").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario10", "Decision ok, suspended, (total support SPP)", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("93122961468").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario11", "Decision ok, active, SPP full support", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("94000087334").withReferenceDate(new org.joda.time.DateTime()).build()},
            new Object[]{"scenario12", "Decision ok, active, SPP partial support", ConsultCarmedInterventionRequestBuilder.aRequest().withSsin("74032853407").withReferenceDate(new org.joda.time.DateTime()).build()}
        };



            [SetUp]
            public static  void setupsession() {
                SessionInitializer.init(".\\be.ehealth.businessconnector.mediprima.v2.test.properties", SupportedLanguages.NET.getAbbreviation());
            }

            [TearDown]
            public static void teardownSession()  {
                Session.getInstance().unloadSession();
            }

            [Test, TestCaseSource("testCase")]
            public void testScenario(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
                   ConsultCarmedInterventionResponse response = MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);

            String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
            MarshallerHelper helper = new MarshallerHelper(typeof(ConsultCarmedInterventionResponse), typeof(ConsultCarmedInterventionResponse));
            be.fgov.ehealth.businessconnector.tests.utils.XmlAsserter.assertSimilar(expected, helper.toString((ConsultCarmedInterventionResponse) response));
            }
        }

        private static String expectedResponse(org.joda.time.DateTime consultationDate, String requestId, String scenarioId) {
            Map velocityContext = new HashMap();
            velocityContext.put("requestId", requestId);
            velocityContext.put("consultationDate", DateUtils.printDate(consultationDate));
            String commentDate = org.joda.time.format.DateTimeFormat.forPattern("dd/MM/yyyy").print(new org.joda.time.DateTime());
            velocityContext.put("commentStartDate", commentDate);
            velocityContext.put("commentEndDate", commentDate);
            velocityContext.put("startDate", DateUtils.printDate(new org.joda.time.DateTime()));
            velocityContext.put("endDate", DateUtils.printDate(new org.joda.time.DateTime()));
            return TemplateEngineUtils.generate(velocityContext, ".\\examples\\mediprimav2\\consultation\\expected\\persphysician\\" + scenarioId + "Response.xml");
        }
    }
}