/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.mycarenet.attestv2.session;

import static org.junit.Assert.assertEquals;

import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;

import be.ehealth.businessconnector.mycarenet.attestv2.domain.CancelAttestBuilderRequest;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.InputReference;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SendAttestBuilderRequest;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SignedBuilderResponse;
import be.ehealth.businessconnector.mycarenet.attestv2.helper.AttestIntegrationTestHelper;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.CancelAttestationResponse;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.SendAttestationResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * eAttest v2 Integration Test for cancel with profile dentist. The scenarios numbering matches one specified by mycarenet.
 *
 * @author EHP
 */
public class CancelAttestDentistIntegrationTest {

    private static final String NISS1_DENT_OA500 = "62110906574";

    private static final String NISS1_DENT_OA100 = "73052005540";

    private static final String OA_100_ACK = "100-1-190226-9851229-88";

    private static final String OA_500_ACK = "500-1-190110-4567890-16";

    private static final String DENT = "DENT";

    private static final String OA_100 = "100";

    private static final String OA_500 = "500";
    
    @ClassRule
    public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("persdentist").forUser("maxime").build();

    private AttestIntegrationTestHelper helper = new AttestIntegrationTestHelper();


    @Test
    public void cancelAttestationOA100DENT8a() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8a";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_DENT_OA100, DENT, OA_100_ACK);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_100, DENT, response, attestBuilderRequest);
    }


    @Test
    public void cancelAttestationOA500DENT8a() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8a";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_DENT_OA500, DENT, OA_500_ACK);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_500, DENT, response, attestBuilderRequest);
    }


    @Test
    public void cancelAttestationOA100DENT8b() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8b";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_DENT_OA100, DENT, OA_100_ACK);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_100, DENT, response, attestBuilderRequest);
    }

    @Test
    public void cancelAttestationOA500DENT8b() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8b";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_DENT_OA500, DENT, OA_500_ACK);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_500, DENT, response, attestBuilderRequest);
    }


    @Test
    public void cancelAttestationOA100DENT8c() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8c";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_DENT_OA100, DENT, OA_100_ACK);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_100, DENT, response, attestBuilderRequest);
    }

    @Test
    public void cancelAttestationOA500DENT8c() throws Exception {
        InputReference inputReference = new InputReference();
        String scenarioName = "DENT8c";

        CancelAttestBuilderRequest attestBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_DENT_OA500, DENT, OA_500_ACK);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(attestBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_500, DENT, response, attestBuilderRequest);
    }

    @Test
    public void cancelAttestationOA100DENT9() throws Exception {
        InputReference inputReference = new InputReference();
        SendAttestBuilderRequest sendBuilderRequest = helper.buildRequest(inputReference, "DENT4", NISS1_DENT_OA100, DENT);
        SendAttestationResponse sendResponse = AttestSessionServiceFactory.getAttestService()
                .sendAttestation(sendBuilderRequest.getSendAttestationRequest());
        String invoiceNumber = helper.getInvoiceNumber(sendResponse, sendBuilderRequest);
        String scenarioName = "DENT9";

        CancelAttestBuilderRequest cancelBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_DENT_OA100, DENT, invoiceNumber);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(cancelBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_100, DENT, response, cancelBuilderRequest);
    }

    @Test
    public void cancelAttestationOA500DENT9() throws Exception {
        InputReference inputReference = new InputReference();
        SendAttestBuilderRequest sendBuilderRequest = helper.buildRequest(inputReference, "DENT9Send", NISS1_DENT_OA500, DENT);
        SendAttestationResponse sendResponse = AttestSessionServiceFactory.getAttestService()
                .sendAttestation(sendBuilderRequest.getSendAttestationRequest());
        String invoiceNumber = helper.getInvoiceNumber(sendResponse, sendBuilderRequest);
        String scenarioName = "DENT9";

        CancelAttestBuilderRequest cancelBuilderRequest = helper.buildRequest(new InputReference(), scenarioName, NISS1_DENT_OA500, DENT, invoiceNumber);
        CancelAttestationResponse response = AttestSessionServiceFactory.getAttestService().cancelAttestation(cancelBuilderRequest.getCancelAttestationRequest());

        assertCancelResult(inputReference, scenarioName, OA_500, DENT, response, cancelBuilderRequest);
    }

    private void assertCancelResult(InputReference inputReference, String scenarioName, String oa, String profession,
                                    CancelAttestationResponse response, CancelAttestBuilderRequest cancelAttestBuilderRequest) throws Exception {
        SignedBuilderResponse builderResponse = helper.getBuildCancelAttestResponse(response, cancelAttestBuilderRequest.getCancelAttestationRequest());
        assertEquals("SignatureVerificationResult should contain no error", 0, builderResponse.getSignatureVerificationResult().getErrors().size());
        // check signedData is same as content of the business blob in the WS reponse
        Assert.assertEquals(response.getReturn().getDetail().getValue(), builderResponse.getSignedData());
        // check xadesT is same as the one from the WS reponse
        Assert.assertEquals(response.getReturn().getXadesT().getValue(), builderResponse.getXadesT());
        String expectedResponse = helper.buildExpectedCancelAttestResponse(builderResponse, inputReference.getInputReference(), scenarioName, oa, profession);
        XmlAsserter.assertSimilar(expectedResponse, ConnectorXmlUtils.toString(builderResponse.getSendTransactionResponse()));
    }
}
