/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.mycarenet.attestv2.helper;

import java.io.UnsupportedEncodingException;
import java.util.HashMap;
import java.util.Map;

import javax.xml.bind.JAXBException;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.transform.TransformerException;

import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;

import be.ehealth.business.mycarenetdomaincommons.domain.Ssin;
import be.ehealth.businessconnector.mycarenet.attestv2.builders.RequestObjectBuilder;
import be.ehealth.businessconnector.mycarenet.attestv2.builders.RequestObjectBuilderFactory;
import be.ehealth.businessconnector.mycarenet.attestv2.builders.ResponseObjectBuilder;
import be.ehealth.businessconnector.mycarenet.attestv2.builders.ResponseObjectBuilderFactory;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.CancelAttestBuilderRequest;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.InputReference;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SendAttestBuilderRequest;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SignedBuilderResponse;
import be.ehealth.businessconnector.mycarenet.attestv2.domain.SignedEncryptedBuilderResponse;
import be.fgov.ehealth.standards.kmehr.mycarenet.schema.v1.FolderType;
import be.fgov.ehealth.standards.kmehr.mycarenet.schema.v1.Kmehrmessage;
import be.fgov.ehealth.standards.kmehr.mycarenet.schema.v1.TransactionType;
import be.ehealth.technicalconnector.adapter.XmlTimeAdapter;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.utils.ConnectorXmlUtils;
import be.ehealth.technicalconnector.utils.DateUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.ehealth.technicalconnector.utils.TemplateEngineUtils;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.CancelAttestationRequest;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.CancelAttestationResponse;
import be.fgov.ehealth.mycarenet.attest.protocol.v2.SendAttestationResponse;


/**
 * Utilities class needed for cancel and send for eAttest v2 integration tests
 *
 * @author EHP
 */
public class AttestIntegrationTestHelper {


    private DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();


    public SendAttestBuilderRequest buildRequest(InputReference inputReference, String scenarioName, String ssin, String profession) throws ConnectorException, UnsupportedEncodingException, TransformerException, JAXBException, InstantiationException, IllegalAccessException {
        Ssin ssinValue = new Ssin(ssin);
        DateTime dateReference = new DateTime();
        RequestObjectBuilder builder = RequestObjectBuilderFactory.getRequestObjectBuilder();
        Kmehrmessage kmehrRequest = buildKmehrmessage(scenarioName, ssin, profession);
        SendAttestBuilderRequest attestBuilderRequest = builder.buildSendAttestationRequest(true, inputReference, ssinValue, dateReference, kmehrRequest);
        ConnectorXmlUtils.dump(attestBuilderRequest.getSendTransactionRequest());

        return attestBuilderRequest;

    }

    public CancelAttestBuilderRequest buildRequest(InputReference inputReference, String scenarioName, String ssin, String profession, String invoiceNumber) throws ConnectorException {
        Ssin ssinValue = new Ssin(ssin);
        DateTime dateReference = new DateTime();
        RequestObjectBuilder builder = RequestObjectBuilderFactory.getRequestObjectBuilder();
        Kmehrmessage kmehrRequest = buildKmehrmessage(scenarioName, ssin, profession, invoiceNumber);
        CancelAttestBuilderRequest attestBuilderRequest = builder.buildCancelAttestationRequest(true, inputReference, ssinValue, dateReference, kmehrRequest);
        ConnectorXmlUtils.dump(attestBuilderRequest.getCancelAttestationRequest());

        return attestBuilderRequest;

    }

    private Kmehrmessage buildKmehrmessage(String scenarioName, String ssin, String profession) {
        Map<String, Object> velocityContext = fillContext(ssin);
        return getKmehrMessage(scenarioName, profession, velocityContext);
    }

    private Kmehrmessage buildKmehrmessage(String scenarioName, String ssin, String profession, String invoiceNumber) {
        Map<String, Object> velocityContext = fillContext(ssin);
        velocityContext.put("invoicingNumber", invoiceNumber);

        return getKmehrMessage(scenarioName, profession, velocityContext);
    }

    private Map<String, Object> fillContext(String ssin) {
        Map<String, Object> velocityContext = new HashMap<String, Object>();
        velocityContext.put("today", DateUtils.printDateTime(new DateTime()));
        velocityContext.put("yesterday", DateUtils.printDateTime(new DateTime().minusDays(1)));
        velocityContext.put("ssin", ssin);
        velocityContext.put("patient", ssin);
        return velocityContext;
    }

    private Kmehrmessage getKmehrMessage(String scenarioName, String profession, Map<String, Object> velocityContext) {
        String kmerh = TemplateEngineUtils.generate(velocityContext, "/scenarios/requests/" + profession + "/" + scenarioName + "_request" + ".xml");

        MarshallerHelper<Kmehrmessage, Kmehrmessage> kmehrRequestMarshaller = new MarshallerHelper<Kmehrmessage, Kmehrmessage>(Kmehrmessage.class, Kmehrmessage.class);
        return kmehrRequestMarshaller.toObject(kmerh);
    }

    public String buildExpectedSendAttestResponse(SignedEncryptedBuilderResponse builderResponse, String inputReference, String profession, String oa, String scenarioName) throws Exception {
        Map<String, Object> velocityContext = new HashMap<String, Object>();
        velocityContext.put("requestId", "18334780004" + "." + inputReference);
        replaceDateTimes(builderResponse, velocityContext);
        return TemplateEngineUtils.generate(velocityContext, "/scenarios/expected/" + profession + "/" + oa + "/" + scenarioName + "_response" + ".xml");
    }

    public String buildExpectedCancelAttestResponse(SignedBuilderResponse builderResponse, String inputReference, String scenarioName, String oa, String profession) throws Exception {
        Map<String, Object> velocityContext = new HashMap<String, Object>();
        velocityContext.put("requestId", "18334780004" + "." + inputReference);
        replaceDateTimes(builderResponse, velocityContext);
        return TemplateEngineUtils.generate(velocityContext, "/scenarios/expected/" + profession + "/" + oa + "/" + scenarioName + "_response" + ".xml");
    }

    private void replaceDateTimes(SignedBuilderResponse builderResponse, Map<String, Object> velocityContext) throws Exception {
        velocityContext.put("today", DateTimeFormat.forPattern("yyyy-MM-dd").print(new DateTime()));
        velocityContext.put("responseTime", new XmlTimeAdapter().marshal(builderResponse.getSendTransactionResponse().getResponse().getTime()));
        velocityContext.put("requestTime", new XmlTimeAdapter().marshal(builderResponse.getSendTransactionResponse().getResponse().getRequest().getTime()));

        if (builderResponse.getSendTransactionResponse().getKmehrmessage() != null) {
            velocityContext.put("headerTime", new XmlTimeAdapter().marshal(builderResponse.getSendTransactionResponse().getKmehrmessage().getHeader().getTime()));
            for (FolderType folder : builderResponse.getSendTransactionResponse().getKmehrmessage().getFolders()) {
                for (TransactionType transaction : folder.getTransactions()) {
                    velocityContext.put("transactionTime" + transaction.getIds().get(0).getValue(), new XmlTimeAdapter().marshal(transaction.getTime()));
                }
            }
        }
    }

    private void replaceDateTimes(SignedEncryptedBuilderResponse builderResponse, Map<String, Object> velocityContext) throws Exception {
        velocityContext.put("today", DateTimeFormat.forPattern("yyyy-MM-dd").print(new DateTime()));
        velocityContext.put("yesterday", DateTimeFormat.forPattern("yyyy-MM-dd").print(new DateTime().minusDays(1)));
        velocityContext.put("responseTime", new XmlTimeAdapter().marshal(builderResponse.getSendTransactionResponse().getResponse().getTime()));
        velocityContext.put("requestTime", new XmlTimeAdapter().marshal(builderResponse.getSendTransactionResponse().getResponse().getRequest().getTime()));

        if (builderResponse.getSendTransactionResponse().getKmehrmessage() != null) {
            velocityContext.put("headerTime", new XmlTimeAdapter().marshal(builderResponse.getSendTransactionResponse().getKmehrmessage().getHeader().getTime()));
            for (FolderType folder : builderResponse.getSendTransactionResponse().getKmehrmessage().getFolders()) {
                for (TransactionType transaction : folder.getTransactions()) {
                    velocityContext.put("transactionTime" + transaction.getIds().get(0).getValue(), new XmlTimeAdapter().marshal(transaction.getTime()));
                }
            }
        }
    }

    public String getInvoiceNumber(SendAttestationResponse sendResponse, SendAttestBuilderRequest builderRequest) throws ConnectorException {
        ConnectorXmlUtils.dump(sendResponse);
        ResponseObjectBuilder responseBuilder = ResponseObjectBuilderFactory.getResponseObjectBuilder();

        SignedEncryptedBuilderResponse builderCancelResponse = responseBuilder.handleSendResponseType(sendResponse, builderRequest);

        return builderCancelResponse.getSendTransactionResponse().getKmehrmessage().getFolders().get(0).getTransactions().get(0).getItem().get(0).getContents().get(0).getTexts().get(0).getValue();
    }
    

    public SignedBuilderResponse getBuildCancelAttestResponse(CancelAttestationResponse response, CancelAttestationRequest request) throws TechnicalConnectorException {
        ConnectorXmlUtils.dump(response);
        ResponseObjectBuilder responseBuilder = ResponseObjectBuilderFactory.getResponseObjectBuilder();

        return responseBuilder.handleCancelResponseType(response, request);
    }

    public SignedEncryptedBuilderResponse getBuildSendAttestResponse(SendAttestationResponse response, SendAttestBuilderRequest builderRequest) throws TechnicalConnectorException {
        ConnectorXmlUtils.dump(response);
        ResponseObjectBuilder responseBuilder = ResponseObjectBuilderFactory.getResponseObjectBuilder();

        return responseBuilder.handleSendResponseType(response, builderRequest);
    }
}
