/*
 * Copyright (c) eHealth
 */
package be.ehealth.businessconnector.mediprimav2.session;


import static be.ehealth.businessconnector.mediprimav2.builder.ConsultCarmedInterventionRequestBuilder.aRequest;

import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.Map;

import org.joda.time.DateTime;
import org.junit.Assert;
import org.junit.ClassRule;
import org.junit.Test;
import org.junit.experimental.runners.Enclosed;
import org.junit.runner.RunWith;
import org.junit.runners.Parameterized;

import be.ehealth.technicalconnector.exception.SoaErrorException;
import be.ehealth.technicalconnector.utils.DateUtils;
import be.ehealth.technicalconnector.utils.MarshallerHelper;
import be.ehealth.technicalconnector.utils.TemplateEngineUtils;
import be.fgov.ehealth.mediprima.protocol.v2.ConsultCarmedInterventionRequest;
import be.fgov.ehealth.mediprima.protocol.v2.ConsultCarmedInterventionResponse;
import be.fgov.ehealth.technicalconnector.tests.junit.rule.SessionRule;
import be.fgov.ehealth.technicalconnector.tests.utils.XmlAsserter;

/**
 * Integration tests for consultation mediprima v2 for orgpharmacy. The scenarios numbering match those of the MediPrima specification.
 *
 *  /!\ The configuration must have this Mediprima Consult specific saml attribute property:
 * sessionmanager.samlattributedesignator.8=urn:be:fgov:certified-namespace:ehealth,urn:be:fgov:ehealth:1.0:pharmacy:nihii-number:recognisedpharmacy:nihii11
 * sessionmanager.samlattributedesignator.9=urn:be:fgov:person:ssin:ehealth:1.0:pharmacist:boolean,urn:be:fgov:ehealth:1.0:certificateholder:person:ssin:usersession:boolean

 *
 * @author EHP
 */
@RunWith(Enclosed.class)
public final class MediprimaConsultationOrgpharmacyIntegrationTest {

    @RunWith(Parameterized.class)
    public static class WithSoaErrorIntegrationTest {

        private static final String NISS_10 = "85000132773";


        private static final String NISS_01 = "70410100375";


        @ClassRule
        public static SessionRule rule = SessionRule.withActiveSession().forEnvironment("acc").forProfession("orgpharmacy").forUser("maxime").build();

        private String scenarioId;
        private String scenarioName;
        private ConsultCarmedInterventionRequest request;


        public WithSoaErrorIntegrationTest(String scenarioId, String scenarioName, ConsultCarmedInterventionRequest request) {
            super();
            this.scenarioId = scenarioId;
            this.scenarioName = scenarioName;
            this.request = request;
        }

        @Parameterized.Parameters(name = "{0}: {1}")
        public static Collection<Object[]> data() throws Exception {
            Collection<Object[]> testData = new ArrayList<Object[]>();
            testData.add(new Object[]{"scenario01", "Invalid ssin", aRequest().withSsin(NISS_10).withPeriod().build()});
            testData.add(new Object[]{"scenario02", "call pharmacist on period", aRequest().withSsin(NISS_01).withPeriod().build()});
            testData.add(new Object[]{"scenario03", "Date requested is in the future (not allowed)", aRequest().withSsin(NISS_01).withReferenceDate(new DateTime().plusDays(10)).build()});
            testData.add(new Object[]{"scenario04", "Date requested is too much in the past (not allowed)", aRequest().withSsin(NISS_01).withReferenceDate(new DateTime().minusYears(2)).build()});

            return testData;
        }

        @Test
        public void testScenario() throws Exception {
            try {
                MediprimaSessionServiceFactory.getConsultationMediprimaSession().consultCarmedIntervention(request);
                Assert.fail("should have thrown SoaErrorException");
            } catch (SoaErrorException e) {
                ConsultCarmedInterventionResponse response = (ConsultCarmedInterventionResponse) e.getResponseTypeV2();
                String expected = expectedResponse(request.getSelectionCriteria().getBySsin().getReferenceDate(), request.getId(), scenarioId);
                MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse> helper = new MarshallerHelper<ConsultCarmedInterventionResponse, ConsultCarmedInterventionResponse>(ConsultCarmedInterventionResponse.class, ConsultCarmedInterventionResponse.class);
                XmlAsserter.assertSimilar(expected, helper.toString(response));
            }
        }
    }




    private static String expectedResponse(DateTime consultationDate, String requestId, String scenarioId) {
        Map<String, Object> templateContext = new HashMap<String, Object>();
        templateContext.put("requestId", requestId);
        templateContext.put("consultationDate", DateUtils.printDate(consultationDate));
        templateContext.put("startDate", DateUtils.printDate(new DateTime()));
        templateContext.put("endDate", DateUtils.printDate(new DateTime()));
        return TemplateEngineUtils.generate(templateContext, "/examples/mediprimav2/consultation/expected/orgpharmacy/" + scenarioId + "Response.xml");
    }
}
