﻿/*
 * Copyright (c) eHealth
 */
using System;
using ikvm.extensions;
using NUnit.Framework;
using java.io;
using java.util;

using org.joda.time;

using be.fgov.ehealth.technicalconnector.tests.utils;

using be.ehealth.business.mycarenetdomaincommons.builders;
using be.ehealth.business.mycarenetdomaincommons.domain;
using be.ehealth.business.mycarenetcommons.mapper;
using be.ehealth.businessconnector.tarification.builder;
using be.ehealth.technicalconnector.config;
using be.ehealth.technicalconnector.exception;
using be.ehealth.technicalconnector.session;
using be.ehealth.technicalconnector.utils;
using be.fgov.ehealth.messageservices.core.v1;
using be.fgov.ehealth.mycarenet.commons.core.v2;
using be.fgov.ehealth.mycarenet.commons.protocol.v2;
using be.fgov.ehealth.standards.kmehr.cd.v1;
using be.fgov.ehealth.standards.kmehr.schema.v1;
using be.fgov.ehealth.technicalconnector.tests.session;
using be.ehealth.technicalconnector.idgenerator;
using be.ehealth.business.kmehrcommons;
using be.ehealth.businessconnector.tarification.helper;

namespace be.ehealth.businessconnector.mediprima.session
{


/**
 * Integration tests for tarification mediprima. The scenarios numbering match those of the MediPrima specification.
 *
 * @author EHP
 */
[TestFixture]
public class MediprimaTarificationIntegrationTest {

    private static HashMap nissProvider = new HashMap();

    private static HashMap nissMatchScenario = new HashMap();

    static object[] testCase =
{
    new Object[] {"scenario01", "Unexisting nomenclature code", "28033135008", new org.joda.time.DateTime(), Arrays.asList("101075"), new ExpectedResult("130")},
    new Object[] {"scenario02", "Non general practitioner nomenclature code", "28033135008", new org.joda.time.DateTime(), Arrays.asList("101793"), new ExpectedResult("171")},
    new Object[] {"scenario03", "Medical specialist nomenclature code", "92010442932", new org.joda.time.DateTime(), Arrays.asList("102034"), new ExpectedResult("171")},
    new Object[] {"scenario04", "Consultation date < today - 2 months", "92010442932",  new org.joda.time.DateTime().minusMonths(3), Arrays.asList("101032"), new ExpectedResult("166")},
    new Object[] {"scenario05", "Consultation date > today", "28033135008", new org.joda.time.DateTime().plusDays(2), Arrays.asList("101032"), new ExpectedResult("146")},
    new Object[] {"scenario06", "No decision", "28033135008", new org.joda.time.DateTime(), Arrays.asList("101032"), new ExpectedResult("202")},
    new Object[] {"scenario07", "No decision for date", "35061540420", new org.joda.time.DateTime(), Arrays.asList("101032"), new ExpectedResult("201")},
    new Object[] {"scenario08", "Decision without required rights - missing coverage", "55051059009", new org.joda.time.DateTime(), Arrays.asList("101010"), new ExpectedResult("207")},
    new Object[] {"scenario09", "Decision without required rights - missing coverage with type physician", "76091051964", new org.joda.time.DateTime(), Arrays.asList("101032"), new ExpectedResult("207")},
    new Object[] {"scenario10", "Decision without required rights - coverage exists but nok", "61011060097", new org.joda.time.DateTime(), Arrays.asList("101032"), new ExpectedResult("206")},
    new Object[] {"scenario11", "Decision without required rights - coverage ok but care provider nok", "74052543021", new org.joda.time.DateTime(), Arrays.asList("101032"), new ExpectedResult("204")},
    new Object[] {"scenario12", "Decision pending", "11030441141", new org.joda.time.DateTime(), Arrays.asList("101032"), new ExpectedResult("203")},
    new Object[] {"scenario13", "Decision is active - full support", "92010442932", new org.joda.time.DateTime(), Arrays.asList("101032"), new ExpectedResult(null)},
    new Object[] {"scenario14", "Decision is active - partial support", "75052657066", new org.joda.time.DateTime(), Arrays.asList("101032"), new ExpectedResult(null)},
    new Object[] {"scenario15", "Decision is active - full support - multiple codes provided", "92010442932", new org.joda.time.DateTime(), Arrays.asList("101032", "475075"), new ExpectedResult(null)}

};



    [SetUp]
    public static  void setupsession() {
        SessionInitializer.init(".\\be.ehealth.businessconnector.mediprima.test.properties", SupportedLanguages.NET.getAbbreviation());
    }

    [TearDown]
    public static void teardownSession()  {
        Session.getInstance().unloadSession();
    }

    [Test, TestCaseSource("testCase")]
    public void testScenario(String scenarioId, String scenarioName, String niss, org.joda.time.DateTime dateTime, java.util.List claims, ExpectedResult expectedResult) {
        TarificationConsultationRequest request = givenRequestWith(scenarioId, niss, dateTime, claims);

        byte[] response = whenConsultingTarification(request);

        then(scenarioId, expectedResult, response);
    }


    private TarificationConsultationRequest givenRequestWith(String scenarioId, String niss, org.joda.time.DateTime dateTime, java.util.List claims) {
        RequestBuilder requestBuilder = TarificationRequestBuilderFactory.getRequestObjectBuilder();
        InputStream businessXmlContentStream = buildRequest(claims);
        XmlTestRequest xmlTestRequest = replaceVariableTagsWithContent(niss, dateTime, claims, System.Text.Encoding.UTF8.GetString(ConnectorIOUtils.getBytes(businessXmlContentStream)));
        String businessXml = xmlTestRequest.textContent;
        String inputReference = IdGeneratorFactory.getIdGenerator("xsid").generateId();
        TarificationConsultationRequest request = requestBuilder.buildConsultationRequest(getRoutingForNiss(niss), businessXml.getBytes(), inputReference);
        request.getCommonInput().setInputReference(xmlTestRequest.kmehrId);
        request.getRouting().setReferenceDate(xmlTestRequest.time);
        return request;
        }

    private byte[] whenConsultingTarification(TarificationConsultationRequest request) {
        MediprimaTarificationSessionService session = MediprimaSessionServiceFactory.getTarificationMediprimaSession();
        TarificationConsultationResponse consultTarificationResponse = session.consultTarification(request);
        BlobType detail = consultTarificationResponse.getReturn().getDetail();
        Blob blob = SendRequestMapper.mapBlobTypeToBlob(detail);
        byte[] content = BlobBuilderFactory.getBlobBuilder("mediprima.tarification").checkAndRetrieveContent(blob);
        String responseContent = System.Text.Encoding.UTF8.GetString(content);

        return content;
    }

    private void then(String scenarioId, ExpectedResult expectedResult, byte[] content) {
        if (expectedResult.errorCode != null) {
            assertErrorCode(expectedResult, content);
        } else {
            XmlAsserter.assertSimilar(ConnectorIOUtils.getResourceAsString(".\\tarification\\expected\\" + scenarioId + "Response.xml"), System.Text.Encoding.UTF8.GetString(content));
        }
    }

    private void assertErrorCode(ExpectedResult expectedResult, byte[] content) {
        MarshallerHelper helper = new MarshallerHelper(typeof(RetrieveTransactionResponse), typeof(RetrieveTransactionResponse));
        RetrieveTransactionResponse commonInputResponse = (RetrieveTransactionResponse) helper.toObject(content);
        ResponseHelper.validateResponse(commonInputResponse);
        Boolean flagFound = false;
        java.util.List errors = commonInputResponse.getAcknowledge().getErrors();
        for (int i =0;i<errors.size();i++) {
            ErrorMyCarenetType error = (ErrorMyCarenetType)errors.get(i);
            java.util.List cds = error.getCds();
            for (int j=0;j < cds.size();j++) {
                CDERRORMYCARENET code = (CDERRORMYCARENET)cds.get(j);
                flagFound = true;
            }
        }
        if (!flagFound) {
            Assert.Fail("Wait for error code " + expectedResult.errorCode + " but not found in response");
        }
    }

    private InputStream buildRequest(java.util.List claims) {
        return ConnectorIOUtils.getResourceAsStream(".\\tarification\\request\\" + (claims.size() > 1 ? "multipleCodesRequestTemplate.xml" : "requestTemplate.xml"));
    }

    private XmlTestRequest replaceVariableTagsWithContent(String niss, org.joda.time.DateTime dateTime, java.util.List claims, String testFileContent) {
        String khmerId = dateTime.toString("YYYYddhhmmssSS");
        testFileContent = testFileContent.replaceAll("\\$\\{currentTimestamp\\}", khmerId);
        String date = dateTime.toString("YYYY-MM-dd");
        testFileContent = testFileContent.replaceAll("\\$\\{currentDate\\}", date);
        String time = dateTime.toString("hh:mm:ss");
        testFileContent = testFileContent.replaceAll("\\$\\{currentTime\\}", time);
        String userNihiiNumber = retrieveCareProviderNihii();
        testFileContent = testFileContent.replaceAll("\\$\\{nihiiNumber\\}", userNihiiNumber);
        testFileContent = testFileContent.replaceAll("\\$\\{lastName\\}", SessionUtil.getLastname());
        testFileContent = testFileContent.replaceAll("\\$\\{firstName\\}", SessionUtil.getFirstname());
        String nissNumber = retrieveCareProviderNiss();
        testFileContent = testFileContent.replaceAll("\\$\\{niss\\}", nissNumber);
        testFileContent = testFileContent.replaceAll("\\$\\{professionType\\}", HcPartyUtil.getAuthorKmehrQuality());
        testFileContent = testFileContent.replaceAll("\\$\\{patientNiss\\}", niss);
        Iterator it = claims.iterator();
        for (int i = 0; it.hasNext(); i++) {
        	testFileContent = testFileContent.replaceAll("\\$\\{claim" + i +"\\}", (String) it.next());
        }

        return new XmlTestRequest(testFileContent, khmerId, dateTime);
    }


    private String retrieveCareProviderNiss() {
        return ConfigFactory.getConfigValidator().getProperty("user.inss");
    }


    private String retrieveCareProviderNihii() {
        return ConfigFactory.getConfigValidator().getProperty("user.nihii");
    }

    private static Routing getRoutingForNiss(String niss) {
        CareReceiverId careReceiver = new CareReceiverId(niss);
        return new Routing(careReceiver, new org.joda.time.DateTime());
    }

    private class XmlTestRequest {
        public String textContent;
        public String kmehrId;
        public org.joda.time.DateTime time;

        public XmlTestRequest(String textContent, String kmehrId, org.joda.time.DateTime time) {
            this.textContent = textContent;
            this.kmehrId = kmehrId;
            this.time = time;
        }
    }

    public class ExpectedResult {
        public String errorCode;

        public ExpectedResult(String errorCode) {
            this.errorCode = errorCode;
        }
    }
}
}
