/*
 * Copyright (c) eHealth
 */
using System;
using be.ehealth.businessconnector.recipe.session;
using be.fgov.ehealth.technicalconnector.tests.session;
using be.ehealth.technicalconnector.utils;
using be.recipe.services;
using NUnit.Framework;
using be.ehealth.technicalconnector.service.sts.security;
using be.ehealth.technicalconnector.service.sts.security.impl;
using be.ehealth.technicalconnector.utils;
using be.ehealth.technicalconnector.config;
using be.ehealth.technicalconnector.service.sts.utils;
using be.ehealth.technicalconnector.session;
using be.fgov.ehealth.technicalconnector.tests.utils;
using java.io;
using java.util;

namespace be.ehealth.businessconnector.recipe.service
{
	public class RecipePrescriberServiceIntegrationTest {
	
	    private static Configuration conf;
	    
	    private static String PRESCRIPTION_TYPE_ID = "P0";
	
	    private static String TEST_PATIENT_INSS = "test.recipe.patient.inss";
	
	    private static String TEST_NIHII_PHARMACY = "test.recipe.nihii.pharmacy";
	
	    private static String TEST_SAMPLE_PRESCRIPTION = "test.recipe.sample.prescription";
	
	    private static String TEST_SAMPLE_NOTIFICATION = "test.recipe.sample.notification";
	
	    private static Properties props = null;
	
	    /** The prescription id. */
	    private static String prescriptionId;
	    
	    
	    [TearDown]
	    public static void tearDown() {
	    	Session.getInstance().unloadSession();
	    }
	
	    [SetUp]
	    public static void init() {
	    	props = TestPropertiesLoader.getProperties(".\\be.ehealth.businessconnector.recipe.test.properties",SupportedLanguages.NET.getAbbreviation());
	        SessionInitializer.init(props, SupportedLanguages.NET.getAbbreviation());	        
	        initPrescription();
	    }
	
	    /**
	     * 
	     * @throws IOException
	     * @throws ConnectorException
	     */
	    public static void initPrescription() {
	        /*
	         * The values used for testing
	         */
	        // The INSS of the patient
	        long patientId = java.lang.Long.parseLong(props.getProperty(TEST_PATIENT_INSS));
	        // The type of the prescription
	        String prescriptionType = PRESCRIPTION_TYPE_ID;
	        // The prescription in KMEHR format
	        byte[] prescription = getKmehrPrescription();
	        // Is feedback requested?
	        Boolean feedback = true;
	
	        /*
	         * Invoke the business connector framework's Recip-e's createPrescription operation
	         */
	        // RecipePrescriberService service ;//= ServiceFactory.getRecipePrescriberService();
	        be.ehealth.businessconnector.recipe.session.RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
	
	
	        prescriptionId = service.createPrescription(feedback, patientId, prescription, prescriptionType);
	    }
	
	    /**
	     * Test the prepareCreatePrescription operation from the Recip-e Service through the use of the Business Connector. The operation goes
	     * to KGSS to generate a key and hold it in a cache for later use.
	     * 
	     * The prepareCreatePrescription methods required the following parameters: - INSS of the patient - Type of prescription
	     * 
	     * @throws ConnectorException
	     */
	    [Test]
	    public void testPrepareCreatePrescription(){
	        /*
	         * Create the parameters of the new message
	         */
	        // The INSS of the patient (retrieved from property file)
	        long patientId = java.lang.Long.parseLong(props.getProperty(TEST_PATIENT_INSS));
	        // The type of the prescription
	        String prescriptionType = PRESCRIPTION_TYPE_ID;
	
	        /*
	         * Invoke the business connector framework's Recip-e's prepareCreatePrescription operation
	         */
	        be.ehealth.businessconnector.recipe.session.RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
	        service.prepareCreatePrescription(patientId, prescriptionType);
	    }
	
	
	    /**
	     * Test the createPrescription operation from the Recip-e Service through the use of the Business Connector. This test creates a new
	     * prescription based on a given KMEHR prescription
	     * 
	     * The createPrescription methods required the following parameters: - INSS of the patient - Type of prescription - Prescription in
	     * KMEHR format - Is feedback requested?
	     * 
	     * @throws IOException Signals that an I/O exception has occurred.
	     * @throws ConnectorException
	     */
	    [Test]
	    public void testCreatePrescription() {
	        createPrescription();
	
	        /*
	         * Verify the response
	         */
	        // check if the precriptionId is set
	        Assert.IsNotNull(prescriptionId);
	    }
	
	    /**
	     * @throws IOException
	     * @throws TechnicalConnectorException
	     * @throws ConnectorException
	     * @throws SessionManagementException
	     * @throws RecipeBusinessConnectorException
	     */
	    private String createPrescription() {
	        /*
	         * The values used for testing
	         */
	        // The INSS of the patient
	        long patientId = java.lang.Long.parseLong(props.getProperty(TEST_PATIENT_INSS));
	        // The type of the prescription
	        String prescriptionType = PRESCRIPTION_TYPE_ID;
	        // The prescription in KMEHR format
	        byte[] prescription = getKmehrPrescription();
	        // Is feedback requested?
	        Boolean feedback = true;
	
	        /*
	         * Invoke the business connector framework's Recip-e's createPrescription operation
	         */
	        be.ehealth.businessconnector.recipe.session.RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
	
	        return service.createPrescription(feedback, patientId, prescription, prescriptionType);
	    }
	
	    /**
	     * Test the getPrescription operation from the Recip-e Service through the use of the Business Connector. This test retrieves the
	     * previously created prescription (retrieved prescription is in kmehr format)
	     * 
	     * The getPrescription methods required the following parameters: - prescriptionId
	     * 
	     * @throws ConnectorException
	     * @throws DataFormatException
	     * @throws IOException
	     */
	    [Test]
	    public void testGetPrescription() {
	        /*
	         * The values used for testing
	         */
	        // should be the prescription id of an already existing prescription (e.g. the one we created in the previous test)
	        // change :to make tests independant , we now just create a new Prescription just to get its id
	        String rid = createPrescription();
	
	        /*
	         * Invoke the business connector framework's Recip-e's getPrescription operation
	         */
	        be.ehealth.businessconnector.recipe.session.RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
	        GetPrescriptionForPrescriberResult result = service.getPrescription(rid);
	
	        /*
	         * Verify the response
	         */
	        // check if there is a prescription in KMEHR format
	        Assert.IsNotNull(result.getPrescription());
	    }
	
	    /**
	     * Test the sendNotification operation from the Recip-e Service through the use of the Business Connector. This test sends a
	     * notification to an executor (pharmacy) (notification is in kmehr format)
	     * 
	     * The sendNotification methods required the following parameters: - INSS of the patient - NIHII of the pharmacy - Notification in KMEHR
	     * format
	     * 
	     * @throws IOException Signals that an I/O exception has occurred.
	     * @throws ConnectorException
	     */
	    [Test]
	    public void testSendNotification() {
	        /*
	         * The values used for testing
	         */
	        // The INSS of the patient
	        long patientId = java.lang.Long.parseLong(props.getProperty(TEST_PATIENT_INSS));
	        // The NIHII of a PHARMACY
	        long executorId = java.lang.Long.parseLong(props.getProperty(TEST_NIHII_PHARMACY));
	        // the Notification in KMEHR format
	        byte[] notification = getKmehrNotification();
	
	        /*
	         * Invoke the business connector framework's Recip-e's sendNotification operation
	         */
	        be.ehealth.businessconnector.recipe.session.RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
	        service.sendNotification(notification, patientId, executorId);
	    }
	
	    /**
	     * Test the listOpenPrescription operation from the Recip-e Service through the use of the Business Connector. This test returns a list
	     * of prescriptionId's that are not yet retrieved from the pharmacy
	     * 
	     * @throws ConnectorException
	     */
	    [Test]
	    public void testListOpenPrescription() {
	        /*
	         * Invoke the business connector framework's Recip-e's listOpenPrescription operation
	         */
	        be.ehealth.businessconnector.recipe.session.RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
	        java.util.List ridList = service.listOpenPrescription();
	
	        /*
	         * Verify the response
	         */
	        // check if there is data
	        Assert.IsNotNull(ridList);
	    }
	
	    /**
	     * Test the listFeedback operation from the Recip-e Service through the use of the Business Connector. This test retrieves a list of
	     * feedback
	     * 
	     * @throws ConnectorException
	     * @throws DataFormatException
	     */
	    [Test]
	    public void testListFeedback() {
	        /*
	         * The values used for testing
	         */
	        Boolean readFlag = false;
	
	        /*
	         * Invoke the business connector framework's Recip-e's listFeedback operation
	         */
	        be.ehealth.businessconnector.recipe.session.RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
	        java.util.List feedbackList = service.listFeedback(readFlag);
	
	        /*
	         * Verify the response
	         */
	        // check if there is data
	        Assert.IsNotNull(feedbackList);
	    }
	
	    /**
	     * Test the updateFeedbackFlag operation from the Recip-e Service through the use of the Business Connector. This test updates the
	     * feedback-flag of a previously created prescription
	     * 
	     * The updateFeedbackFlag methods required the following parameters: - prescriptionId - feedback requested?
	     * 
	     * @throws ConnectorException
	     */
	    [Test]
	    public void testUpdateFeedbackFlag() {
	        /*
	         * The values used for testing
	         */
	        // The prescription id of an already existing prescription
	        String rid = prescriptionId;
	        // Request feedback for this prescription?
	        Boolean newFeedbackFlag = false;
	
	        /*
	         * Invoke the business connector framework's Recip-e's updateFeedbackFlag operation
	         */
	        be.ehealth.businessconnector.recipe.session.RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
	        service.updateFeedbackFlag(rid, newFeedbackFlag);
	
	    }
	
	    /**
	     * Test the revokePrescription operation from the Recip-e Service through the use of the Business Connector. This test revokes a
	     * previously created prescription based on the prescription id.
	     * 
	     * The revokePrescription methods required the following parameters: - prescriptionId - reason of revocation
	     * 
	     * @throws ConnectorException
	     */
	    [Test]
	    public void testRevokePrescription() {
	        /*
	         * The values used for testing
	         */
	        // The prescription id of an already existing prescription
	        String rid = prescriptionId;
	        // The reason why you are revoking the prescription
	        String reason = "Prescription added by mistake";
	
	        /*
	         * Invoke the business connector framework's Recip-e's revokePrescription operation
	         */
	        be.ehealth.businessconnector.recipe.session.RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
	        service.revokePrescription(rid, reason);
	
	    }
	
	    /**
	     * Helper method Gets the prescription in KMEHR format.
	     * 
	     * @return the kmehr prescription
	     * @throws IOException Signals that an I/O exception has occurred.
	     * @throws BusinessConnectorException
	     * @throws TechnicalConnectorException
	     */
	    private static byte[] getKmehrPrescription() {
	        // location of the kmehr message
	        String file = props.getProperty(TEST_SAMPLE_PRESCRIPTION);
	        
	        // read out the KMEHR prescription
	        InputStream ins = ConnectorIOUtils.getResourceAsStream(file);
	
	        return ConnectorIOUtils.getBytes(ins);
	    }
	
	    /**
	     * Helper method Gets the notification in KMEHR format.
	     * 
	     * @return the kmehr notification
	     * @throws IOException Signals that an I/O exception has occurred.
	     * @throws BusinessConnectorException
	     * @throws TechnicalConnectorException
	     */
	    private byte[] getKmehrNotification() {
	        // location of the kmehr message
	        String file = props.getProperty(TEST_SAMPLE_NOTIFICATION);
	
	        // read out the KMEHR prescription
	        InputStream ins = ConnectorIOUtils.getResourceAsStream(@file);
	        return ConnectorIOUtils.getBytes(ins);
	    }
	}
}
