package be.ehealth.businessconnector.recipe.service;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;
import java.util.zip.DataFormatException;

import org.junit.AfterClass;
import org.junit.Assert;
import org.junit.Before;
import org.junit.BeforeClass;
import org.junit.Test;

import be.ehealth.businessconnector.recipe.exception.RecipeBusinessConnectorException;
import be.ehealth.businessconnector.recipe.session.RecipePrescriberService;
import be.ehealth.businessconnector.recipe.session.RecipeSessionServiceFactory;
import be.ehealth.technicalconnector.exception.ConnectorException;
import be.ehealth.technicalconnector.exception.SessionManagementException;
import be.ehealth.technicalconnector.exception.TechnicalConnectorException;
import be.ehealth.technicalconnector.session.Session;
import be.ehealth.technicalconnector.utils.ConnectorIOUtils;
import be.fgov.ehealth.technicalconnector.tests.session.SessionInitializer;
import be.fgov.ehealth.technicalconnector.tests.utils.TestPropertiesLoader;
import be.recipe.services.GetPrescriptionForPrescriberResult;
import be.recipe.services.ListFeedbackItem;

/**
 * The Class RecipePrescriberServiceTest.
 * 
 * The tests below use the Session Management Service to manage the session and SAML/STS token.
 * 
 * @author EHP
 */
@Deprecated
public class RecipePrescriberServiceIntegrationTest {

    private static final String PRESCRIPTION_TYPE_ID = "P0";

    private static final String TEST_PATIENT_INSS = "test.recipe.patient.inss";

    private static final String TEST_NIHII_PHARMACY = "test.recipe.nihii.pharmacy";

    private static final String TEST_SAMPLE_PRESCRIPTION = "test.recipe.sample.prescription";

    private static final String TEST_SAMPLE_NOTIFICATION = "test.recipe.sample.notification";

    private static Properties props;

    /** The prescription id. */
    private static String prescriptionId;

    @SuppressWarnings("javadoc")
    @BeforeClass
    public static void init() throws Exception {
        SessionInitializer.init("/be.ehealth.businessconnector.recipe.test.properties", true);
        props = TestPropertiesLoader.getProperties("/be.ehealth.businessconnector.recipe.test.properties");
    }

    /**
     * 
     * @throws IOException
     * @throws ConnectorException
     */
    @Before
    public void initPrescription() throws IOException, ConnectorException {
        /*
         * The values used for testing
         */
        // The INSS of the patient
        long patientId = Long.parseLong(props.getProperty(TEST_PATIENT_INSS));
        // The type of the prescription
        String prescriptionType = PRESCRIPTION_TYPE_ID;
        // The prescription in KMEHR format
        byte[] prescription = getKmehrPrescription();
        // Is feedback requested?
        boolean feedback = true;

        /*
         * Invoke the business connector framework's Recip-e's createPrescription operation
         */
        // RecipePrescriberService service ;//= ServiceFactory.getRecipePrescriberService();
        RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();


        prescriptionId = service.createPrescription(feedback, patientId, prescription, prescriptionType);
    }

    @AfterClass
    public static void tearDown() throws Exception {
        Session.getInstance().unloadSession();
    }

    /**
     * Test the prepareCreatePrescription operation from the Recip-e Service through the use of the Business Connector. The operation goes
     * to KGSS to generate a key and hold it in a cache for later use.
     * 
     * The prepareCreatePrescription methods required the following parameters: - INSS of the patient - Type of prescription
     * 
     * @throws ConnectorException
     */
    @Test
    public void testPrepareCreatePrescription() throws ConnectorException {
        /*
         * Create the parameters of the new message
         */
        // The INSS of the patient (retrieved from property file)
        long patientId = Long.parseLong(props.getProperty(TEST_PATIENT_INSS));
        // The type of the prescription
        String prescriptionType = PRESCRIPTION_TYPE_ID;

        /*
         * Invoke the business connector framework's Recip-e's prepareCreatePrescription operation
         */
        RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
        service.prepareCreatePrescription(patientId, prescriptionType);
    }


    /**
     * Test the createPrescription operation from the Recip-e Service through the use of the Business Connector. This test creates a new
     * prescription based on a given KMEHR prescription
     * 
     * The createPrescription methods required the following parameters: - INSS of the patient - Type of prescription - Prescription in
     * KMEHR format - Is feedback requested?
     * 
     * @throws IOException Signals that an I/O exception has occurred.
     * @throws ConnectorException
     */
    @Test
    public void testCreatePrescription() throws IOException, ConnectorException {
        createPrescription();

        /*
         * Verify the response
         */
        // check if the precriptionId is set
        Assert.assertNotNull(prescriptionId);
    }

    /**
     * @throws IOException
     * @throws TechnicalConnectorException
     * @throws ConnectorException
     * @throws SessionManagementException
     * @throws RecipeBusinessConnectorException
     */
    private String createPrescription() throws IOException, TechnicalConnectorException, ConnectorException, SessionManagementException, RecipeBusinessConnectorException {
        /*
         * The values used for testing
         */
        // The INSS of the patient
        long patientId = Long.parseLong(props.getProperty(TEST_PATIENT_INSS));
        // The type of the prescription
        String prescriptionType = PRESCRIPTION_TYPE_ID;
        // The prescription in KMEHR format
        byte[] prescription = getKmehrPrescription();
        // Is feedback requested?
        boolean feedback = true;

        /*
         * Invoke the business connector framework's Recip-e's createPrescription operation
         */
        RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();

        return service.createPrescription(feedback, patientId, prescription, prescriptionType);
    }

    /**
     * Test the getPrescription operation from the Recip-e Service through the use of the Business Connector. This test retrieves the
     * previously created prescription (retrieved prescription is in kmehr format)
     * 
     * The getPrescription methods required the following parameters: - prescriptionId
     * 
     * @throws ConnectorException
     * @throws DataFormatException
     * @throws IOException
     */
    @Test
    public void testGetPrescription() throws ConnectorException, DataFormatException, IOException {
        /*
         * The values used for testing
         */
        // should be the prescription id of an already existing prescription (e.g. the one we created in the previous test)
        // change :to make tests independant , we now just create a new Prescription just to get its id
        String rid = createPrescription();

        /*
         * Invoke the business connector framework's Recip-e's getPrescription operation
         */
        RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
        GetPrescriptionForPrescriberResult result = service.getPrescription(rid);

        /*
         * Verify the response
         */
        // check if there is a prescription in KMEHR format
        Assert.assertNotNull(result.getPrescription());
    }

    /**
     * Test the sendNotification operation from the Recip-e Service through the use of the Business Connector. This test sends a
     * notification to an executor (pharmacy) (notification is in kmehr format)
     * 
     * The sendNotification methods required the following parameters: - INSS of the patient - NIHII of the pharmacy - Notification in KMEHR
     * format
     * 
     * @throws IOException Signals that an I/O exception has occurred.
     * @throws ConnectorException
     */
    @Test
    public void testSendNotification() throws IOException, ConnectorException {
        /*
         * The values used for testing
         */
        // The INSS of the patient
        long patientId = Long.parseLong(props.getProperty(TEST_PATIENT_INSS));
        // The NIHII of a PHARMACY
        long executorId = Long.parseLong(props.getProperty(TEST_NIHII_PHARMACY));
        // the Notification in KMEHR format
        byte[] notification = getKmehrNotification();

        /*
         * Invoke the business connector framework's Recip-e's sendNotification operation
         */
        RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
        service.sendNotification(notification, patientId, executorId);
    }

    /**
     * Test the listOpenPrescription operation from the Recip-e Service through the use of the Business Connector. This test returns a list
     * of prescriptionId's that are not yet retrieved from the pharmacy
     * 
     * @throws ConnectorException
     */
    @Test
    public void testListOpenPrescription() throws ConnectorException {
        /*
         * Invoke the business connector framework's Recip-e's listOpenPrescription operation
         */
        RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
        List<String> ridList = service.listOpenPrescription();

        /*
         * Verify the response
         */
        // check if there is data
        Assert.assertNotNull(ridList);
    }

    /**
     * Test the listFeedback operation from the Recip-e Service through the use of the Business Connector. This test retrieves a list of
     * feedback
     * 
     * @throws ConnectorException
     * @throws DataFormatException
     */
    @Test
    public void testListFeedback() throws ConnectorException, DataFormatException {
        /*
         * The values used for testing
         */
        boolean readFlag = false;

        /*
         * Invoke the business connector framework's Recip-e's listFeedback operation
         */
        RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
        List<ListFeedbackItem> feedbackList = service.listFeedback(readFlag);

        /*
         * Verify the response
         */
        // check if there is data
        Assert.assertNotNull(feedbackList);
    }

    /**
     * Test the updateFeedbackFlag operation from the Recip-e Service through the use of the Business Connector. This test updates the
     * feedback-flag of a previously created prescription
     * 
     * The updateFeedbackFlag methods required the following parameters: - prescriptionId - feedback requested?
     * 
     * @throws ConnectorException
     */
    @Test
    public void testUpdateFeedbackFlag() throws ConnectorException {
        /*
         * The values used for testing
         */
        // The prescription id of an already existing prescription
        String rid = prescriptionId;
        // Request feedback for this prescription?
        boolean newFeedbackFlag = false;

        /*
         * Invoke the business connector framework's Recip-e's updateFeedbackFlag operation
         */
        RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
        service.updateFeedbackFlag(rid, newFeedbackFlag);

    }

    /**
     * Test the revokePrescription operation from the Recip-e Service through the use of the Business Connector. This test revokes a
     * previously created prescription based on the prescription id.
     * 
     * The revokePrescription methods required the following parameters: - prescriptionId - reason of revocation
     * 
     * @throws ConnectorException
     */
    @Test
    public void testRevokePrescription() throws ConnectorException {
        /*
         * The values used for testing
         */
        // The prescription id of an already existing prescription
        String rid = prescriptionId;
        // The reason why you are revoking the prescription
        String reason = "Prescription added by mistake";

        /*
         * Invoke the business connector framework's Recip-e's revokePrescription operation
         */
        RecipePrescriberService service = RecipeSessionServiceFactory.getRecipePrescriberService();
        service.revokePrescription(rid, reason);

    }

    /**
     * Helper method Gets the prescription in KMEHR format.
     * 
     * @return the kmehr prescription
     * @throws IOException Signals that an I/O exception has occurred.
     * @throws BusinessConnectorException
     * @throws TechnicalConnectorException
     */
    private byte[] getKmehrPrescription() throws IOException, TechnicalConnectorException {
        // location of the kmehr message
        String file = props.getProperty(TEST_SAMPLE_PRESCRIPTION);

        // read out the KMEHR prescription
        InputStream is = ConnectorIOUtils.getResourceAsStream(file);

        return ConnectorIOUtils.getBytes(is);
    }

    /**
     * Helper method Gets the notification in KMEHR format.
     * 
     * @return the kmehr notification
     * @throws IOException Signals that an I/O exception has occurred.
     * @throws BusinessConnectorException
     * @throws TechnicalConnectorException
     */
    private byte[] getKmehrNotification() throws IOException, TechnicalConnectorException {
        // location of the kmehr message
        String file = props.getProperty(TEST_SAMPLE_NOTIFICATION);

        // read out the KMEHR prescription
        InputStream is = ConnectorIOUtils.getResourceAsStream(file);
        return ConnectorIOUtils.getBytes(is);
    }
}
